/************************************************************************************
TerraView - visualization and exploration of geographical databases
using TerraLib.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.
This file is part of TerraView. TerraView is free software; you can redistribute it 
and/or modify it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

You should have received a copy of the GNU General Public License
along with TerraView.
The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The software provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use of
this program and its documentation.
*************************************************************************************/

#ifndef  __TERRALIB_INTERNAL_QTBASICCANVAS_H
#define  __TERRALIB_INTERNAL_QTBASICCANVAS_H

#include <TeDatabase.h>
#include <TeRaster.h>

#include <qpainter.h>
#include <qbrush.h> 
#include <qpixmap.h>
#include <qimage.h>
#include <qbitmap.h>
#include <qprinter.h>
#include <qscrollview.h>   
 
#include <string>
using namespace std;

class TeQtProgress;
/**
  *@author R Cartaxo
  */

class TeQtBasicCanvas : public QScrollView
{
public:
	TeQtBasicCanvas(QWidget *parent, const char *name );
	virtual ~TeQtBasicCanvas();
	void setWorld(TeBox b, int w = 0, int h = 0, QPaintDevice *pd = 0);
	virtual void setView(int /*w*/ = 0, int /*h*/ = 0, QPaintDevice * /* pd */ = 0) {}
	void setTransformation(double xmin, double xmax, double ymin, double ymax, bool extend = true);
	TeBox getWorld() {return wc_ ;}
	TeBox getDataWorld();

// Coordinates system transformation
// Viewport to Canvas coordinates
	double mapVtoW (int);
	TeCoord2D mapVtoW (QPoint);
// Viewport to Data World coordinates
	double mapVtoDW (int);
	TeCoord2D mapVtoDW (QPoint);
// Data World to Viewport coordinates
	QPoint mapWtoV (TeCoord2D);
	void mapWtoV (TeBox& box);
//  Canvas World to Data World coordinates
	void mapCanvasWorldToDataWorld (TeBox& w);
// Data World to Canvas World coordinates
	void mapDataWorldToCanvasWorld (TeBox& w);
// Data Canvas World to Viewport coordinates
	void mapCanvasWorldToViewport (TeBox& w);
// Viewport to Canvas World coordinates
	void mapViewportToCanvasWorld (TeBox& w);
	QPoint mapCanvasWorldToViewport (TeCoord2D& c);

	virtual void correctScrolling (QPoint & ) {}
	virtual QPoint offset () {QPoint p; return p;}
// Pixel size in world coordinates
	double pixelSize () { return (xmax_-xmin_)/(double)width_; }

// Sets the projection of next incoming data
	void  setDataProjection ( TeProjection* proj );

// Sets the projection
	void  setProjection ( TeProjection* proj );

// Retrive the projection
	TeProjection* projection()
	{	return canvasProjection_; }

// Plotting primitives

//	void setPainter (QPainter* p) {painter_ = p;}
	QPainter* getPainter () {return &painter_;}
	void plotOnWindow ();
	void plotOnPixmap0 ();
	void plotOnPixmap1 ();
	virtual void copyPixmapToWindow(QPixmap*, int, int, int, int);
	virtual void copyPixmap0ToPrinter();
	virtual void copyPixmap0To(QPaintDevice*);
	virtual void copyPixmap0ToWindow();
	virtual void copyPixmap1ToWindow();
	virtual void copyPixmap2ToWindow();
	virtual void copyPixmap3ToWindow();
	virtual void copyPixmap0ToWindow(int, int, int, int);
	virtual void copyPixmap0ToPixmap1();
	virtual void copyPixmap0ToPixmap1(int, int, int, int);
	virtual void copyPixmap1ToPixmap0();
	virtual void copyPixmap1ToPixmap0(int, int, int, int);
	virtual void copyPixmap1ToPixmap2();
	virtual void copyPixmap1ToPixmap2(int, int, int, int);
	virtual void copyPixmap2ToPixmap1();
	virtual void copyPixmap2ToPixmap1(int, int, int, int);
	virtual void copyPixmap2ToPixmap3();
	virtual void copyPixmap2ToPixmap3(int, int, int, int);
	virtual void copyPixmap3ToPixmap2();
	virtual void copyPixmap3ToPixmap2(int, int, int, int);
	virtual void copyPanArea(int, int);

	void plotPoint (TeCoord2D &p);
	void plotPoint (TePoint &p){ plotPoint(p.location()); }

	void setPointColor (int r, int g, int b);
	void setPointStyle (int s, int w = 3);
	void setPointPixmap (char* p);

	void plotCell (TeCell &s);

	QRect getLegendRect (QPoint p, const QPixmap* pix, string tx);
	void plotLegend (QPoint p, const QPixmap* pix, string tx);

	void plotPolygon (TePolygon &p);
	void setPolygonColor (int r, int g, int b);
	void setPolygonStyle (int s, int width = 1);
	void setPolygonLineColor (int r, int g, int b);
	void setPolygonLineStyle (int s, int width = 1);

	void plotLine (TeLine2D &l);
	void setLineColor (int r, int g, int b);
	void setLineStyle (int s, int w = 1);

	void plotNode (TeNode &n);
	void setNodeColor (int r, int g, int b);
	void setNodeStyle (int s, int w = 4);
	void setNodePixmap (char* p);

	void plotArc (TeArc &a);
	void setArcColor (int r, int g, int b);
	void setArcStyle ( int s, int w = 1 );

	void plotPie (double x, double y, double w, double h, double a, double alen);
	void setPieColor (int r, int g, int b);
	void plotRect (double x, double y, double w, double h, int transp=0, bool legend=false);
	void plotRect (QRect&);
	void setRectColor (int r, int g, int b);

	void plotText (TeCoord2D &p, string &str, double angle = 0., double alignh = 0., double alignv = 0.);
	void plotText (TeText &tx, TeVisual& visual);
	QRect textRect (TeText &tx, TeVisual visual);
	void plotTextRects (TeText &tx, TeVisual visual);
	void setTextColor (int r, int g, int b);
	void setTextSize (int size);
	void setTextStyle (string& family, int size, bool bold, bool italic);
	void setTextTTStyle (QString& font, int size);
	void useTextTT (bool u) { textTT_ = u; }
	void textExtent ( string &str, int &w, int &h, double angle = 0.);
	void plotXorPolyline (QPointArray& PA, bool cdev=true);

	void setDB (TeDatabase* db) { db_ = db; }
	void plotRaster (TeRaster* raster, TeRasterTransform* transf, TeQtProgress *progress);
	void clearRaster();
	void endPrinting();

	QPixmap* getPixmap0 () {return pixmap0_; }
	QPixmap* getPixmap1 () {return pixmap1_; }
	QPixmap* getPixmap2 () {return pixmap2_; }
	QPixmap* getPixmap3 () {return pixmap3_; }

	void plotGraphicScale(TeVisual& visual);

	void setClipRegion(int x, int y, int w, int h);
	void setClipRegion(QRegion region);
	void setClipping (bool enable);

	virtual void copyPixmapToWindow(){}

	TeRasterParams& getParams()
	{	return params_; }

	double scaleApx() {return scaleApx_;}
	void scaleApx(double);

protected:

	void plotMark(QPoint &p, int s, int w);
	bool buildRaster ();

// Sizes and transformation
	TeBox		wc_;
	double 		xmin_,
				xmax_,
				ymin_,
				ymax_;
	
	int			x0_;
	int			y0_;
	int			width_;
	int			height_;
	double		f_;
	double		scaleApx_;
	TeProjection* canvasProjection_;
	TeProjection* dataProjection_;

// The Painter
	QPainter	painter_;

// Pie
	QBrush		pieBrush_;
	QColor		pieColor_;
	QBrush		rectBrush_;
	QColor		rectColor_;
	
// Polygon
	int			polygonTransparency_;
	QBrush		polygonBrush_;
	QPen		polygonPen_;
	QColor		polygonColor_;
	map<TePolyBasicType, Qt::BrushStyle> brushStyleMap_;
	
// Line
	QPen		linePen_;
	QColor		lineColor_;
	map<TeLnBasicType, Qt::PenStyle> penStyleMap_;
	
// Point
	int			pointStyle_;
	int			pointSize_;
	QPen		pointPen_;
	QColor		pointColor_;
	QPixmap*	pointPixmap_;

// Arc
	QPen		arcPen_;
	QColor		arcColor_;
	
// Node
	int			nodeStyle_;
	int			nodeSize_;
	QPen		nodePen_;
	QColor		nodeColor_;
	QPixmap*	nodePixmap_;

// Text
	QPen		textPen_;
	QColor		textColor_;
	QFont		textFont_;
	string		textTTFont_;
	int			textSize_;
	bool		textTT_;

// Pixmap
	QPixmap		*pixmap0_;
	QPixmap		*pixmap1_;
	QPixmap		*pixmap2_;
	QPixmap		*pixmap3_;

	QImage		qimage_;

	int			scale_;
	TeRaster	*backRaster_;
	TeRasterParams	params_;
	int			lx1_,
				ly1_,
				lx2_,
				ly2_;

	TeDatabase	*db_;
};

#endif

