/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  treeviewcolumn.cc - GtkTreeViewColumn C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "treeview.h"
#include "private/treeviewcolumn_p.h"
#include "treemodel.h"
#include "widget.h"

using namespace Inti;

/*  Gtk::TreeViewColumn
 */

Gtk::TreeViewColumn::TreeViewColumn(GtkTreeViewColumn *tree_column, bool reference)
: Object((GtkObject*)tree_column, reference)
{
}
	
Gtk::TreeViewColumn::TreeViewColumn()
: Object((GtkObject*)TreeViewColumnClass::create())
{
}

namespace { // set_attributesv

void set_attributesv(GtkTreeViewColumn *tree_column, GtkCellRenderer *cell, va_list args)
{
	g_return_if_fail(tree_column != 0);
	g_return_if_fail (GTK_IS_TREE_VIEW_COLUMN(tree_column));

	gchar *attribute = va_arg(args, gchar*);
	gtk_tree_view_column_clear_attributes(tree_column, cell);
	while (attribute)
	{
		gint column = va_arg(args, gint);
		gtk_tree_view_column_add_attribute(tree_column, cell, attribute, column);
		attribute = va_arg(args, gchar*);
	}
}

} // set_attributesv

Gtk::TreeViewColumn::TreeViewColumn(const String& title, CellRenderer *cell, ...)
: Object((GtkObject*)TreeViewColumnClass::create())
{
	set_title(title);
	pack_start(*cell);
	va_list args;
	va_start(args, cell);
	set_attributesv(gtk_tree_view_column(), *cell, args);
	va_end(args);
}

Gtk::TreeViewColumn::TreeViewColumn(const String& title, CellRenderer& cell, const std::map<int, String>& attributes)
: Object((GtkObject*)TreeViewColumnClass::create())
{
	set_title(title);
	pack_start(cell);
	set_attributes(cell, attributes);
}

Gtk::TreeViewColumn::~TreeViewColumn() 
{
}
	
GtkTreeViewColumnClass*
Gtk::TreeViewColumn::gtk_tree_view_column_class() const 
{ 
	return get_class<GtkTreeViewColumnClass>(); 
}
	
Gtk::TreeViewColumn::operator GtkTreeViewColumn* () const 
{ 
	return this ? gtk_tree_view_column() : 0; 
}
	
int
Gtk::TreeViewColumn::get_spacing() const
{
	return gtk_tree_view_column_get_spacing(gtk_tree_view_column());
}

bool
Gtk::TreeViewColumn::get_visible() const
{
	return gtk_tree_view_column_get_visible(gtk_tree_view_column());
}

bool
Gtk::TreeViewColumn::get_resizable() const
{
	return gtk_tree_view_column_get_resizable(gtk_tree_view_column());
}

Gtk::TreeViewColumnSizing
Gtk::TreeViewColumn::get_sizing() const
{
	return (TreeViewColumnSizing)gtk_tree_view_column_get_sizing(gtk_tree_view_column());
}

int
Gtk::TreeViewColumn::get_fixed_width() const
{
	return gtk_tree_view_column_get_fixed_width(gtk_tree_view_column());
}

int
Gtk::TreeViewColumn::get_width() const
{
	return gtk_tree_view_column_get_width(gtk_tree_view_column());
}

int
Gtk::TreeViewColumn::get_min_width() const
{
	return gtk_tree_view_column_get_min_width(gtk_tree_view_column());
}

int
Gtk::TreeViewColumn::get_max_width() const
{
	return gtk_tree_view_column_get_max_width(gtk_tree_view_column());
}

String 
Gtk::TreeViewColumn::get_title() const
{
	return gtk_tree_view_column_get_title(gtk_tree_view_column());
}

bool 
Gtk::TreeViewColumn::get_clickable() const
{
	return gtk_tree_view_column_get_clickable(gtk_tree_view_column());
}

Gtk::Widget*
Gtk::TreeViewColumn::get_widget() const
{
	return G::Object::wrap<Widget>(gtk_tree_view_column_get_widget(gtk_tree_view_column()));
}

float
Gtk::TreeViewColumn::get_alignment() const
{
	return gtk_tree_view_column_get_alignment(gtk_tree_view_column());
}

bool
Gtk::TreeViewColumn::get_reorderable() const
{
	return gtk_tree_view_column_get_reorderable(gtk_tree_view_column());
}

int
Gtk::TreeViewColumn::get_sort_column_id() const
{
	return gtk_tree_view_column_get_sort_column_id(gtk_tree_view_column());
}

bool 
Gtk::TreeViewColumn::get_sort_indicator() const
{
	return gtk_tree_view_column_get_sort_indicator(gtk_tree_view_column());
}

Gtk::SortType
Gtk::TreeViewColumn::get_sort_order() const
{
	return (SortType)gtk_tree_view_column_get_sort_order(gtk_tree_view_column());
}

bool 
Gtk::TreeViewColumn::get_cell_renderers(std::vector<CellRenderer*>& cell_list) const
{
	g_return_val_if_fail(cell_list.empty(), false);
	GList *first = gtk_tree_view_column_get_cell_renderers(gtk_tree_view_column());
	GList *next = first;

	while (next != 0)
	{
		cell_list.push_back(G::Object::wrap<CellRenderer>((GtkCellRenderer*)next->data));
		next = g_list_next(next);
	}

	g_list_free(first);
	return !cell_list.empty();
}	

void 
Gtk::TreeViewColumn::cell_get_size(const Gdk::Rectangle *cell_area, int *x_offset, int *y_offset, int *width, int *height) const
{
	gtk_tree_view_column_cell_get_size(gtk_tree_view_column(), *cell_area, x_offset, y_offset, width, height);
}

bool 
Gtk::TreeViewColumn::cell_is_visible() const
{
	return gtk_tree_view_column_cell_is_visible(gtk_tree_view_column());
}

void
Gtk::TreeViewColumn::pack_start(CellRenderer& cell, bool expand)
{
	gtk_tree_view_column_pack_start(gtk_tree_view_column(), cell.gtk_cell_renderer(), expand);
}

void 
Gtk::TreeViewColumn::pack_end(CellRenderer& cell, bool expand)
{
	gtk_tree_view_column_pack_end(gtk_tree_view_column(), cell.gtk_cell_renderer(), expand);
}

void
Gtk::TreeViewColumn::clear()
{
	gtk_tree_view_column_clear(gtk_tree_view_column());
}

void 
Gtk::TreeViewColumn::add_attribute(CellRenderer& cell, const String& attribute, int column)
{
	gtk_tree_view_column_add_attribute(gtk_tree_view_column(), cell.gtk_cell_renderer(), attribute.c_str(), column);
}

void
Gtk::TreeViewColumn::set_attributes(CellRenderer *cell, ...)
{
	g_return_if_fail(cell != 0);
	va_list args;
	va_start(args, cell);
	set_attributesv(gtk_tree_view_column(), *cell, args);
	va_end (args);
}	

void
Gtk::TreeViewColumn::set_attributes(CellRenderer& cell, const std::map<int, String>& attributes)
{
	g_return_if_fail(!attributes.empty());
	clear_attributes(cell);

	std::map<int, String>::const_iterator i = attributes.begin();
	while (i != attributes.end())
	{
		add_attribute(cell, (*i).second, (*i).first);
		++i;
	}	 
}

namespace { // cell_data_slot_callback

void cell_data_slot_callback(GtkTreeViewColumn*, GtkCellRenderer *cell, GtkTreeModel *tree_model, GtkTreeIter *iter, gpointer data)
{
	Gtk::TreeViewColumn::CellDataSlot *slot = (Gtk::TreeViewColumn::CellDataSlot*)data;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(tree_model));
	Gtk::TreeModel *tmp_tree_model = dynamic_cast<Gtk::TreeModel*>(object);
	Gtk::TreeIter tmp_iter(iter);
	slot->call(*G::Object::wrap<Gtk::CellRenderer>(cell), tmp_tree_model, tmp_iter);
}

} // cell_data_slot_callback
	
void
Gtk::TreeViewColumn::set_cell_data_slot(CellRenderer& cell, const CellDataSlot *data)
{
	gtk_tree_view_column_set_cell_data_func(gtk_tree_view_column(), cell.gtk_cell_renderer(), 
	                                        data ? &cell_data_slot_callback : 0, (void*)data, 0);
}

void
Gtk::TreeViewColumn::clear_attributes(CellRenderer& cell)
{
	gtk_tree_view_column_clear_attributes(gtk_tree_view_column(), cell.gtk_cell_renderer());
}

void 
Gtk::TreeViewColumn::set_spacing(int spacing)
{
	gtk_tree_view_column_set_spacing(gtk_tree_view_column(), spacing);
}

void 
Gtk::TreeViewColumn::set_visible(bool visible)
{
	gtk_tree_view_column_set_visible(gtk_tree_view_column(), visible);
}

void
Gtk::TreeViewColumn::set_resizable(bool resizable)
{
	gtk_tree_view_column_set_resizable(gtk_tree_view_column(), resizable);
}

void
Gtk::TreeViewColumn::set_sizing(TreeViewColumnSizing type)
{
	gtk_tree_view_column_set_sizing(gtk_tree_view_column(), (GtkTreeViewColumnSizing)type);
}

void 
Gtk::TreeViewColumn::set_fixed_width(int fixed_width)
{
	gtk_tree_view_column_set_fixed_width(gtk_tree_view_column(), fixed_width);
}

void
Gtk::TreeViewColumn::set_min_width(int min_width)
{
	gtk_tree_view_column_set_min_width(gtk_tree_view_column(), min_width);
}

void 
Gtk::TreeViewColumn::set_max_width(int max_width)
{
	gtk_tree_view_column_set_max_width(gtk_tree_view_column(), max_width);
}

void
Gtk::TreeViewColumn::set_title(const String& title)
{
	gtk_tree_view_column_set_title(gtk_tree_view_column(), title.c_str());
}

void
Gtk::TreeViewColumn::clicked()
{
	return gtk_tree_view_column_clicked(gtk_tree_view_column());
}

void
Gtk::TreeViewColumn::set_clickable(bool active)
{
	gtk_tree_view_column_set_clickable(gtk_tree_view_column(), active);
}
	
void 
Gtk::TreeViewColumn::set_widget(Widget *widget)
{
	gtk_tree_view_column_set_widget(gtk_tree_view_column(), *widget);
}

void 
Gtk::TreeViewColumn::set_alignment(float xalign)
{
	gtk_tree_view_column_set_alignment(gtk_tree_view_column(), xalign);
}

void 
Gtk::TreeViewColumn::set_reorderable(bool reorderable)
{
	gtk_tree_view_column_set_reorderable(gtk_tree_view_column(), reorderable);
}

void 
Gtk::TreeViewColumn::set_sort_column_id(int sort_column_id)
{
	gtk_tree_view_column_set_sort_column_id(gtk_tree_view_column(), sort_column_id);
}

void 
Gtk::TreeViewColumn::set_sort_indicator(bool setting)
{
	gtk_tree_view_column_set_sort_indicator(gtk_tree_view_column(), setting);
}

void 
Gtk::TreeViewColumn::set_sort_order(SortType order)
{
	gtk_tree_view_column_set_sort_order(gtk_tree_view_column(), (GtkSortType)order);
}

void
Gtk::TreeViewColumn::cell_set_cell_data(const TreeModel& model, const TreeIter& iter, bool is_expander, bool is_expanded)
{
	gtk_tree_view_column_cell_set_cell_data(gtk_tree_view_column(), model.gtk_tree_model(),
	                                        iter.gtk_tree_iter(), is_expander, is_expanded);
}

void 
Gtk::TreeViewColumn::focus_cell(CellRenderer& cell)
{
	gtk_tree_view_column_focus_cell(gtk_tree_view_column(), cell.gtk_cell_renderer());
}

/*  Gtk::TreeViewColumnClass
 */

void
Gtk::TreeViewColumnClass::init(GtkTreeViewColumnClass *g_class)
{
	ObjectClass::init((GtkObjectClass*)g_class);
	g_class->clicked = &clicked_proxy;
}

GType
Gtk::TreeViewColumnClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_TREE_VIEW_COLUMN, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::TreeViewColumnClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::TreeViewColumnClass::clicked_proxy(GtkTreeViewColumn *tree_column)
{
	TreeViewColumn *tmp_tree_column = G::Object::pointer<TreeViewColumn>(tree_column);
	if (tmp_tree_column)
		tmp_tree_column->on_clicked();
	else
	{
		GtkTreeViewColumnClass *tmp_class = GTK_TREE_VIEW_COLUMN_GET_CLASS(tree_column);
		GtkTreeViewColumnClass *g_class = G::TypeInstance::class_peek_parent<GtkTreeViewColumnClass>(tmp_class);
		if (g_class->clicked)
			g_class->clicked(tree_column);
	}
}

/*  Signal handlers
 */

void
Gtk::TreeViewColumn::on_clicked()
{
	GtkTreeViewColumnClass *g_class = class_peek_parent<GtkTreeViewColumnClass>(gtk_tree_view_column_class());
	if (g_class->clicked)
		g_class->clicked(gtk_tree_view_column());
}

/*  Properties
 */

const Gtk::TreeViewColumn::VisiblePropertyType Gtk::TreeViewColumn::visible_property("visible");

const Gtk::TreeViewColumn::ResizablePropertyType Gtk::TreeViewColumn::resizable_property("resizable");

const Gtk::TreeViewColumn::WidthPropertyType Gtk::TreeViewColumn::width_property("width");

const Gtk::TreeViewColumn::SizingPropertyType Gtk::TreeViewColumn::sizing_property("sizing");

const Gtk::TreeViewColumn::FixedWidthPropertyType Gtk::TreeViewColumn::fixed_width_property("fixed_width");

const Gtk::TreeViewColumn::MinWidthPropertyType Gtk::TreeViewColumn::min_width_property("min_width");

const Gtk::TreeViewColumn::MaxWidthPropertyType Gtk::TreeViewColumn::max_width_property("max_width");

const Gtk::TreeViewColumn::TitlePropertyType Gtk::TreeViewColumn::title_property("title");

const Gtk::TreeViewColumn::ClickablePropertyType Gtk::TreeViewColumn::clickable_property("clickable");

const Gtk::TreeViewColumn::WidgetPropertyType Gtk::TreeViewColumn::widget_property("widget");

const Gtk::TreeViewColumn::AlignmentPropertyType Gtk::TreeViewColumn::alignment_property("alignment");

const Gtk::TreeViewColumn::ReorderablePropertyType Gtk::TreeViewColumn::reorderable_property("reorderable");

const Gtk::TreeViewColumn::SortIndicatorPropertyType Gtk::TreeViewColumn::sort_indicator_property("sort_indicator");

const Gtk::TreeViewColumn::SortOrderPropertyType Gtk::TreeViewColumn::sort_order_property("sort_order");

/*  Signals
 */

const Gtk::TreeViewColumn::ClickedSignalType Gtk::TreeViewColumn::clicked_signal("clicked");

