/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *  Copyright (C) 2000 Red Hat, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/texttag.h
//! @brief A GtkTextTag C++ wrapper interface.
//!
//! Provides TextAppearance, TextAttributes and TextTag

#ifndef INTI_GTK_TEXT_TAG_H
#define INTI_GTK_TEXT_TAG_H

#ifndef INTI_G_BOXED_H
#include <inti/glib/boxed.h>
#endif

#ifndef INTI_G_OBJECT_H
#include <inti/glib/object.h>
#endif

#ifndef INTI_GTK_ENUMS_H
#include <inti/gtk/enums.h>
#endif

#ifndef INTI_PANGO_ATTRIBUTES_H
#include <inti/pango/attributes.h>
#endif

#ifndef GTK_TEXT_TAG_H
#include <gtk/gtktexttag.h>
#endif

#ifndef GTK_TEXT_TAG_TABLE_H
#include <gtk/gtktexttagtable.h>
#endif

namespace Inti {

namespace Pango {
class FontDescription;
class TabArray;
}

namespace Gdk {
class Color;
class Bitmap;
class Event;
class Pixmap;
}

namespace Gtk {

class TextIter;
class TextTagClass;
class TextTagTable;
class TextTagTableClass;

//! @class TextAppearance texttag.h inti/gtk/texttag.h
//! @brief A GtkTextAppearance C++ wrapper class.
//!
//! TextAppearance is used to describe the attributes of the some text. You can get
//! the text appearance attributes by calling Gtk::TextAttributes::appearance().

class TextAppearance
{
	GtkTextAppearance appearence_;

public:
//! @name Accessors
//! @{

	GtkTextAppearance* gtk_text_appearance() const { return const_cast<GtkTextAppearance*>(&appearence_); }
	//!< Get a pointer to the GtkTextAppearance structure.

	Gdk::Color bg_color() const;
	//!< Gets the text background color.

	Gdk::Color fg_color() const;
	//!< Gets the text forground color.

	Gdk::Bitmap* bg_stipple() const;
	//!< Gets the text background stipple bitmap, or null if there isn't one.

	Gdk::Bitmap* fg_stipple() const;
	//!< Gets the text forground stipple bitmap, or null if there isn't one.

	int rise() const;
	//!< Gets the super/subscript rise, can be negative.

	Pango::Underline underline() const;
	//!< Get the current pango underline value.

	bool strikethrough() const;
	//!< Returns whether the text has strike through enabled.

//! @}
//! @name Methods
//! @{

	void set_colors(const Gdk::Color& fg_color, const Gdk::Color& bg_color);
	//!< Set the text foreground and background colors.
	//!< @param fg_color The foreground color.
	//!< @param bg_color The background color.

	void set_bg_color(const Gdk::Color& color);
	//!< Set the text background color.
	//!< @param color The background color.

	void set_fg_color(const Gdk::Color& color);
	//!< Set the text foreground color.
	//!< @param color The foreground color.

	void set_stipple(const Gdk::Bitmap& fg_mask, const Gdk::Bitmap& bg_mask);
	//!< Set the text foreground and background stipple bitmaps.
	//!< @param fg_mask The foreground bitmap.
	//!< @param bg_mask The background bitmap.

	void set_bg_stipple(const Gdk::Bitmap& mask);
	//!< Set the text background stipple bitmap.
	//!< @param mask The background bitmap.

	void set_fg_stipple(const Gdk::Bitmap& mask);
	//!< Set the text foreground stipple bitmap.
	//!< @param mask The foreground bitmap.

	void set_rise(int rise);
	//!< Sets the super/subscript rise.
	//!< @param rise The rise, can be negative.

	void set_underline(Pango::Underline underline);
	//!< Sets the Pango::Underline type.
	//!< @param underline One of the values from the Pango::Underline enumeration.

	void set_strikethrough(bool setting);
	//!< Sets whether the text should have strike through enabled.
	//!< @param setting <EM>true</EM> to enable strike through.
	
//! @}
};

//! @class TextAttributes texttag.h inti/gtk/texttag.h
//! @brief A GtkTextAttributes C++ wrapper class.
//!
//! TextAttributes describes a set of properties on some text. Using TextAttributes directly
//! should rarely be necessary. It's primarily useful with Gtk::TextIter::get_attributes().

class TextAttributes : public G::Boxed
{
public:
//! @name Constructors
//! @{

	TextAttributes();
	//!< Construct a TextAttributes.

	explicit TextAttributes(GtkTextAttributes *values);
	//!< Construct a TextAttributes from an existing GtkTextAttributes.
	//!< @param values A pointer to a GtkTextAttributes.
	//!<
	//!< <BR>The <EM>values</EM> can be a newly created GtkTextAttributes or an existing
	//!< GtkTextAttributes. The TextAttributes object created is a temporary object. It
	//!< doesn't take over the ownership of GtkTextAttributes and GtkTextAttributes is 
	//!< not freed by the destructor.

	TextAttributes(GtkTextAttributes *values, bool copy);
	//!< Construct a TextAttributes from an existing GtkTextAttributes.
	//!< @param values A pointer to a GtkTextAttributes.
	//!< @param copy Whether the TextAttributes object should make a copy of GtkTextAttributes or not.
	//!<
	//!< <BR>The <EM>values</EM> can be a newly created GtkTextAttributes or an existing
	//!< GtkTextAttributes. If <EM>copy</EM> is true TextAttributes will make a copy of
	//!< GtkTextAttributes. If <EM>copy</EM> is false TextAttributes wont make a copy but
	//!< instead takes over the ownership of GtkTextAttributes. Either way, the destructor
	//!< will free GtkTextAttributes when the TextAttributes object is destroyed. This
	//!< constructor is used by G::Boxed::wrap() to wrap GtkTextAttributes objects in a
	//!< C++ wrapper.

	explicit TextAttributes(const TextAttributes& src);
	//!< Copy constructor.
	//!< @param src The source attributes.

	~TextAttributes();
	//!< Destructor.

	TextAttributes& operator=(const TextAttributes& src);
	//!< Assignment operator.
	//!< @param src The source attributes.

//! @}
//! @name Accessors
//! @{

	GtkTextAttributes* gtk_text_attributes() const { return (GtkTextAttributes*)boxed_; }
	//!< Get a pointer to the GtkTextAttributes structure.

	operator GtkTextAttributes* () const;
	//!< Conversion operator; safely converts a TextAttributes to a GtkTextAttributes pointer.

	TextAppearance& appearance() const;
	//!< Obtains the appearance attributes of some text.

	Justification justification() const;
	//!< Returns the current text justification: left, right, or center.

	TextDirection direction() const;
	//!< Returns the current text direction.

	Pointer<Pango::FontDescription> font() const;
	//!< Gets the font description for the some text.

	double font_scale() const;
	//!< Gets the font size as a scale factor relative to the default font size.

	int left_margin() const;
	//!< Gets the width of the left margin in pixels.

	int indent() const;
	//!< Gets the amount to indent the paragraph, in pixels.

	int right_margin() const;
	//!< Gets the width of the right margin in pixels.

	int pixels_above_lines() const;
	//!< Gets the number of pixels of blank space above paragraphs.

	int pixels_below_lines() const;
	//!< Gets the number of pixels of blank space below paragraphs.

	int pixels_inside_wrap() const;
	//!< Gets the number of pixels of blank space between wrapped lines in a paragraph.

	Pointer<Pango::TabArray> tabs() const;
	//!< Gets the custom tabs for the text.

	WrapMode wrap_mode() const;
	//!< Returns the current wrap mode: never, at word boundaries, or at character boundaries.

	PangoLanguage* language() const;
	//!< Gets the language the text is in, as an ISO code. Pango can use this as a hint when
	//!< rendering the text. If you don't understand this parameter, you probably don't need it.

	bool invisible() const;
	//!< Gets whether the text is hidden. Not implemented in GTK 2.0.

	bool bg_full_height() const;
	//!< Gets whether the background color fills the entire line height or only the height
	//!< of the tagged characters.

	bool editable() const;
	//!< Gets whether the text can be modified by the user.

	bool realized() const;
	//!< Gets whether the TextAttribute's colors have been allocated.

//! @}
//! @name Methods
//! @{

	void set_justification(Justification justification);
	//!< Sets the justification of the text.
	//!< @param justification One of the values from the Justification enumeration.

	void set_direction(TextDirection direction);
	//!< Sets the text direction.
	//!< @param direction One of the values from the TextDirection enumeration.

	void set_margins(int left_margin, int right_margin);
	//!< Set the text margins.
	//!< @param left_margin The width of the left margin in pixels.
	//!< @param right_margin The width of the right margin in pixels.

	void set_left_margin(int margin);
	//!< Set the left text margin.
	//!< @param margin The width of the left margin in pixels.

	void set_right_margin(int margin);
	//!< Set the right text margin.
	//!< @param margin The width of the right margin in pixels.

	void set_indent(int indent);
	//!< Set the paragraph indent in pixels.

	void set_pixels(int pixels_above_lines, int pixels_below_lines, int pixels_inside_wrap);
	//!< Sets the amount blank space in and around a paragraph.
	//!< @param pixels_above_lines The number of pixels of blank space above a paragraph.
	//!< @param pixels_below_lines The number of pixels of blank space below a paragraph.
	//!< @param pixels_inside_wrap The number of pixels of blank space between wrapped lines in a paragraph.

	void set_pixels_above_lines(int pixels);
	//!< Sets the amount blank space above a paragraph.
	//!< @param pixels The number of pixels of blank space.

	void set_pixels_below_lines(int pixels);
	//!< Sets the amount blank space below a paragraph.
	//!< @param pixels The number of pixels of blank space.

	void set_pixels_inside_wrap(int pixels);
	//!< Sets the amount blank space between wrapped lines in a paragraph.
	//!< @param pixels The number of pixels of blank space.

	void set_tab_array(Pango::TabArray& tabs);
	//!< Sets the custom tabs for the text.
	//!< @param tabs The array of tab stops for the text.

	void set_wrap_mode(WrapMode wrap_mode);
	//!< Sets the current wrap mode: never, at word boundaries, or at character boundaries.
	//!< @param wrap_mode One of the values from the WrapMode enumeration.

	void set_language(PangoLanguage *language);
	//!< Sets the language the text is in, as an ISO code.
	//!< @param language The Pango language code.
	//!< 
	//!< <BR>Pango can use this as a hint when rendering the text. If you don't understand
	//!< this parameter, you probably don't need it.

	void set_invisible(bool setting);
	//!< Sets whether the text is hidden.
	//!< @param setting <EM>true</EM> if the text is hidden.

	void set_bg_full_height(bool setting);
	//!< Sets whether the background color fills the entire line height or only the height
	//!< of the tagged characters.
	//!< @param setting <EM>true</EM> if the background color fills the entire line height.

	void set_editable(bool setting);
	//!< Sets whether the text is editable.
	
//! @}
};

//! @class TextTag texttag.h inti/gtk/texttag.h
//! @brief A GtkTextTag C++ wrapper class.
//!
//! Text in a buffer can be marked with tags. A tag is an attribute that can be applied
//! to some range of text. For example, a tag might be called "bold" and make the text
//! inside the tag bold. However, the tag concept is more general than that; tags don't
//! have to affect appearance. They can instead affect the behavior of mouse and key 
//! presses, "lock" a range of text so the user can't edit it, or countless other things.
//!
//! A tag is represented by a TextTag object. One TextTag can be applied to any number
//! of text ranges in any number of buffers. Each tag is stored in a TextTagTable. A tag
//! table defines a set of tags that can be used together. Each buffer has one tag table
//! associated with it; only tags from that tag table can be used with the buffer. A
//! single tag table can be shared between multiple buffers, however.
//!
//! Tags can have names, which is convenient sometimes (for example, you can name your tag
//! that makes things bold "bold"), but they can also be anonymous (which is convenient if
//! you're creating tags on-the-fly).

class TextTag : public G::Object
{
	friend class G::Object;
	friend class TextTagClass;

	TextTag(const TextTag&);
	TextTag& operator=(const TextTag&);

protected:
//! @name Constructors
//! @{

	explicit TextTag(GtkTextTag *tag, bool reference = true);
	//!< Construct a new TextTag from an existing GtkTextTag.
	//!< @param tag A pointer to a GtkTextTag.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>tag</EM> can be a newly created GtkTextTag or an existing
	//!< GtkTextTag (see G::Object::Object).

//! @}
//! @name Signal Handlers
//! @{
	bool on_event(G::Object& event_object, Gdk::Event& event, const TextIter& iter);
	//!< Called when the text tag receives a motion, button, or key event signal.
	//!< @param event_object The object that received the event, such as a widget.
	//!< @param event The event.
	//!< @param iter The location where the event was received.
	//!< @return <EM>true</EM> if the event was handled, <EM>false</EM> if it wasn't handled.

//! @}
//  Properties
	
	typedef G::Property<String> NamePropertyType;
	typedef G::PropertyProxy<G::Object, NamePropertyType> NamePropertyProxy;
	static const NamePropertyType name_property;

	typedef G::WritableProperty<String> BackgroundPropertyType;
	typedef G::PropertyProxy<G::Object, BackgroundPropertyType> BackgroundPropertyProxy;
	static const BackgroundPropertyType background_property;

	typedef G::Property<Pointer<Gdk::Color> > BackgroundGdkPropertyType;
	typedef G::PropertyProxy<G::Object, BackgroundGdkPropertyType> BackgroundGdkPropertyProxy;
	static const BackgroundGdkPropertyType background_gdk_property;

	typedef G::Property<bool> BackgroundFullHeightPropertyType;
	typedef G::PropertyProxy<G::Object, BackgroundFullHeightPropertyType> BackgroundFullHeightPropertyProxy;
	static const BackgroundFullHeightPropertyType background_full_height_property;

	typedef G::Property<Gdk::Bitmap*, G::Object*> BackgroundStipplePropertyType;
	typedef G::PropertyProxy<G::Object, BackgroundStipplePropertyType> BackgroundStipplePropertyProxy;
	static const BackgroundStipplePropertyType background_stipple_property;

	typedef G::WritableProperty<String> ForegroundPropertyType;
	typedef G::PropertyProxy<G::Object, ForegroundPropertyType> ForegroundPropertyProxy;
	static const ForegroundPropertyType foreground_property;

	typedef G::Property<Pointer<Gdk::Color> > ForegroundGdkPropertyType;
	typedef G::PropertyProxy<G::Object, ForegroundGdkPropertyType> ForegroundGdkPropertyProxy;
	static const ForegroundGdkPropertyType foreground_gdk_property;

	typedef G::Property<Gdk::Bitmap*, G::Object*> ForegroundStipplePropertyType;
	typedef G::PropertyProxy<G::Object, ForegroundStipplePropertyType> ForegroundStipplePropertyProxy;
	static const ForegroundStipplePropertyType foreground_stipple_property;

	typedef G::Property<TextDirection, int> DirectionPropertyType;
	typedef G::PropertyProxy<G::Object, DirectionPropertyType> DirectionPropertyProxy;
	static const DirectionPropertyType direction_property;

	typedef G::Property<bool> EditablePropertyType;
	typedef G::PropertyProxy<G::Object, EditablePropertyType> EditablePropertyProxy;
	static const EditablePropertyType editable_property;

	typedef G::Property<String> FontPropertyType;
	typedef G::PropertyProxy<G::Object, FontPropertyType> FontPropertyProxy;
	static const FontPropertyType font_property;

	typedef G::Property<Pointer<Pango::FontDescription> > FontDescPropertyType;
	typedef G::PropertyProxy<G::Object, FontDescPropertyType> FontDescPropertyProxy;
	static const FontDescPropertyType font_desc_property;

	typedef G::Property<String> FamilyPropertyType;
	typedef G::PropertyProxy<G::Object, FamilyPropertyType> FamilyPropertyProxy;
	static const FamilyPropertyType family_property;

	typedef G::Property<Pango::Style, int> StylePropertyType;
	typedef G::PropertyProxy<G::Object, StylePropertyType> StylePropertyProxy;
	static const StylePropertyType style_property;

	typedef G::Property<Pango::Variant, int> VariantPropertyType;
	typedef G::PropertyProxy<G::Object, VariantPropertyType> VariantPropertyProxy;
	static const VariantPropertyType variant_property;

	typedef G::Property<Pango::Weight, int> WeightPropertyType;
	typedef G::PropertyProxy<G::Object, WeightPropertyType> WeightPropertyProxy;
	static const WeightPropertyType weight_property;

	typedef G::Property<Pango::Stretch, int> StretchPropertyType;
	typedef G::PropertyProxy<G::Object, StretchPropertyType> StretchPropertyProxy;
	static const StretchPropertyType stretch_property;

	typedef G::Property<int> SizePropertyType;
	typedef G::PropertyProxy<G::Object, SizePropertyType> SizePropertyProxy;
	static const SizePropertyType size_property;

	typedef G::Property<double> ScalePropertyType;
	typedef G::PropertyProxy<G::Object, ScalePropertyType> ScalePropertyProxy;
	static const ScalePropertyType scale_property;

	typedef G::Property<double> SizePointsPropertyType;
	typedef G::PropertyProxy<G::Object, SizePointsPropertyType> SizePointsPropertyProxy;
	static const SizePointsPropertyType size_points_property;

	typedef G::Property<Justification, int> JustificationPropertyType;
	typedef G::PropertyProxy<G::Object, JustificationPropertyType> JustificationPropertyProxy;
	static const JustificationPropertyType justification_property;

	typedef G::Property<String> LanguagePropertyType;
	typedef G::PropertyProxy<G::Object, LanguagePropertyType> LanguagePropertyProxy;
	static const LanguagePropertyType language_property;

	typedef G::Property<int> LeftMarginPropertyType;
	typedef G::PropertyProxy<G::Object, LeftMarginPropertyType> LeftMarginPropertyProxy;
	static const LeftMarginPropertyType left_margin_property;

	typedef G::Property<int> RightMarginPropertyType;
	typedef G::PropertyProxy<G::Object, RightMarginPropertyType> RightMarginPropertyProxy;
	static const RightMarginPropertyType right_margin_property;

	typedef G::Property<int> IndentPropertyType;
	typedef G::PropertyProxy<G::Object, IndentPropertyType> IndentPropertyProxy;
	static const IndentPropertyType indent_property;

	typedef G::Property<int> RisePropertyType;
	typedef G::PropertyProxy<G::Object, RisePropertyType> RisePropertyProxy;
	static const RisePropertyType rise_property;

	typedef G::Property<int> PixelsAboveLinesPropertyType;
	typedef G::PropertyProxy<G::Object, PixelsAboveLinesPropertyType> PixelsAboveLinesPropertyProxy;
	static const PixelsAboveLinesPropertyType pixels_above_lines_property;

	typedef G::Property<int> PixelsBelowLinesPropertyType;
	typedef G::PropertyProxy<G::Object, PixelsBelowLinesPropertyType> PixelsBelowLinesPropertyProxy;
	static const PixelsBelowLinesPropertyType pixels_below_lines_property;

	typedef G::Property<int> PixelsInsideWrapPropertyType;
	typedef G::PropertyProxy<G::Object, PixelsInsideWrapPropertyType> PixelsInsideWrapPropertyProxy;
	static const PixelsInsideWrapPropertyType pixels_inside_wrap_property;

	typedef G::Property<bool> StrikethroughPropertyType;
	typedef G::PropertyProxy<G::Object, StrikethroughPropertyType> StrikethroughPropertyProxy;
	static const StrikethroughPropertyType strikethrough_property;

	typedef G::Property<Pango::Underline, int> UnderlinePropertyType;
	typedef G::PropertyProxy<G::Object, UnderlinePropertyType> UnderlinePropertyProxy;
	static const UnderlinePropertyType underline_property;

	typedef G::Property<WrapMode, int> WrapModePropertyType;
	typedef G::PropertyProxy<G::Object, WrapModePropertyType> WrapModePropertyProxy;
	static const WrapModePropertyType wrap_mode_property;

	typedef G::Property<Pointer<Pango::TabArray> > TabsPropertyType;
	typedef G::PropertyProxy<G::Object, TabsPropertyType> TabsPropertyProxy;
	static const TabsPropertyType tabs_property;

	typedef G::Property<bool> InvisiblePropertyType;
	typedef G::PropertyProxy<G::Object, InvisiblePropertyType> InvisiblePropertyProxy;
	static const InvisiblePropertyType invisible_property;

	typedef G::Property<bool> BackgroundSetPropertyType;
	typedef G::PropertyProxy<G::Object, BackgroundSetPropertyType> BackgroundSetPropertyProxy;
	static const BackgroundSetPropertyType background_set_property;

	typedef G::Property<bool> BackgroundFullHeightSetPropertyType;
	typedef G::PropertyProxy<G::Object, BackgroundFullHeightSetPropertyType> BackgroundFullHeightSetPropertyProxy;
	static const BackgroundFullHeightSetPropertyType background_full_height_set_property;

	typedef G::Property<bool> BackgroundStippleSetPropertyType;
	typedef G::PropertyProxy<G::Object, BackgroundStippleSetPropertyType> BackgroundStippleSetPropertyProxy;
	static const BackgroundStippleSetPropertyType background_stipple_set_property;

	typedef G::Property<bool> ForegroundSetPropertyType;
	typedef G::PropertyProxy<G::Object, ForegroundSetPropertyType> ForegroundSetPropertyProxy;
	static const ForegroundSetPropertyType foreground_set_property;

	typedef G::Property<bool> ForegroundStippleSetPropertyType;
	typedef G::PropertyProxy<G::Object, ForegroundStippleSetPropertyType> ForegroundStippleSetPropertyProxy;
	static const ForegroundStippleSetPropertyType foreground_stipple_set_property;

	typedef G::Property<bool> EditableSetPropertyType;
	typedef G::PropertyProxy<G::Object, EditableSetPropertyType> EditableSetPropertyProxy;
	static const EditableSetPropertyType editable_set_property;

	typedef G::Property<bool> FamilySetPropertyType;
	typedef G::PropertyProxy<G::Object, FamilySetPropertyType> FamilySetPropertyProxy;
	static const FamilySetPropertyType family_set_property;

	typedef G::Property<bool> StyleSetPropertyType;
	typedef G::PropertyProxy<G::Object, StyleSetPropertyType> StyleSetPropertyProxy;
	static const StyleSetPropertyType style_set_property;

	typedef G::Property<bool> VariantSetPropertyType;
	typedef G::PropertyProxy<G::Object, VariantSetPropertyType> VariantSetPropertyProxy;
	static const VariantSetPropertyType variant_set_property;

	typedef G::Property<bool> WeightSetPropertyType;
	typedef G::PropertyProxy<G::Object, WeightSetPropertyType> WeightSetPropertyProxy;
	static const WeightSetPropertyType weight_set_property;

	typedef G::Property<bool> StretchSetPropertyType;
	typedef G::PropertyProxy<G::Object, StretchSetPropertyType> StretchSetPropertyProxy;
	static const StretchSetPropertyType stretch_set_property;

	typedef G::Property<bool> SizeSetPropertyType;
	typedef G::PropertyProxy<G::Object, SizeSetPropertyType> SizeSetPropertyProxy;
	static const SizeSetPropertyType size_set_property;

	typedef G::Property<bool> ScaleSetPropertyType;
	typedef G::PropertyProxy<G::Object, ScaleSetPropertyType> ScaleSetPropertyProxy;
	static const ScaleSetPropertyType scale_set_property;

	typedef G::Property<bool> JustificationSetPropertyType;
	typedef G::PropertyProxy<G::Object, JustificationSetPropertyType> JustificationSetPropertyProxy;
	static const JustificationSetPropertyType justification_set_property;

	typedef G::Property<bool> LanguageSetPropertyType;
	typedef G::PropertyProxy<G::Object, LanguageSetPropertyType> LanguageSetPropertyProxy;
	static const LanguageSetPropertyType language_set_property;

	typedef G::Property<bool> LeftMarginSetPropertyType;
	typedef G::PropertyProxy<G::Object, LeftMarginSetPropertyType> LeftMarginSetPropertyProxy;
	static const LeftMarginSetPropertyType left_margin_set_property;

	typedef G::Property<bool> IndentSetPropertyType;
	typedef G::PropertyProxy<G::Object, IndentSetPropertyType> IndentSetPropertyProxy;
	static const IndentSetPropertyType indent_set_property;

	typedef G::Property<bool> RiseSetPropertyType;
	typedef G::PropertyProxy<G::Object, RiseSetPropertyType> RiseSetPropertyProxy;
	static const RiseSetPropertyType rise_set_property;

	typedef G::Property<bool> PixelsAboveLinesSetPropertyType;
	typedef G::PropertyProxy<G::Object, PixelsAboveLinesSetPropertyType> PixelsAboveLinesSetPropertyProxy;
	static const PixelsAboveLinesSetPropertyType pixels_above_lines_set_property;

	typedef G::Property<bool> PixelsBelowLinesSetPropertyType;
	typedef G::PropertyProxy<G::Object, PixelsBelowLinesSetPropertyType> PixelsBelowLinesSetPropertyProxy;
	static const PixelsBelowLinesSetPropertyType pixels_below_lines_set_property;

	typedef G::Property<bool> PixelsInsideWrapSetPropertyType;
	typedef G::PropertyProxy<G::Object, PixelsInsideWrapSetPropertyType> PixelsInsideWrapSetPropertyProxy;
	static const PixelsInsideWrapSetPropertyType pixels_inside_wrap_set_property;

	typedef G::Property<bool> StrikethroughSetPropertyType;
	typedef G::PropertyProxy<G::Object, StrikethroughSetPropertyType> StrikethroughSetPropertyProxy;
	static const StrikethroughSetPropertyType strikethrough_set_property;

	typedef G::Property<bool> RightMarginSetPropertyType;
	typedef G::PropertyProxy<G::Object, RightMarginSetPropertyType> RightMarginSetPropertyProxy;
	static const RightMarginSetPropertyType right_margin_set_property;

	typedef G::Property<bool> UnderlineSetPropertyType;
	typedef G::PropertyProxy<G::Object, UnderlineSetPropertyType> UnderlineSetPropertyProxy;
	static const UnderlineSetPropertyType underline_set_property;

	typedef G::Property<bool> WrapModeSetPropertyType;
	typedef G::PropertyProxy<G::Object, WrapModeSetPropertyType> WrapModeSetPropertyProxy;
	static const WrapModeSetPropertyType wrap_mode_set_property;

	typedef G::Property<bool> TabsSetPropertyType;
	typedef G::PropertyProxy<G::Object, TabsSetPropertyType> TabsSetPropertyProxy;
	static const TabsSetPropertyType tabs_set_property;

	typedef G::Property<bool> InvisibleSetPropertyType;
	typedef G::PropertyProxy<G::Object, InvisibleSetPropertyType> InvisibleSetPropertyProxy;
	static const InvisibleSetPropertyType invisible_set_property;

//  Signals
	
	typedef G::Signal3<bool, GObject*, GdkEvent*, const GtkTextIter*> EventSignalType;
	typedef G::SignalProxy<TypeInstance, EventSignalType> EventSignalProxy;
	static const EventSignalType event_signal;

public:
//! @name Constructors
//! @{

	TextTag(const char *name = 0);
	//!< Construct a new text tag.
	//!< @param name The name for this tag, or null for an anonymous tag.

	virtual ~TextTag();
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	GtkTextTag* gtk_text_tag() const { return (GtkTextTag*)instance; }
	//!< Get a pointer to the GtkTextTag structure.

	GtkTextTagClass* gtk_text_tag_class() const;
	//!< Get a pointer to the GtkTextTagClass structure.

	operator GtkTextTag* () const;
	//!< Conversion operator; safely converts a TextTag to a GtkTextTag pointer.

	TextTagTable* table() const;
	//!< Returns the tag table for the tag.
	//!< @return The tag table.

	Pointer<TextAttributes> values() const;
	//!< Returns a smart pointer to the TextAttributes the tag uses for displaying text.
	//!< The information specified acts an override on information specified by
	//!< lower-priority tags. If no value is specified, then the next-lower-priority
	//!< tag on the text determines the value. The text widget itself provides defaults
	//!< if no tag specifies an override.

	int get_priority() const;
	//!< Gets the tag priority.
	//!< @return The tag's priority.

//! @}
//! @name Methods
//! @{

	void set_priority(int priority);
	//!< Sets the priority of the TextTag.
	//!< @param priority The new priority.
	//!<
	//!< <BR>Valid priorities are start at 0 and go to one less than Gtk::TextTagTable::get_size().
	//!< Each tag in a table has a unique priority; setting the priority of one tag shifts the 
	//!< priorities of all the other tags in the table to maintain a unique priority for each tag.
	//!< Higher priority tags "win" if two tags both set the same text attribute. When adding a 
	//!< tag to a tag table, it will be assigned the highest priority in the table by default; 
	//!< so normally the precedence of a set of tags is the order in which they were added to the
	//!< table, or created with gtk_text_buffer_create_tag(), which adds the tag to the buffer's
	//!< table automatically.

	bool event(G::Object& event_object, Gdk::Event& event, const TextIter& iter);
	//!< Emits the <EM>event</EM> signal on the TextTag.
	//!< @param event_object The object that received the event, such as a widget.
	//!< @param event The event.
	//!< @param iter The location where the event was received.
	//!< @return The result of signal emission (whether the event was handled).
	
//! @}
//! @name Property Proxies
//! @{
	
	const NamePropertyProxy prop_name()
	{
		return NamePropertyProxy(this, &name_property);
	}
	//!< The name used to refer to the text tag; null for anonymous tags
	//!< (String : Read / Write / Construct Only).

	const BackgroundPropertyProxy prop_background()
	{
		return BackgroundPropertyProxy(this, &background_property);
	}
	//!< The background color as a string (String : Write).

	const BackgroundGdkPropertyProxy prop_background_gdk()
	{
		return BackgroundGdkPropertyProxy(this, &background_gdk_property);
	}
	//!< The background color as a (possibly unallocated) Gdk::Color (Pointer<Gdk::Color> : Read / Write).

	const BackgroundFullHeightPropertyProxy prop_background_full_height()
	{
		return BackgroundFullHeightPropertyProxy(this, &background_full_height_property);
	}
	//!< Whether the background color fills the entire line height or only the height
	//!< of the tagged characters (bool : Read / Write).

	const BackgroundStipplePropertyProxy prop_background_stipple()
	{
		return BackgroundStipplePropertyProxy(this, &background_stipple_property);
	}
	//!< The bitmap to use as a mask when drawing the text background (Gdk::Bitmap* : Read / Write).

	const ForegroundPropertyProxy prop_foreground()
	{
		return ForegroundPropertyProxy(this, &foreground_property);
	}
	//!< The foreground color as a string (String : Write).

	const ForegroundGdkPropertyProxy prop_foreground_gdk()
	{
		return ForegroundGdkPropertyProxy(this, &foreground_gdk_property);
	}
	//!< The foreground color as a (possibly unallocated) Gdk::Color (Pointer<Gdk::Color> : Read / Write).

	const ForegroundStipplePropertyProxy prop_foreground_stipple()
	{
		return ForegroundStipplePropertyProxy(this, &foreground_stipple_property);
	}
	//!< The bitmap to use as a mask when drawing the text foreground (Gdk::Bitmap* : Read / Write).

	const DirectionPropertyProxy prop_direction()
	{
		return DirectionPropertyProxy(this, &direction_property);
	}
	//!< The text direction, for example, right-to-left or left-to-right (TextDirection : Read / Write).

	const EditablePropertyProxy prop_editable()
	{
		return EditablePropertyProxy(this, &editable_property);
	}
	//!< Whether the text can be modified by the user (bool : Read / Write).

	const FontPropertyProxy prop_font()
	{
		return FontPropertyProxy(this, &font_property);
	}
	//!< The font description as a string, e.g. "Sans Italic 12" (String : Read / Write).

	const FontDescPropertyProxy prop_font_desc()
	{
		return FontDescPropertyProxy(this, &font_desc_property);
	}
	//!< The font description as a Pango::FontDescription (Pointer<Pango::FontDescription> : Read / Write).

	const FamilyPropertyProxy prop_family()
	{
		return FamilyPropertyProxy(this, &family_property);
	}
	//!< The name of the font family, e.g. Sans, Helvetica, Times, Monospace (String : Read / Write).

	const StylePropertyProxy prop_style()
	{
		return StylePropertyProxy(this, &style_property);
	}
	//!< The font style as a Pango::Style, e.g. Pango::STYLE_ITALIC (Pango::Style : Read / Write).

	const VariantPropertyProxy prop_variant()
	{
		return VariantPropertyProxy(this, &variant_property);
	}
	//!< The font variant as a Pango::Variant, e.g. Pango::VARIANT_SMALL_CAPS (Pango::Variant : Read / Write).

	const WeightPropertyProxy prop_weight()
	{
		return WeightPropertyProxy(this, &weight_property);
	}
	//!< The font weight as Pango::Weight, for example, Pango::WEIGHT_BOLD (Pango::Weight : Read / Write).

	const StretchPropertyProxy prop_stretch()
	{
		return StretchPropertyProxy(this, &stretch_property);
	}
	//!< The font stretch as a Pango::Stretch, e.g. Pango::STRETCH_CONDENSED (Pango::Stretch : Read / Write).

	const SizePropertyProxy prop_size()
	{
		return SizePropertyProxy(this, &size_property);
	}
	//!< The font size in Pango units (int : Read / Write).

	const ScalePropertyProxy prop_scale()
	{
		return ScalePropertyProxy(this, &scale_property);
	}
	//!< The font size as a scale factor relative to the default font size. This properly
	//!< adapts to theme changes etc. so is recommended. Pango predefines some scales such
	//!< as PANGO_SCALE_X_LARGE (double : Read / Write).

	const SizePointsPropertyProxy prop_size_points()
	{
		return SizePointsPropertyProxy(this, &size_points_property);
	}
	//!< The font size in points (double : Read / Write).

	const JustificationPropertyProxy prop_justification()
	{
		return JustificationPropertyProxy(this, &justification_property);
	}
	//!< Left, right, or center justification (Justification : Read / Write).

	const LanguagePropertyProxy prop_language()
	{
		return LanguagePropertyProxy(this, &language_property);
	}
	//!< The language this text is in, as an ISO code. Pango can use this as a hint when
	//!< rendering the text. If you don't understand this parameter, you probably don't
	//!< need it  (String : Read / Write).

	const LeftMarginPropertyProxy prop_left_margin()
	{
		return LeftMarginPropertyProxy(this, &left_margin_property);
	}
	//!< The width of the left margin in pixels (int : Read / Write).

	const RightMarginPropertyProxy prop_right_margin()
	{
		return RightMarginPropertyProxy(this, &right_margin_property);
	}
	//!< THe width of the right margin in pixels (int : Read / Write).

	const IndentPropertyProxy prop_indent()
	{
		return IndentPropertyProxy(this, &indent_property);
	}
	//!< The amount to indent the paragraph, in pixels (int : Read / Write).

	const RisePropertyProxy prop_rise()
	{
		return RisePropertyProxy(this, &rise_property);
	}
	//!< The offset of text above the baseline (below the baseline if rise is negative)
	//!< in pixels (int : Read / Write).

	const PixelsAboveLinesPropertyProxy prop_pixels_above_lines()
	{
		return PixelsAboveLinesPropertyProxy(this, &pixels_above_lines_property);
	}
	//!< The pixels of blank space above paragraphs (int : Read / Write).

	const PixelsBelowLinesPropertyProxy prop_pixels_below_lines()
	{
		return PixelsBelowLinesPropertyProxy(this, &pixels_below_lines_property);
	}
	//!< The pixels of blank space below paragraphs (int : Read / Write).

	const PixelsInsideWrapPropertyProxy prop_pixels_inside_wrap()
	{
		return PixelsInsideWrapPropertyProxy(this, &pixels_inside_wrap_property);
	}
	//!< The pixels of blank space between wrapped lines in a paragraph (int : Read / Write).

	const StrikethroughPropertyProxy prop_strikethrough()
	{
		return StrikethroughPropertyProxy(this, &strikethrough_property);
	}
	//!< Whether to strike through the text (bool : Read / Write).

	const UnderlinePropertyProxy prop_underline()
	{
		return UnderlinePropertyProxy(this, &underline_property);
	}
	//!< The style of underline for this text (Pango::Underline : Read / Write).

	const WrapModePropertyProxy prop_wrap_mode()
	{
		return WrapModePropertyProxy(this, &wrap_mode_property);
	}
	//!< Whether to wrap lines never, at word boundaries, or at character boundaries 
	//!< (WrapMode : Read / Write).

	const TabsPropertyProxy prop_tabs()
	{
		return TabsPropertyProxy(this, &tabs_property);
	}
	//!< The custom tabs for this text (Pointer<Pango::TabArray> : Read / Write).

	const InvisiblePropertyProxy prop_invisible()
	{
		return InvisiblePropertyProxy(this, &invisible_property);
	}
	//!< Whether this text is hidden. Not implemented in GTK 2.0 (bool : Read / Write).

	const BackgroundSetPropertyProxy prop_background_set()
	{
		return BackgroundSetPropertyProxy(this, &background_set_property);
	}
	//!< Whether this tag affects the background color (bool : Read / Write).

	const BackgroundFullHeightSetPropertyProxy prop_background_full_height_set()
	{
		return BackgroundFullHeightSetPropertyProxy(this, &background_full_height_set_property);
	}
	//!< Whether this tag affects background height (bool : Read / Write).

	const BackgroundStippleSetPropertyProxy prop_background_stipple_set()
	{
		return BackgroundStippleSetPropertyProxy(this, &background_stipple_set_property);
	}
	//!< Whether this tag affects the background stipple (bool : Read / Write).

	const ForegroundSetPropertyProxy prop_foreground_set()
	{
		return ForegroundSetPropertyProxy(this, &foreground_set_property);
	}
	//!< Whether this tag affects the foreground color (bool : Read / Write).

	const ForegroundStippleSetPropertyProxy prop_foreground_stipple_set()
	{
		return ForegroundStippleSetPropertyProxy(this, &foreground_stipple_set_property);
	}
	//!< Whether this tag affects the foreground stipple (bool : Read / Write).

	const EditableSetPropertyProxy prop_editable_set()
	{
		return EditableSetPropertyProxy(this, &editable_set_property);
	}
	//!< Whether this tag affects text editability (bool : Read / Write).

	const FamilySetPropertyProxy prop_family_set()
	{
		return FamilySetPropertyProxy(this, &family_set_property);
	}
	//!< Whether this tag affects the font family (bool : Read / Write).

	const StyleSetPropertyProxy prop_style_set()
	{
		return StyleSetPropertyProxy(this, &style_set_property);
	}
	//!< Whether this tag affects the font style (bool : Read / Write).

	const VariantSetPropertyProxy prop_variant_set()
	{
		return VariantSetPropertyProxy(this, &variant_set_property);
	}
	//!< Whether this tag affects the font variant (bool : Read / Write).

	const WeightSetPropertyProxy prop_weight_set()
	{
		return WeightSetPropertyProxy(this, &weight_set_property);
	}
	//!< Whether this tag affects the font weight (bool : Read / Write).

	const StretchSetPropertyProxy prop_stretch_set()
	{
		return StretchSetPropertyProxy(this, &stretch_set_property);
	}
	//!< Whether this tag affects the font stretch (bool : Read / Write).

	const SizeSetPropertyProxy prop_size_set()
	{
		return SizeSetPropertyProxy(this, &size_set_property);
	}
	//!< Whether this tag affects the font size  (bool : Read / Write).

	const ScaleSetPropertyProxy prop_scale_set()
	{
		return ScaleSetPropertyProxy(this, &scale_set_property);
	}
	//!< Whether this tag scales the font size by a factor (bool : Read / Write).

	const JustificationSetPropertyProxy prop_justification_set()
	{
		return JustificationSetPropertyProxy(this, &justification_set_property);
	}
	//!< Whether this tag affects paragraph justification (bool : Read / Write).

	const LanguageSetPropertyProxy prop_language_set()
	{
		return LanguageSetPropertyProxy(this, &language_set_property);
	}
	//!< Whether this tag affects the language the text is rendered in (bool : Read / Write).

	const LeftMarginSetPropertyProxy prop_left_margin_set()
	{
		return LeftMarginSetPropertyProxy(this, &left_margin_set_property);
	}
	//!< Whether this tag affects the left margin (bool : Read / Write).

	const IndentSetPropertyProxy prop_indent_set()
	{
		return IndentSetPropertyProxy(this, &indent_set_property);
	}
	//!< Whether this tag affects indentation (bool : Read / Write).

	const RiseSetPropertyProxy prop_rise_set()
	{
		return RiseSetPropertyProxy(this, &rise_set_property);
	}
	//!< Whether this tag affects the rise (bool : Read / Write).

	const PixelsAboveLinesSetPropertyProxy prop_pixels_above_lines_set()
	{
		return PixelsAboveLinesSetPropertyProxy(this, &pixels_above_lines_set_property);
	}
	//!< Whether this tag affects the number of pixels above lines (bool : Read / Write).

	const PixelsBelowLinesSetPropertyProxy prop_pixels_below_lines_set()
	{
		return PixelsBelowLinesSetPropertyProxy(this, &pixels_below_lines_set_property);
	}
	//!< Whether this tag affects the number of pixels below lines (bool : Read / Write).

	const PixelsInsideWrapSetPropertyProxy prop_pixels_inside_wrap_set()
	{
		return PixelsInsideWrapSetPropertyProxy(this, &pixels_inside_wrap_set_property);
	}
	//!< Whether this tag affects the number of pixels between wrapped lines (bool : Read / Write).

	const StrikethroughSetPropertyProxy prop_strikethrough_set()
	{
		return StrikethroughSetPropertyProxy(this, &strikethrough_set_property);
	}
	//!< Whether this tag affects strikethrough (bool : Read / Write).

	const RightMarginSetPropertyProxy prop_right_margin_set()
	{
		return RightMarginSetPropertyProxy(this, &right_margin_set_property);
	}
	//!< Whether this tag affects the right margin (bool : Read / Write).

	const UnderlineSetPropertyProxy prop_underline_set()
	{
		return UnderlineSetPropertyProxy(this, &underline_set_property);
	}
	//!< Whether this tag affects underlining (bool : Read / Write).

	const WrapModeSetPropertyProxy prop_wrap_mode_set()
	{
		return WrapModeSetPropertyProxy(this, &wrap_mode_set_property);
	}
	//!< Whether this tag affects line wrap mode (bool : Read / Write).

	const TabsSetPropertyProxy prop_tabs_set()
	{
		return TabsSetPropertyProxy(this, &tabs_set_property);
	}
	//!< Whether this tag affects tabs (bool : Read / Write).

	const InvisibleSetPropertyProxy prop_invisible_set()
	{
		return InvisibleSetPropertyProxy(this, &invisible_set_property);
	}
	//!< Whether this tag affects text visibility (bool : Read / Write).

//! @}
//! @name Signal Proxies
//! @{
	
	const EventSignalProxy sig_event()
	{
		return EventSignalProxy(this, &event_signal);
	}
	//!< Connect to the event_signal; emitted when the text tag receives a motion, button, or key event signal.
	
//! @}
};

//! @class TextTagTable texttag.h inti/gtk/texttag.h
//! @brief A GtkTextTagTable C++ wrapper class.
//!
//! TextTags are stored in a TexTagTable. A tag table defines a set of tags
//! that can be used together. Each buffer has one tag table associated with
//! it; only tags from that tag table can be used with the buffer. A single
//! tag table can be shared between multiple buffers, however.

class TextTagTable : public G::Object
{
	friend class G::Object;
	friend class TextTagTableClass;

	TextTagTable(const TextTagTable&);
	TextTagTable& operator=(const TextTagTable&);

protected:
//! @name Constructors
//! @{

	explicit TextTagTable(GtkTextTagTable *table, bool reference = true);
	//!< Construct a new TextTagTable from an existing GtkTextTagTable.
	//!< @param table A pointer to a GtkTextTagTable.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>table</EM> can be a newly created GtkTextTagTable or an existing
	//!< GtkTextTagTable (see G::Object::Object).

//! @}
//! @name Signal Handlers
//! @{

	virtual void on_tag_changed(TextTag& tag, bool size_changed);
	//!< Called whenever a tag property is changed.
	//!< @param tag The tag whose property was changed.
	//!< @param size_changed <EM>true</EM> if the changed property resulted in a change
	//!< to the text display size.

	virtual void on_tag_added(TextTag& tag);
	//!< Called whenever a tag is added to the table.
	//!< @param tag The tag being added to the table.

	virtual void on_tag_removed(TextTag& tag);
	//!< Called whenever a tag is removed from the table.
	//!< @param tag The tag being removed to the table.

//! @}
//  Signals

	typedef G::Signal2<void, GtkTextTag*, bool> TagChangedSignalType;
	typedef G::SignalProxy<TypeInstance, TagChangedSignalType> TagChangedSignalProxy;
	static const TagChangedSignalType tag_changed_signal;

	typedef G::Signal1<void, GtkTextTag*> TagAddedSignalType;
	typedef G::SignalProxy<TypeInstance, TagAddedSignalType> TagAddedSignalProxy;
	static const TagAddedSignalType tag_added_signal;

	typedef G::Signal1<void, GtkTextTag*> TagRemovedSignalType;
	typedef G::SignalProxy<TypeInstance, TagRemovedSignalType> TagRemovedSignalProxy;
	static const TagRemovedSignalType tag_removed_signal;

public:
	typedef Slot1<void, TextTag&> ForeachSlot;
	//!< Signature of the callback slot to be called on each tag in the table.
	//!< <B>Example:</B> Method signature for ForeachSlot.
	//!< @code
	//!< void method(TextTag& tag);
	//!<
	//!< // tag: A text tag.
	//!< @endcode

//! @name Constructors
//! @{

	TextTagTable();
	//!< Construct a new TextTagTable. The table contains no tags by default.

	virtual ~TextTagTable();
	//!< Destructor.
	
//! @}
//! @name Accessors
//! @{

	GtkTextTagTable* gtk_text_tag_table() const { return (GtkTextTagTable*)instance; }
	//!< Get a pointer to the GtkTextTagTable structure.

	GtkTextTagTableClass* gtk_text_tag_table_class() const;
	//!< Get a pointer to the GtkTextTagTableClass structure.

	operator GtkTextTagTable* () const;
	//!< Conversion operator; safely converts a TextTagTable to a GtkTextTagTable pointer.

	int get_size() const;
	//!< Returns the size of the table (number of tags)
	//!< @return The number of tags in the table.

	TextTag* lookup(const char *name);
	//!< Look up a named tag.
	//!< @param name The name of a tag.
	//!< @return The tag, or null if none by that name is in the table.

//! @}
//! @name Methods
//! @{

	void add(TextTag& tag);
	//!< Add a tag to the table. 
	//!< @param tag A TextTag.
	//!<
	//!< <BR>The tag is assigned the highest priority in the table. <EM>tag</EM> must not be
	//!< in a tag table already, and may not have the same name as an already-added tag.

	void remove(TextTag& tag);
	//!< Remove a tag from the table. 
	//!< @param tag A TextTag.
	//!<
	//!< <BR>This will remove the table's reference to the tag, so be careful - the tag will
	//!< end up destroyed if you don't have a reference to it.

	void foreach(const ForeachSlot *each);
	//!< Calls <EM>each</EM> on each tag in table.
	//!< @param each The callback slot to call on each tag.

//! @}
//! @name Signal Proxies
//! @{

	const TagChangedSignalProxy sig_tag_changed()
	{
		return TagChangedSignalProxy(this, &tag_changed_signal);
	}
	//!< Connect to the tag_changed_signal; emitted whenever a tag property is changed.

	const TagAddedSignalProxy sig_tag_added()
	{
		return TagAddedSignalProxy(this, &tag_added_signal);
	}
	//!< Connect to the tag_added_signal; emitted whenever a tag is added to the table.

	const TagRemovedSignalProxy sig_tag_removed()
	{
		return TagRemovedSignalProxy(this, &tag_removed_signal);
	}
	//!< Connect to the tag_removed_signal; emitted whenever a tag is removed from the table.

//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_TEXT_TAG_H

