/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  textbuffer.cc - GtkTextBuffer and GtkTextMark C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "textbuffer.h"
#include "private/textbuffer_p.h"
#include "clipboard.h"
#include "widget.h"
#include "../gdk-pixbuf/pixbuf.h"

using namespace Inti;

/*  Gtk::TextMark
 */
 
Gtk::TextMark::TextMark(GtkTextMark *mark, bool reference)
: G::Object((GObject*)mark, reference)
{
}
	
Gtk::TextMark::~TextMark() 
{
}
	
GtkTextMarkClass* 
Gtk::TextMark::gtk_text_mark_class() const 
{ 
	return get_class<GtkTextMarkClass>(); 
}
	
Gtk::TextMark::operator GtkTextMark* () const 
{ 
	return this ? gtk_text_mark() : 0; 
}

bool
Gtk::TextMark::get_visible() const
{
	return gtk_text_mark_get_visible(gtk_text_mark());
}

String
Gtk::TextMark::get_name() const
{
	return gtk_text_mark_get_name(gtk_text_mark());
}

bool 
Gtk::TextMark::get_deleted() const
{
	return gtk_text_mark_get_deleted(gtk_text_mark());
}

Gtk::TextBuffer*
Gtk::TextMark::get_buffer() const
{
	return G::Object::wrap<TextBuffer>(gtk_text_mark_get_buffer(gtk_text_mark()));
}

bool 
Gtk::TextMark::get_left_gravity() const
{
	return gtk_text_mark_get_left_gravity(gtk_text_mark());
}

void 
Gtk::TextMark::set_visible(bool setting)
{
	gtk_text_mark_set_visible(gtk_text_mark(), setting);
}

/*  Gtk::TextMarkClass
 */

void
Gtk::TextMarkClass::init(GtkTextMarkClass *g_class)
{
	G::ObjectClass::init((GObjectClass*)g_class);
}

GType
Gtk::TextMarkClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_TEXT_MARK, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::TextMarkClass::create()
{
	return g_object_new(get_type(), 0);
}

/*  Gtk::TextChildAnchor
 */

Gtk::TextChildAnchor::TextChildAnchor(GtkTextChildAnchor *anchor, bool reference)
: G::Object((GObject*)anchor, reference)
{
}

Gtk::TextChildAnchor::TextChildAnchor()
: G::Object((GObject*)TextChildAnchorClass::create())
{
}

Gtk::TextChildAnchor::~TextChildAnchor()
{
}

GtkTextChildAnchorClass*
Gtk::TextChildAnchor::gtk_text_child_anchor_class() const
{
	return get_class<GtkTextChildAnchorClass>();
}

Gtk::TextChildAnchor::operator GtkTextChildAnchor* () const
{
	return this ? gtk_text_child_anchor() : 0;
}

bool
Gtk::TextChildAnchor::get_widgets(std::vector<Widget*>& widgets) const
{
	g_return_val_if_fail(widgets.empty(), false);
	GList *first = gtk_text_child_anchor_get_widgets(gtk_text_child_anchor());

	GList *next = first;
	while (next)
	{
		widgets.push_back(G::Object::wrap<Widget>((GtkWidget*)next->data));
		next = g_list_next(next);
	}

	g_list_free(first);
	return !widgets.empty();
}

bool
Gtk::TextChildAnchor::get_deleted() const
{
	return gtk_text_child_anchor_get_deleted(gtk_text_child_anchor());
}

/*  Gtk::TextChildAnchorClass
 */

void
Gtk::TextChildAnchorClass::init(GtkTextChildAnchorClass *g_class)
{
	G::ObjectClass::init((GObjectClass*)g_class);
}

GType
Gtk::TextChildAnchorClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_TEXT_CHILD_ANCHOR, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::TextChildAnchorClass::create()
{
	return g_object_new(get_type(), 0);
}

/*  Gtk::TextBuffer
 */

Gtk::TextBuffer::TextBuffer(GtkTextBuffer *buffer, bool reference)
: G::Object((GObject*)buffer, reference)
{
}
	
Gtk::TextBuffer::TextBuffer(TextTagTable *table)
: G::Object((GObject*)TextBufferClass::create(*table))
{
}

Gtk::TextBuffer::~TextBuffer()
{
}
	
GtkTextBufferClass*
Gtk::TextBuffer::gtk_text_buffer_class() const
{
	return get_class<GtkTextBufferClass>();
}

Gtk::TextBuffer::operator GtkTextBuffer* () const
{
	return this ? gtk_text_buffer() : 0; 
}
	
int
Gtk::TextBuffer::get_line_count() const
{
	return gtk_text_buffer_get_line_count(gtk_text_buffer());
}

int
Gtk::TextBuffer::get_char_count() const
{
	return gtk_text_buffer_get_char_count(gtk_text_buffer());
}

Gtk::TextTagTable*
Gtk::TextBuffer::get_tag_table() const
{
	return G::Object::wrap<TextTagTable>(gtk_text_buffer_get_tag_table(gtk_text_buffer()));
}

String 
Gtk::TextBuffer::get_text(const TextIter& start, const TextIter& end, bool include_hidden_chars) const
{
	char *tmp_text = gtk_text_buffer_get_text(gtk_text_buffer(), start.gtk_text_iter(), end.gtk_text_iter(), include_hidden_chars);
	String text(tmp_text);
	g_free(tmp_text);
	return text;
}

String
Gtk::TextBuffer::get_slice(const TextIter& start, const TextIter& end, bool include_hidden_chars) const
{
	char *tmp_slice = gtk_text_buffer_get_slice(gtk_text_buffer(), start.gtk_text_iter(), end.gtk_text_iter(), include_hidden_chars);
	String slice(tmp_slice);
	g_free(tmp_slice);
	return slice;
}

Gtk::TextMark*
Gtk::TextBuffer::get_mark(const String& name) const
{
	GtkTextMark *mark = gtk_text_buffer_get_mark(gtk_text_buffer(), name.c_str());
	return mark ? G::Object::wrap<TextMark>(gtk_text_buffer_get_mark(gtk_text_buffer(), name.c_str())) : 0;
}

Gtk::TextMark*
Gtk::TextBuffer::get_insert() const
{
	return G::Object::wrap<TextMark>(gtk_text_buffer_get_insert(gtk_text_buffer()));
}

Gtk::TextMark*
Gtk::TextBuffer::get_selection_bound() const
{
	return G::Object::wrap<TextMark>(gtk_text_buffer_get_selection_bound(gtk_text_buffer()));
}

Gtk::TextIter
Gtk::TextBuffer::get_iter_at_line_offset(int line_number, int char_offset) const
{
	GtkTextIter iter;
	gtk_text_buffer_get_iter_at_line_offset(gtk_text_buffer(), &iter, line_number, char_offset);
	return TextIter(&iter, true);
}

Gtk::TextIter
Gtk::TextBuffer::get_iter_at_line_index(int line_number, int byte_index) const
{
	GtkTextIter iter;
	gtk_text_buffer_get_iter_at_line_index(gtk_text_buffer(), &iter, line_number, byte_index);
	return TextIter(&iter, true);
}

Gtk::TextIter
Gtk::TextBuffer::get_iter_at_offset(int char_offset) const
{
	GtkTextIter iter;
	gtk_text_buffer_get_iter_at_offset(gtk_text_buffer(), &iter, char_offset);
	return TextIter(&iter, true);
}

Gtk::TextIter
Gtk::TextBuffer::get_iter_at_line(int line_number) const
{
	GtkTextIter iter;
	gtk_text_buffer_get_iter_at_line(gtk_text_buffer(), &iter, line_number);
	return TextIter(&iter, true);
}

Gtk::TextIter
Gtk::TextBuffer::get_start_iter() const
{
	GtkTextIter iter;
	gtk_text_buffer_get_start_iter(gtk_text_buffer(), &iter);
	return TextIter(&iter, true);
}

Gtk::TextIter
Gtk::TextBuffer::get_end_iter() const
{
	GtkTextIter iter;
	gtk_text_buffer_get_end_iter(gtk_text_buffer(), &iter);
	return TextIter(&iter, true);
}

void
Gtk::TextBuffer::get_bounds(TextIter& start, TextIter& end) const
{
	gtk_text_buffer_get_bounds(gtk_text_buffer(), start.gtk_text_iter(), end.gtk_text_iter());
}

Gtk::TextIter
Gtk::TextBuffer::get_iter_at_mark(TextMark& mark) const
{
	GtkTextIter iter;
	gtk_text_buffer_get_iter_at_mark(gtk_text_buffer(), &iter, mark.gtk_text_mark());
	return TextIter(&iter, true);
}

Gtk::TextIter
Gtk::TextBuffer::get_iter_at_child_anchor(TextChildAnchor& anchor) const
{
	GtkTextIter iter;
	gtk_text_buffer_get_iter_at_child_anchor(gtk_text_buffer(), &iter, anchor.gtk_text_child_anchor());
	return TextIter(&iter, true);
}

bool
Gtk::TextBuffer::get_modified() const
{
	return gtk_text_buffer_get_modified(gtk_text_buffer());
}

bool
Gtk::TextBuffer::get_selection_bounds(TextIter *start, TextIter *end) const
{
	return gtk_text_buffer_get_selection_bounds(gtk_text_buffer(), *start, *end);
}

void
Gtk::TextBuffer::begin_user_action()
{
	gtk_text_buffer_begin_user_action(gtk_text_buffer());
}

void
Gtk::TextBuffer::end_user_action()
{
	gtk_text_buffer_end_user_action(gtk_text_buffer());
}

void
Gtk::TextBuffer::set_text(const char* text, int length)
{
	gtk_text_buffer_set_text(gtk_text_buffer(), text, length < 0 ? strlen(text) : length);
}

void
Gtk::TextBuffer::set_text(const String& text)
{
	gtk_text_buffer_set_text(gtk_text_buffer(), text.c_str(), text.size());
}

void
Gtk::TextBuffer::insert(TextIter& iter, const String& text)
{
	gtk_text_buffer_insert(gtk_text_buffer(), iter.gtk_text_iter(), text.c_str(), text.size());
}

void
Gtk::TextBuffer::insert(TextIter& iter, const char *text, int length)
{
	gtk_text_buffer_insert(gtk_text_buffer(), iter.gtk_text_iter(), text, length < 0 ? strlen(text) : length);
}

void
Gtk::TextBuffer::insert_at_cursor(const String& text)
{
	gtk_text_buffer_insert_at_cursor(gtk_text_buffer(), text.c_str(), text.size());
}

void
Gtk::TextBuffer::insert_at_cursor(const char *text, int length)
{
	gtk_text_buffer_insert_at_cursor(gtk_text_buffer(), text, length < 0 ? strlen(text) : length);
}

bool
Gtk::TextBuffer::insert_interactive(TextIter& iter, const String& text, bool default_editable)
{
	return gtk_text_buffer_insert_interactive_at_cursor(gtk_text_buffer(), text.c_str(), 
	                                                    text.size(), default_editable);
}

bool
Gtk::TextBuffer::insert_interactive(TextIter& iter, const char *text, int length, bool default_editable)
{
	return gtk_text_buffer_insert_interactive_at_cursor(gtk_text_buffer(), text, 
	                                                    length < 0 ? strlen(text) : length, 
							    default_editable);
}

bool
Gtk::TextBuffer::insert_interactive_at_cursor(const String& text, bool default_editable)
{
	return gtk_text_buffer_insert_interactive_at_cursor(gtk_text_buffer(), text.c_str(), 
	                                                    text.size(), default_editable);
}

bool
Gtk::TextBuffer::insert_interactive_at_cursor(const char *text, int length, bool default_editable)
{
	return gtk_text_buffer_insert_interactive_at_cursor(gtk_text_buffer(), text, length, default_editable);
}

void
Gtk::TextBuffer::insert_range(TextIter& iter, const TextIter& start, const TextIter& end)
{
	gtk_text_buffer_insert_range(gtk_text_buffer(), iter.gtk_text_iter(), start.gtk_text_iter(), 
	                             end.gtk_text_iter());
}

bool
Gtk::TextBuffer::insert_range_interactive(TextIter& iter, const TextIter& start, const TextIter& end, bool default_editable)
{
	return gtk_text_buffer_insert_range_interactive(gtk_text_buffer(), iter.gtk_text_iter(), start.gtk_text_iter(),
	                                                end.gtk_text_iter(), default_editable);
}

void
Gtk::TextBuffer::insert_with_tag(TextIter& iter, const String& text, TextTag& tag)
{
	gtk_text_buffer_insert_with_tags(gtk_text_buffer(), iter.gtk_text_iter(), text.c_str(), 
	                                 text.size(), tag.gtk_text_tag(), 0);
}

void
Gtk::TextBuffer::insert_with_tag(TextIter& iter, const char *text, int length, TextTag& tag)
{
	gtk_text_buffer_insert_with_tags(gtk_text_buffer(), iter.gtk_text_iter(), text, 
	                                 length < 0 ? strlen(text) : length,
	                                 tag.gtk_text_tag(), 0);
}

namespace { // real_insert_with_tags

void real_insert_with_tags(GtkTextBuffer *buffer, GtkTextIter *iter, const char *text, int length,
                           Gtk::TextTag *first_tag, va_list args)
{
	int start_offset = gtk_text_iter_get_offset(iter);
	gtk_text_buffer_insert(buffer, iter, text, length < 0 ? strlen(text) : length);
	if (!first_tag)
		return;

	GtkTextIter start;
	gtk_text_buffer_get_iter_at_offset(buffer, &start, start_offset);

	Gtk::TextTag *tag = first_tag;
	while (tag)
	{
		gtk_text_buffer_apply_tag(buffer, *tag, &start, iter);
		tag = va_arg(args, Gtk::TextTag*);
	}
}

} // real_insert_with_tags

void
Gtk::TextBuffer::insert_with_tags(TextIter& iter, const String& text, TextTag *first_tag, ...)
{
	va_list args;
	va_start(args, first_tag);
	real_insert_with_tags(gtk_text_buffer(), iter.gtk_text_iter(), text.c_str(), text.size(), first_tag, args);
	va_end(args);
}

void
Gtk::TextBuffer::insert_with_tags(TextIter& iter, const char *text, int length, TextTag *first_tag, ...)
{
	va_list args;
	va_start(args, first_tag);
	real_insert_with_tags(gtk_text_buffer(), iter.gtk_text_iter(), text, length, first_tag, args);
	va_end(args);
}

void
Gtk::TextBuffer::insert_with_tag_by_name(TextIter& iter, const String& text, const char *tag_name)
{
	gtk_text_buffer_insert_with_tags_by_name(gtk_text_buffer(), iter.gtk_text_iter(), text.c_str(), text.size(), tag_name, 0);
}

void
Gtk::TextBuffer::insert_with_tag_by_name(TextIter& iter, const char *text, int length, const char *tag_name)
{
	gtk_text_buffer_insert_with_tags_by_name(gtk_text_buffer(), iter.gtk_text_iter(), text, 
	                                         length < 0 ? strlen(text) : length, tag_name, 0);
}

namespace { // real_insert_with_tags_by_name

void real_insert_with_tags_by_name(GtkTextBuffer *buffer, GtkTextIter *iter, const char *text,
                                   int length, const char *first_tag_name, va_list args)
{
	int start_offset = gtk_text_iter_get_offset(iter);
	gtk_text_buffer_insert(buffer, iter, text, length < 0 ? strlen(text) : length);
	if (!first_tag_name)
		return;

	GtkTextIter start;
	gtk_text_buffer_get_iter_at_offset(buffer, &start, start_offset);

	const char *tag_name = first_tag_name;
	while (tag_name)
	{
		GtkTextTag *tag = gtk_text_tag_table_lookup(buffer->tag_table, tag_name);
		if (!tag)
		{
			g_warning("%s: no tag with name '%s'!", __FILE__ ":" G_STRINGIFY (__LINE__), tag_name);
			return;
		}
		gtk_text_buffer_apply_tag(buffer, tag, &start, iter);
		tag_name = va_arg(args, const char*);
	}
}

} // real_insert_with_tags_by_name

void
Gtk::TextBuffer::insert_with_tags_by_name(TextIter& iter, const String& text, const char *first_tag_name, ...)
{
	va_list args;
	va_start(args, first_tag_name);
	real_insert_with_tags_by_name(gtk_text_buffer(), iter.gtk_text_iter(), text.c_str(), text.size(), first_tag_name, args);
	va_end(args);
}

void
Gtk::TextBuffer::insert_with_tags_by_name(TextIter& iter, const char *text, int length, const char *first_tag_name, ...)
{
	va_list args;
	va_start(args, first_tag_name);
	real_insert_with_tags_by_name(gtk_text_buffer(), iter.gtk_text_iter(), text, length, first_tag_name, args);
	va_end(args);
}

void
Gtk::TextBuffer::delete_range(TextIter& start, TextIter& end)
{
	gtk_text_buffer_delete(gtk_text_buffer(), start.gtk_text_iter(), end.gtk_text_iter());
}

bool
Gtk::TextBuffer::delete_range_interactive(TextIter& start, TextIter& end, bool default_editable)
{
	return gtk_text_buffer_delete_interactive(gtk_text_buffer(), start.gtk_text_iter(), 
	                                          end.gtk_text_iter(), default_editable);
}

void
Gtk::TextBuffer::insert_pixbuf(TextIter& iter, Gdk::Pixbuf& pixbuf)
{
	gtk_text_buffer_insert_pixbuf(gtk_text_buffer(), iter.gtk_text_iter(), pixbuf.gdk_pixbuf());
}

void
Gtk::TextBuffer::insert_child_anchor(TextIter& iter, TextChildAnchor& anchor)
{
	gtk_text_buffer_insert_child_anchor(gtk_text_buffer(), iter.gtk_text_iter(), anchor.gtk_text_child_anchor());
}

Gtk::TextChildAnchor*
Gtk::TextBuffer::create_child_anchor(TextIter& iter)
{
	TextChildAnchor *anchor = new Gtk::TextChildAnchor;
	insert_child_anchor(iter, *anchor);
	anchor->unref();
	return anchor;
}

Gtk::TextMark*
Gtk::TextBuffer::create_mark(const String& mark_name, const TextIter& where, bool left_gravity)
{
	GtkTextMark *mark = gtk_text_buffer_create_mark(gtk_text_buffer(), mark_name.c_str(), where.gtk_text_iter(), left_gravity);
	return G::Object::wrap_new<TextMark>(mark);
}

void
Gtk::TextBuffer::move_mark(TextMark& mark, const TextIter& where)
{
	gtk_text_buffer_move_mark(gtk_text_buffer(), mark.gtk_text_mark(), where.gtk_text_iter());
}

void
Gtk::TextBuffer::move_mark_by_name(const String& name, const TextIter& where)
{
	gtk_text_buffer_move_mark_by_name(gtk_text_buffer(), name.c_str(), where.gtk_text_iter());
}

void
Gtk::TextBuffer::delete_mark(TextMark& mark)
{
	gtk_text_buffer_delete_mark(gtk_text_buffer(), mark.gtk_text_mark());
}

void
Gtk::TextBuffer::delete_mark_by_name(const String& name)
{
	gtk_text_buffer_delete_mark_by_name(gtk_text_buffer(), name.c_str());
}

void
Gtk::TextBuffer::place_cursor(const TextIter& where)
{
	gtk_text_buffer_place_cursor(gtk_text_buffer(), where.gtk_text_iter());
}

void
Gtk::TextBuffer::apply_tag(TextTag& tag, const TextIter& start, const TextIter& end)
{
	gtk_text_buffer_apply_tag(gtk_text_buffer(), tag.gtk_text_tag(), start.gtk_text_iter(), end.gtk_text_iter());
}

void
Gtk::TextBuffer::remove_tag(TextTag& tag, const TextIter& start, const TextIter& end)
{
	gtk_text_buffer_remove_tag(gtk_text_buffer(), tag.gtk_text_tag(), start.gtk_text_iter(), end.gtk_text_iter());
}

void
Gtk::TextBuffer::apply_tag_by_name(const String& name, const TextIter& start, const TextIter& end)
{
	gtk_text_buffer_apply_tag_by_name(gtk_text_buffer(), name.c_str(), start.gtk_text_iter(), end.gtk_text_iter());
}

void
Gtk::TextBuffer::remove_tag_by_name(const String& name, const TextIter& start, const TextIter& end)
{
	gtk_text_buffer_remove_tag_by_name(gtk_text_buffer(), name.c_str(), start.gtk_text_iter(), end.gtk_text_iter());
}

void
Gtk::TextBuffer::remove_all_tags(const TextIter& start, const TextIter& end)
{
	gtk_text_buffer_remove_all_tags(gtk_text_buffer(), start.gtk_text_iter(), end.gtk_text_iter());
}

Gtk::TextTag*
Gtk::TextBuffer::create_tag(const String& tag_name, const char *first_property_name, ...)
{
	GtkTextTag *tag = gtk_text_tag_new(tag_name.c_str());
	gtk_text_tag_table_add(gtk_text_buffer()->tag_table, tag);

	if (first_property_name)
	{
		va_list list;
		va_start(list, first_property_name);
		g_object_set_valist(G_OBJECT(tag), first_property_name, list);
		va_end (list);
	}

	g_object_unref(tag);
	return G::Object::wrap<TextTag>(tag);
}

void
Gtk::TextBuffer::set_modified(bool setting)
{
	gtk_text_buffer_set_modified(gtk_text_buffer(), setting);
}

void
Gtk::TextBuffer::add_selection_clipboard(Clipboard& clipboard)
{
	gtk_text_buffer_add_selection_clipboard(gtk_text_buffer(), clipboard.gtk_clipboard());
}

void
Gtk::TextBuffer::remove_selection_clipboard(Clipboard& clipboard)
{
	gtk_text_buffer_remove_selection_clipboard(gtk_text_buffer(), clipboard.gtk_clipboard());
}

void
Gtk::TextBuffer::cut_clipboard(Clipboard& clipboard, bool default_editable)
{
	gtk_text_buffer_cut_clipboard(gtk_text_buffer(), clipboard.gtk_clipboard(), default_editable);
}

void
Gtk::TextBuffer::copy_clipboard(Clipboard& clipboard)
{
	gtk_text_buffer_copy_clipboard(gtk_text_buffer(), clipboard.gtk_clipboard());
}

void
Gtk::TextBuffer::paste_clipboard(Clipboard& clipboard, TextIter *override_location, bool default_editable)
{
	gtk_text_buffer_paste_clipboard(gtk_text_buffer(), clipboard.gtk_clipboard(), *override_location, default_editable);
}

bool
Gtk::TextBuffer::delete_selection(bool interactive, bool default_editable)
{
	return gtk_text_buffer_delete_selection(gtk_text_buffer(), interactive, default_editable);
}

/*  Gtk::TextBufferClass
 */

void
Gtk::TextBufferClass::init(GtkTextBufferClass *g_class)
{
	G::ObjectClass::init((GObjectClass*)g_class);
	g_class->insert_text = &insert_text_proxy;
	g_class->insert_pixbuf = &insert_pixbuf_proxy;
	g_class->insert_child_anchor = &insert_child_anchor_proxy;
	g_class->delete_range = &delete_range_proxy;
	g_class->changed = &changed_proxy;
	g_class->modified_changed = &modified_changed_proxy;
	g_class->mark_set = &mark_set_proxy;
	g_class->mark_deleted = &mark_deleted_proxy;
	g_class->apply_tag = &apply_tag_proxy;
	g_class->remove_tag = &remove_tag_proxy;
	g_class->begin_user_action = &begin_user_action_proxy;
	g_class->end_user_action = &end_user_action_proxy;
}

GType
Gtk::TextBufferClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_TEXT_BUFFER, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::TextBufferClass::create(GtkTextTagTable *table)
{
	return g_object_new(get_type(), "tag_table", table, 0);
}

void
Gtk::TextBufferClass::insert_text_proxy(GtkTextBuffer *buffer, GtkTextIter *pos, const gchar *text, gint length)
{
	TextBuffer *tmp_buffer = G::Object::pointer<TextBuffer>(buffer);
	if (tmp_buffer)
	{
		String tmp_text(text, length);
		TextIter tmp_pos(pos);
		tmp_buffer->on_insert_text(tmp_pos, tmp_text);
	}
	else
	{
		GtkTextBufferClass *g_class = G::TypeInstance::class_peek_parent<GtkTextBufferClass>(GTK_TEXT_BUFFER_GET_CLASS(buffer));
		if (g_class->insert_text)
			g_class->insert_text(buffer, pos, text, length);
	}
}

void
Gtk::TextBufferClass::insert_pixbuf_proxy(GtkTextBuffer *buffer, GtkTextIter *pos, GdkPixbuf *pixbuf)
{
	TextBuffer *tmp_buffer = G::Object::pointer<TextBuffer>(buffer);
	if (tmp_buffer)
	{
		TextIter tmp_pos(pos);
		tmp_buffer->on_insert_pixbuf(tmp_pos, *G::Object::wrap<Gdk::Pixbuf>(pixbuf));
	}
	else
	{
		GtkTextBufferClass *g_class = G::TypeInstance::class_peek_parent<GtkTextBufferClass>(GTK_TEXT_BUFFER_GET_CLASS(buffer));
		if (g_class->insert_pixbuf)
			g_class->insert_pixbuf(buffer, pos, pixbuf);
	}
}

void
Gtk::TextBufferClass::insert_child_anchor_proxy(GtkTextBuffer *buffer, GtkTextIter *pos, GtkTextChildAnchor *anchor)
{
	TextBuffer *tmp_buffer = G::Object::pointer<TextBuffer>(buffer);
	if (tmp_buffer)
	{
		TextIter tmp_pos(pos);
		tmp_buffer->on_insert_child_anchor(tmp_pos, *G::Object::wrap<TextChildAnchor>(anchor));
	}
	else
	{
		GtkTextBufferClass *g_class = G::TypeInstance::class_peek_parent<GtkTextBufferClass>(GTK_TEXT_BUFFER_GET_CLASS(buffer));
		if (g_class->insert_child_anchor)
			g_class->insert_child_anchor(buffer, pos, anchor);
	}
}

void
Gtk::TextBufferClass::delete_range_proxy(GtkTextBuffer *buffer, GtkTextIter *start, GtkTextIter *end)
{
	TextBuffer *tmp_buffer = G::Object::pointer<TextBuffer>(buffer);
	if (tmp_buffer)
	{
		TextIter tmp_start(start);
		TextIter tmp_end(end);
		tmp_buffer->on_delete_range(tmp_start, tmp_end);
	}
	else
	{
		GtkTextBufferClass *g_class = G::TypeInstance::class_peek_parent<GtkTextBufferClass>(GTK_TEXT_BUFFER_GET_CLASS(buffer));
		if (g_class->delete_range)
			g_class->delete_range(buffer, start, end);
	}
}

void
Gtk::TextBufferClass::changed_proxy(GtkTextBuffer *buffer)
{
	TextBuffer *tmp_buffer = G::Object::pointer<TextBuffer>(buffer);
	if (tmp_buffer)
		tmp_buffer->on_changed();
	else
	{
		GtkTextBufferClass *g_class = G::TypeInstance::class_peek_parent<GtkTextBufferClass>(GTK_TEXT_BUFFER_GET_CLASS(buffer));
		if (g_class->changed)
			g_class->changed(buffer);
	}
}

void
Gtk::TextBufferClass::modified_changed_proxy(GtkTextBuffer *buffer)
{
	TextBuffer *tmp_buffer = G::Object::pointer<TextBuffer>(buffer);
	if (tmp_buffer)
		tmp_buffer->on_modified_changed();
	else
	{
		GtkTextBufferClass *g_class = G::TypeInstance::class_peek_parent<GtkTextBufferClass>(GTK_TEXT_BUFFER_GET_CLASS(buffer));
		if (g_class->modified_changed)
			g_class->modified_changed(buffer);
	}
}

void
Gtk::TextBufferClass::mark_set_proxy(GtkTextBuffer *buffer, const GtkTextIter *location, GtkTextMark *mark)
{
	TextBuffer *tmp_buffer = G::Object::pointer<TextBuffer>(buffer);
	if (tmp_buffer)
	{
		TextIter tmp_location(const_cast<GtkTextIter*>(location));
		tmp_buffer->on_mark_set(tmp_location, *G::Object::wrap<TextMark>(mark));
	}
	else
	{
		GtkTextBufferClass *g_class = G::TypeInstance::class_peek_parent<GtkTextBufferClass>(GTK_TEXT_BUFFER_GET_CLASS(buffer));
		if (g_class->mark_set)
			g_class->mark_set(buffer, location, mark);
	}
}

void
Gtk::TextBufferClass::mark_deleted_proxy(GtkTextBuffer *buffer, GtkTextMark *mark)
{
	TextBuffer *tmp_buffer = G::Object::pointer<TextBuffer>(buffer);
	if (tmp_buffer)
		tmp_buffer->on_mark_deleted(*G::Object::wrap<TextMark>(mark));
	else
	{
		GtkTextBufferClass *g_class = G::TypeInstance::class_peek_parent<GtkTextBufferClass>(GTK_TEXT_BUFFER_GET_CLASS(buffer));
		if (g_class->mark_deleted)
			g_class->mark_deleted(buffer, mark);
	}
}

void
Gtk::TextBufferClass::apply_tag_proxy(GtkTextBuffer *buffer, GtkTextTag *tag, const GtkTextIter *start_char, const GtkTextIter *end_char)
{
	TextBuffer *tmp_buffer = G::Object::pointer<TextBuffer>(buffer);
	if (tmp_buffer)
	{
		TextIter tmp_start_char(const_cast<GtkTextIter*>(start_char));
		TextIter tmp_end_char(const_cast<GtkTextIter*>(end_char));
		tmp_buffer->on_apply_tag(*G::Object::wrap<TextTag>(tag), tmp_start_char, tmp_end_char);
	}
	else
	{
		GtkTextBufferClass *g_class = G::TypeInstance::class_peek_parent<GtkTextBufferClass>(GTK_TEXT_BUFFER_GET_CLASS(buffer));
		if (g_class->apply_tag)
			g_class->apply_tag(buffer, tag, start_char, end_char);
	}
}

void
Gtk::TextBufferClass::remove_tag_proxy(GtkTextBuffer *buffer, GtkTextTag *tag, const GtkTextIter *start_char, const GtkTextIter *end_char)
{
	TextBuffer *tmp_buffer = G::Object::pointer<TextBuffer>(buffer);
	if (tmp_buffer)
	{
		TextIter tmp_start_char(const_cast<GtkTextIter*>(start_char));
		TextIter tmp_end_char(const_cast<GtkTextIter*>(end_char));
		tmp_buffer->on_remove_tag(*G::Object::wrap<TextTag>(tag), tmp_start_char, tmp_end_char);
	}
	else
	{
		GtkTextBufferClass *g_class = G::TypeInstance::class_peek_parent<GtkTextBufferClass>(GTK_TEXT_BUFFER_GET_CLASS(buffer));
		if (g_class->remove_tag)
			g_class->remove_tag(buffer, tag, start_char, end_char);
	}
}

void
Gtk::TextBufferClass::begin_user_action_proxy(GtkTextBuffer *buffer)
{
	TextBuffer *tmp_buffer = G::Object::pointer<TextBuffer>(buffer);
	if (tmp_buffer)
		tmp_buffer->on_begin_user_action();
	else
	{
		GtkTextBufferClass *g_class = G::TypeInstance::class_peek_parent<GtkTextBufferClass>(GTK_TEXT_BUFFER_GET_CLASS(buffer));
		if (g_class->begin_user_action)
			g_class->begin_user_action(buffer);
	}
}

void
Gtk::TextBufferClass::end_user_action_proxy(GtkTextBuffer *buffer)
{
	TextBuffer *tmp_buffer = G::Object::pointer<TextBuffer>(buffer);
	if (tmp_buffer)
		tmp_buffer->on_end_user_action();
	else
	{
		GtkTextBufferClass *g_class = G::TypeInstance::class_peek_parent<GtkTextBufferClass>(GTK_TEXT_BUFFER_GET_CLASS(buffer));
		if (g_class->end_user_action)
			g_class->end_user_action(buffer);
	}
}

/*  Gtk::TextBuffer Signal handlers
 */

void
Gtk::TextBuffer::on_insert_text(TextIter& pos, const String& text)
{
	GtkTextBufferClass *g_class = class_peek_parent<GtkTextBufferClass>(gtk_text_buffer_class());
	if (g_class->insert_text)
		g_class->insert_text(gtk_text_buffer(), pos.gtk_text_iter(), text.c_str(), text.size());
}

void
Gtk::TextBuffer::on_insert_pixbuf(TextIter& pos, Gdk::Pixbuf& pixbuf)
{
	GtkTextBufferClass *g_class = class_peek_parent<GtkTextBufferClass>(gtk_text_buffer_class());
	if (g_class->insert_pixbuf)
		g_class->insert_pixbuf(gtk_text_buffer(), pos.gtk_text_iter(), pixbuf.gdk_pixbuf());
}

void
Gtk::TextBuffer::on_insert_child_anchor(TextIter& pos, TextChildAnchor& anchor)
{
	GtkTextBufferClass *g_class = class_peek_parent<GtkTextBufferClass>(gtk_text_buffer_class());
	if (g_class->insert_child_anchor)
		g_class->insert_child_anchor(gtk_text_buffer(), pos.gtk_text_iter(), anchor.gtk_text_child_anchor());
}

void
Gtk::TextBuffer::on_delete_range(TextIter& start, TextIter& end)
{
	GtkTextBufferClass *g_class = class_peek_parent<GtkTextBufferClass>(gtk_text_buffer_class());
	if (g_class->delete_range)
		g_class->delete_range(gtk_text_buffer(), start.gtk_text_iter(), end.gtk_text_iter());
}

void
Gtk::TextBuffer::on_changed()
{
	GtkTextBufferClass *g_class = class_peek_parent<GtkTextBufferClass>(gtk_text_buffer_class());
	if (g_class->changed)
		g_class->changed(gtk_text_buffer());
}

void
Gtk::TextBuffer::on_modified_changed()
{
	GtkTextBufferClass *g_class = class_peek_parent<GtkTextBufferClass>(gtk_text_buffer_class());
	if (g_class->modified_changed)
		g_class->modified_changed(gtk_text_buffer());
}

void
Gtk::TextBuffer::on_mark_set(const TextIter& location, TextMark& mark)
{
	GtkTextBufferClass *g_class = class_peek_parent<GtkTextBufferClass>(gtk_text_buffer_class());
	if (g_class->mark_set)
		g_class->mark_set(gtk_text_buffer(), location.gtk_text_iter(), mark.gtk_text_mark());
}

void
Gtk::TextBuffer::on_mark_deleted(TextMark& mark)
{
	GtkTextBufferClass *g_class = class_peek_parent<GtkTextBufferClass>(gtk_text_buffer_class());
	if (g_class->mark_deleted)
		g_class->mark_deleted(gtk_text_buffer(), mark.gtk_text_mark());
}

void
Gtk::TextBuffer::on_apply_tag(TextTag& tag, const TextIter& start_char, const TextIter& end_char)
{
	GtkTextBufferClass *g_class = class_peek_parent<GtkTextBufferClass>(gtk_text_buffer_class());
	if (g_class->apply_tag)
		g_class->apply_tag(gtk_text_buffer(), tag.gtk_text_tag(), start_char.gtk_text_iter(), end_char.gtk_text_iter());
}

void
Gtk::TextBuffer::on_remove_tag(TextTag& tag, const TextIter& start_char, const TextIter& end_char)
{
	GtkTextBufferClass *g_class = class_peek_parent<GtkTextBufferClass>(gtk_text_buffer_class());
	if (g_class->remove_tag)
		g_class->remove_tag(gtk_text_buffer(), tag.gtk_text_tag(), start_char.gtk_text_iter(), end_char.gtk_text_iter());
}

void
Gtk::TextBuffer::on_begin_user_action()
{
	GtkTextBufferClass *g_class = class_peek_parent<GtkTextBufferClass>(gtk_text_buffer_class());
	if (g_class->begin_user_action)
		g_class->begin_user_action(gtk_text_buffer());
}

void
Gtk::TextBuffer::on_end_user_action()
{
	GtkTextBufferClass *g_class = class_peek_parent<GtkTextBufferClass>(gtk_text_buffer_class());
	if (g_class->end_user_action)
		g_class->end_user_action(gtk_text_buffer());
}

/*  Gtk::TextBuffer Properties
 */

const Gtk::TextBuffer::TagTablePropertyType Gtk::TextBuffer::tag_table_property("tag_table");

/*  Gtk::TextBuffer Signals
 */

const Gtk::TextBuffer::InsertTextSignalType Gtk::TextBuffer::insert_text_signal("insert_text");

const Gtk::TextBuffer::InsertPixbufSignalType Gtk::TextBuffer::insert_pixbuf_signal("insert_pixbuf");

const Gtk::TextBuffer::InsertChildAnchorSignalType Gtk::TextBuffer::insert_child_anchor_signal("insert_child_anchor");

const Gtk::TextBuffer::DeleteRangeSignalType Gtk::TextBuffer::delete_range_signal("delete_range");

const Gtk::TextBuffer::ChangedSignalType Gtk::TextBuffer::changed_signal("changed");

const Gtk::TextBuffer::ModifiedChangedSignalType Gtk::TextBuffer::modified_changed_signal("modified_changed");

const Gtk::TextBuffer::MarkSetSignalType Gtk::TextBuffer::mark_set_signal("mark_set");

const Gtk::TextBuffer::MarkDeletedSignalType Gtk::TextBuffer::mark_deleted_signal("mark_deleted");

const Gtk::TextBuffer::ApplyTagSignalType Gtk::TextBuffer::apply_tag_signal("apply_tag");

const Gtk::TextBuffer::RemoveTagSignalType Gtk::TextBuffer::remove_tag_signal("remove_tag");

const Gtk::TextBuffer::BeginUserActionSignalType Gtk::TextBuffer::begin_user_action_signal("begin_user_action");

const Gtk::TextBuffer::EndUserActionSignalType Gtk::TextBuffer::end_user_action_signal("end_user_action");
