/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/stock.h
//! @brief A GtkStockItem C++ wrapper interface.
//!
//! Provides Stock and StockItem, an interface to access commonly-used menu or toolbar items
//! such as "Open" or "Exit".

#ifndef INTI_GTK_STOCK_H
#define INTI_GTK_STOCK_H

#ifndef INTI_UTF_STRING_H
#include <inti/utf-string.h>
#endif

#ifndef INTI_GDK_TYPES_H
#include <inti/gdk/types.h>
#endif

#ifndef __GTK_STOCK_H__
#include <gtk/gtkstock.h>
#endif

#ifndef _CPP_VECTOR
#include <vector>
#endif

namespace Inti {

namespace Gtk {

class AccelKey;

//! @class StockItem stock.h inti/gtk/stock.h
//! @brief A GtkStockItem C++ wrapper class.
//!
//! Stock items represent commonly-used menu or toolbar items such as "Open" or "Exit".
//! Each stock item is identified by a stock ID; stock IDs are just strings, but macros
//! such as GTK_STOCK_OPEN are provided to avoid typing mistakes in the strings.
//! Applications can register their own stock items in addition to those built-in to GTK+.
//!
//! Each stock ID can be associated with a StockItem, which contains the user-visible
//! label, keyboard accelerator, and translation domain of the menu or toolbar item;
//! and/or with an icon stored in an IconFactory. See IconFactory for more information
//! on stock icons. The connection between a StockItem and stock icons is purely 
//! conventional (by virtue of using the same stock ID); it's possible to register a
//! stock item but no icon, and vice versa.

class StockItem
{
	GtkStockItem item_;

public:
//! @name Constructors
//! @{

	StockItem();
	//!< Construct a new stock item.

//! @}
//! @name Accessors
//! @{

	GtkStockItem* gtk_stock_item() const { return const_cast<GtkStockItem*>(&item_); }
	//!< Get a pointer to the GtkStockItem structure.

	const char* stock_id() const;
	//!< Returns the stock_id for the stock item.

	const char* label() const;
	//!< Returns the user-visible label for the stock item.

	Gdk::ModifierTypeField modifier() const;
	//!< Gets the modifier combination of the keyboard accelerator for the stock item.

	unsigned int keyval() const;
	//!< Gets the key value of the keyboard accelerator for the stock item.

	const char* translation_domain() const;
	//!< Returns the translation domain for the stock item.

	AccelKey accel_key() const;
	//!< Returns an AccelKey object for the keyboard accelerator.

//! @}
};

//! @class Stock stock.h inti/gtk/stock.h
//! @brief A C++ wrapper class for various stock methods.
//!
//! Stock is a convenience class that wraps the miscellaneous GTK+ stock functions.

class Stock
{
public:
//! @name Methods
//! @{

	static void add(const char *stock_id, const char *label);
	//!< Registers a stock item.
	//!< @param stock_id The string identifier for the stock item.
	//!< @param label The user-visible label.
	//!<
	//!< <BR>If the item already exists with the same stock ID as one of the items,
	//!< the old item gets replaced.

	static void add(const char *stock_id, const char *label, Gdk::ModifierTypeField modifier, unsigned int key);
	//!< Registers a stock item.
	//!< @param stock_id The string identifier for the stock item.
	//!< @param label The user-visible label.
	//!< @param modifier The modifier combination of the accelerator.
	//!< @param key The key value of the accelerator
	//!<
	//!< <BR>If the item already exists with the same stock ID as one of the items,
	//!< the old item gets replaced.

	static void add(const char *stock_id, const char *label, Gdk::ModifierTypeField modifier, unsigned int key, const char *domain);
	//!< Registers a stock item.
	//!< @param stock_id The string identifier for the stock item.
	//!< @param label The user-visible label.
	//!< @param modifier The modifier combination of the accelerator.
	//!< @param key The key value of the accelerator
	//!< @param domain The translation domain.
	//!<
	//!< <BR>If the item already exists with the same stock ID as one of the items,
	//!< the old item gets replaced.

	static bool lookup(const char *stock_id, StockItem& item);
	//!< Fills <EM>item</EM> with the registered values for stock_id, returning true if stock_id was known.
	//!< @param stock_id The string identifier for the stock item.
	//!< @param item A stock item to initialize with values.
	//!< @return <EM>true</EM> if item was initialized.

	static std::vector<String> list_ids();
	//!< Retrieves a list of all known stock IDs added to an IconFactory or registered with add().
	//!< @return A vector of String that holds the list of known stock IDs.

//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_STOCK_H

