/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gtk/imagemenuitem.h
//! @brief A GtkImageMenuItem C++ wrapper interface.
//!
//! Provides ImageMenuItem, a MenuItem widget that displays an image at the left side of the menu item text.

#ifndef INTI_GTK_MENU_IMAGE_ITEM_H
#define INTI_GTK_MENU_IMAGE_ITEM_H

#ifndef INTI_GTK_MENU_ITEM_H
#include <inti/gtk/menuitem.h>
#endif

#ifndef __GTK_MENU_IMAGE_ITEM_H__
#include <gtk/gtkimagemenuitem.h>
#endif

namespace Inti {

namespace Gtk {

class ImageMenuItemClass;

//! @class ImageMenuItem imagemenuitem.h inti/gtk/imagemenuitem.h
//! @brief A GtkImageMenuItem C++ wrapper class.
//!
//! ImageMenuItem is a MenuItem widget that displays an image at the left side of the menu item text.

class ImageMenuItem : public MenuItem
{
	friend class G::Object;
	friend class ImageMenuItemClass;

	ImageMenuItem(const ImageMenuItem&);
	ImageMenuItem& operator=(const ImageMenuItem&);
	
protected:
//! @name Constructors
//! @{

	explicit ImageMenuItem(GtkImageMenuItem *menu_item, bool reference = false);
	//!< Construct a new ImageMenuItem from an existing GtkImageMenuItem.
	//!< @param menu_item A pointer to a GtkImageMenuItem.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>menu_item</EM> can be a newly created GtkImageMenuItem or an existing
	//!< GtkImageMenuItem (see G::Object::Object).

//! @}
//  Properties

	typedef G::Property<Widget*, G::Object*> ImagePropertyType;
	typedef G::PropertyProxy<G::Object, ImagePropertyType> ImagePropertyProxy;
	static const ImagePropertyType image_property;

public:
//! @name Constructors
//! @{

	ImageMenuItem();
	//!< Construct a new ImageMenuItem.

	explicit ImageMenuItem(const String& label, bool use_underline = false);
	//!< Constructs a new ImageMenuItem containing a label.
	//!< @param label The text of the menu item.
	//!< @param use_underline Set <EM>true</EM> if label contains a mnemonic character.
	//!<
	//!< <BR>If characters in label are preceded by an underscore, they are underlined.
	//!< If you need a literal underscore character in a label, use '__' (two underscores).
	//!< The first underlined character represents a keyboard accelerator called a mnemonic.
	//!< Pressing Alt and that key activates the menu item.

	ImageMenuItem(Widget& image, const String& label, bool use_underline = false);
	//!< Constructs a new ImageMenuItem containing an image and a label.
	//!< @param image A widget to set as the image for the menu item.
	//!< @param label The text of the menu item.
	//!< @param use_underline Set <EM>true</EM> if label contains a mnemonic character.
	//!<
	//!< <BR>If characters in label are preceded by an underscore, they are underlined.
	//!< If you need a literal underscore character in a label, use '__' (two underscores).
	//!< The first underlined character represents a keyboard accelerator called a mnemonic.
	//!< Pressing Alt and that key activates the menu item.

	ImageMenuItem(Widget& image, const String& label, Menu& submenu, bool use_underline = false);
	//!< Constructs a new ImageMenuItem that displays a submenu.
	//!< @param image A widget to set as the image for the menu item.
	//!< @param label The text of the menu item.
	//!< @param submenu The submenu under this menu item. 
	//!< @param use_underline Set <EM>true</EM> if label contains a mnemonic character.
	//!<
	//!< <BR>If characters in label are preceded by an underscore, they are underlined.
	//!< If you need a literal underscore character in a label, use '__' (two underscores).
	//!< The first underlined character represents a keyboard accelerator called a mnemonic.
	//!< Pressing Alt and that key activates the menu item.

	virtual ~ImageMenuItem();
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	GtkImageMenuItem* gtk_image_menu_item() const { return (GtkImageMenuItem*)instance; }
	//!< Get a pointer to the GtkImageMenuItem structure.

	GtkImageMenuItemClass* gtk_image_menu_item_class() const;
	//!< Get a pointer to the GtkImageMenuItemClass structure.

	operator GtkImageMenuItem* () const;
	//!< Conversion operator; safely converts an ImageMenuItem to a GtkImageMenuItem pointer.

	Widget* get_image() const;
	//!< Gets the widget that is currently set as the image of ImageMenuItem (see set_image()).

//! @}
//! @name Methods
//! @{

	void set_image(Widget& image);
	//!< Sets the image of ImageMenuItem to the given widget.
	//!< @param image A widget to set as the image for the menu item.

//! @}
//! @name Property Proxies
//! @{

	const ImagePropertyProxy prop_image()
	{
		return ImagePropertyProxy(this, &image_property);
	}
	//!< Child widget to appear next to the menu text (Widget* : Read / Write).

//! @}
};

//! @class StockMenuItem imagemenuitem.h inti/gtk/imagemenuitem.h
//! @brief A GtkImageMenuItem C++ wrapper class for stock items.
//!
//! StockMenuItem is an ImageMenuItem that displays the image and text from a stock item.

class StockMenuItem : public ImageMenuItem
{
public:
	StockMenuItem(const char *stock_id, AccelGroup *accel_group = 0);
	//!< Construct a new StockMenuItem containing the image and text from a stock item.
	//!< @param stock_id The name of the stock item.
	//!< @param accel_group The AccelGroup to add the menu items accelerator to, or null.
	//!<
	//!< <BR>Some stock ids have preprocessor macros like GTK_STOCK_OK and GTK_STOCK_APPLY.
	//!< If you want this menu item to have changeable accelerators, then pass in null
	//!< for accel_group and call Gtk::MenuItem::set_accel_path() with an appropriate
	//!< path for the menu item, then use Gtk::Stock::lookup() too look up the standard
	//!< accelerator for the stock item and if one is found, call 
	//!< Gtk::AccelMap::add_entry() to register it.
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_MENU_IMAGE_ITEM_H

