/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  iconfactory.cc - GtkIconSource, GtkIconSet and GtkIconFactory C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "iconfactory.h"
#include "private/iconfactory_p.h"
#include "settings.h"
#include "style.h"
#include "widget.h"
#include "../gdk-pixbuf/pixbuf.h"

using namespace Inti;

/*  Gtk::icon_size_lookup
 */

bool
Gtk::icon_size_lookup(IconSize size, int *width, int *height, Settings *settings)
{
	GtkSettings *tmp_settings = *settings;
	if (!tmp_settings)
	{
		GdkScreen *screen = gdk_screen_get_default();
		tmp_settings = gtk_settings_get_for_screen(screen);
	}
	return gtk_icon_size_lookup_for_settings(tmp_settings, (GtkIconSize)size, width, height);
}

/*  Gtk::icon_size_register
 */

Gtk::IconSize
Gtk::icon_size_register(const char *name, int width, int height)
{
	return (IconSize)gtk_icon_size_register(name, width, height);
}

/*  Gtk::icon_size_register_alias
 */

void
Gtk::icon_size_register_alias(const char *alias, IconSize target)
{
	gtk_icon_size_register_alias(alias,(GtkIconSize)target);
}

/*  Gtk::icon_size_from_name
 */

Gtk::IconSize
Gtk::icon_size_from_name(const char *name)
{
	return (IconSize)gtk_icon_size_from_name(name);
}

/*  Gtk::icon_size_get_name
 */

String
Gtk::icon_size_get_name(IconSize size)
{
	return gtk_icon_size_get_name((GtkIconSize)size);
}

/*  Gtk::IconSource
 */

Gtk::IconSource::IconSource()
: G::Boxed(GTK_TYPE_ICON_SOURCE, gtk_icon_source_new(), false)
{
}

Gtk::IconSource::IconSource(GtkIconSource *source)
: G::Boxed(GTK_TYPE_ICON_SOURCE, source)
{
}

Gtk::IconSource::IconSource(GtkIconSource *source, bool copy)
: G::Boxed(GTK_TYPE_ICON_SOURCE, source, copy)
{
}

Gtk::IconSource::IconSource(const IconSource& src)
: G::Boxed(GTK_TYPE_ICON_SOURCE, src.g_boxed(), true)
{
}

Gtk::IconSource::~IconSource()
{
}

Gtk::IconSource&
Gtk::IconSource::operator=(const IconSource& src)
{
	if (src.boxed_ != boxed_)
		copy(src);
	return *this;
}

Gtk::IconSource::operator GtkIconSource* () const
{
	return this ? gtk_icon_source() : 0;
}

String
Gtk::IconSource::get_filename() const
{
	return gtk_icon_source_get_filename(gtk_icon_source());
}

Gdk::Pixbuf*
Gtk::IconSource::get_pixbuf() const
{
	return G::Object::wrap<Gdk::Pixbuf>(gtk_icon_source_get_pixbuf(gtk_icon_source()));
}

bool
Gtk::IconSource::get_size_wildcarded() const
{
	return gtk_icon_source_get_size_wildcarded(gtk_icon_source());
}

bool
Gtk::IconSource::get_state_wildcarded() const
{
	return gtk_icon_source_get_state_wildcarded(gtk_icon_source());
}

bool
Gtk::IconSource::get_direction_wildcarded() const
{
	return gtk_icon_source_get_direction_wildcarded(gtk_icon_source());
}

Gtk::TextDirection
Gtk::IconSource::get_direction() const
{
	return (TextDirection)gtk_icon_source_get_direction(gtk_icon_source());
}

Gtk::StateType
Gtk::IconSource::get_state() const
{
	return (StateType)gtk_icon_source_get_state(gtk_icon_source());
}

Gtk::IconSize
Gtk::IconSource::get_size() const
{
	return (IconSize)gtk_icon_source_get_size(gtk_icon_source());
}

void
Gtk::IconSource::set_filename(const String& filename)
{
	gtk_icon_source_set_filename(gtk_icon_source(), filename.c_str());
}

void
Gtk::IconSource::set_pixbuf(Gdk::Pixbuf *pixbuf)
{
	gtk_icon_source_set_pixbuf(gtk_icon_source(), *pixbuf);
}

void
Gtk::IconSource::set_direction_wildcarded(bool setting)
{
	gtk_icon_source_set_direction_wildcarded(gtk_icon_source(), setting);
}

void
Gtk::IconSource::set_state_wildcarded(bool setting)
{
	gtk_icon_source_set_state_wildcarded(gtk_icon_source(), setting);
}

void
Gtk::IconSource::set_size_wildcarded(bool setting)
{
	gtk_icon_source_set_size_wildcarded(gtk_icon_source(), setting);
}

void
Gtk::IconSource::set_direction(TextDirection direction)
{
	gtk_icon_source_set_direction(gtk_icon_source(), (GtkTextDirection)direction);
}

void
Gtk::IconSource::set_state(StateType state)
{
	gtk_icon_source_set_state(gtk_icon_source(), (GtkStateType)state);
}

void
Gtk::IconSource::set_size(IconSize size)
{
	gtk_icon_source_set_size(gtk_icon_source(), (GtkIconSize)size);
}

/*  Gtk::IconSet
 */

Gtk::IconSet::IconSet()
: G::Boxed(GTK_TYPE_ICON_SET, gtk_icon_set_new(), false)
{
}

Gtk::IconSet::IconSet(Gdk::Pixbuf& pixbuf)
: G::Boxed(GTK_TYPE_ICON_SET, gtk_icon_set_new_from_pixbuf(pixbuf.gdk_pixbuf()), false)
{
}

Gtk::IconSet::IconSet(GtkIconSet *icon_set, bool copy)
: G::Boxed(GTK_TYPE_ICON_SET, icon_set, copy)
{
}

Gtk::IconSet::IconSet(const IconSet& src)
: G::Boxed(GTK_TYPE_ICON_SET, src.g_boxed(), true)
{
}

Gtk::IconSet::~IconSet()
{
}

Gtk::IconSet&
Gtk::IconSet::operator=(const IconSet& src)
{
	if (src.boxed_ == boxed_)
		return *this;

	copy(src);
	return *this;
}

Gtk::IconSet::operator GtkIconSet* () const
{
	return this ? gtk_icon_set() : 0;
}

bool
Gtk::IconSet::get_sizes(std::vector<IconSize>& sizes) const
{
	g_return_val_if_fail(sizes.empty(), false);
	GtkIconSize *tmp_sizes = 0;
	int n_sizes = 0;

	gtk_icon_set_get_sizes(gtk_icon_set(), &tmp_sizes, &n_sizes);

	int i = 0;
	while (i < n_sizes)
	{
		sizes.push_back((IconSize)tmp_sizes[i]);
		++i;
	}

	g_free(tmp_sizes);
	return !sizes.empty();
}

Pointer<Gdk::Pixbuf>
Gtk::IconSet::render_icon(Style& style, StateType state, IconSize size, Widget *widget, 
                          const char *detail, TextDirection direction)
{
	GdkPixbuf *pixbuf = gtk_icon_set_render_icon(gtk_icon_set(), style.gtk_style(), (GtkTextDirection)direction,
	                                             (GtkStateType)state, (GtkIconSize)size, *widget, detail);
	return G::Object::wrap<Gdk::Pixbuf>(pixbuf, true);
}

void
Gtk::IconSet::add_source(const IconSource& source)
{
	gtk_icon_set_add_source(gtk_icon_set(), source.gtk_icon_source());
}

/*  Gtk::IconFactory
 */

Gtk::IconFactory::IconFactory(GtkIconFactory *icon_factory, bool reference)
: G::Object((GObject*)icon_factory, reference)
{
}

Gtk::IconFactory::IconFactory() 
: G::Object((GObject*)IconFactoryClass::create())
{
}

Gtk::IconFactory::~IconFactory() 
{
}

GtkIconFactoryClass* 
Gtk::IconFactory::gtk_icon_factory_class() const
{
	return get_class<GtkIconFactoryClass>(); 
}
	
Gtk::IconFactory::operator GtkIconFactory* () const 
{
	return this ? gtk_icon_factory() : 0;
}

void 
Gtk::IconFactory::add(const char *stock_id, IconSet& icon_set)
{
	gtk_icon_factory_add(gtk_icon_factory(), stock_id, icon_set.gtk_icon_set());
}

Pointer<Gtk::IconSet>
Gtk::IconFactory::lookup(const char *stock_id)
{
	return G::Boxed::wrap<IconSet>(GTK_TYPE_ICON_SET, gtk_icon_factory_lookup(gtk_icon_factory(), stock_id), true);
}

void 
Gtk::IconFactory::add_default()
{
	gtk_icon_factory_add_default(gtk_icon_factory());
}

void 
Gtk::IconFactory::remove_default()
{
	gtk_icon_factory_remove_default(gtk_icon_factory());
}

Pointer<Gtk::IconSet>
Gtk::IconFactory::lookup_default(const char *stock_id)
{
	return G::Boxed::wrap<IconSet>(GTK_TYPE_ICON_SET, gtk_icon_factory_lookup_default(stock_id), true);
}

/*  Gtk::IconFactoryClass
 */

void
Gtk::IconFactoryClass::init(GtkIconFactoryClass *g_class)
{
	G::ObjectClass::init((GObjectClass*)g_class);
}

GType
Gtk::IconFactoryClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_ICON_FACTORY, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::IconFactoryClass::create()
{
	return g_object_new(get_type(), 0);
}

