/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *  Copyright (C) 2000 Red Hat, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/gtk/cellrenderer.h
//! @brief A GtkCellRenderer C++ wrapper interface.

#ifndef INTI_GTK_CELL_RENDERER_H
#define INTI_GTK_CELL_RENDERER_H

#ifndef INTI_GTK_OBJECT_H
#include <inti/gtk/object.h>
#endif

#ifndef INTI_GTK_ENUMS_H
#include <inti/gtk/enums.h>
#endif

#ifndef INTI_PANGO_ATTRIBUTES_H
#include <inti/pango/attributes.h>
#endif

#ifndef __GTK_CELL_RENDERER_H__
#include <gtk/gtkcellrenderer.h>
#endif

#ifndef __GTK_CELL_RENDERER_PIXBUF_H__
#include <gtk/gtkcellrendererpixbuf.h>
#endif

#ifndef __GTK_CELL_RENDERER_TEXT_H__
#include <gtk/gtkcellrenderertext.h>
#endif

#ifndef __GTK_CELL_RENDERER_TOGGLE_H__
#include <gtk/gtkcellrenderertoggle.h>
#endif

namespace Inti {

namespace Gdk {
class Color;
class Event;
class Pixbuf;
class Rectangle;
class Window;
}

namespace Gtk {

class CellEditable;
class CellRendererClass;
class CellRendererPixbufClass;
class CellRendererTextClass;
class CellRendererToggleClass;
class Widget;

/*  CellRendererState
 */

//! @enum CellRendererState
//! Specifies how the CellRenderer should render a cell.

enum CellRendererState
{
	CELL_RENDERER_SELECTED = GTK_CELL_RENDERER_SELECTED, //!< The cell can be selected.
	CELL_RENDERER_PRELIT = GTK_CELL_RENDERER_PRELIT, //!< The cell can be prelit.
	CELL_RENDERER_INSENSITIVE = GTK_CELL_RENDERER_INSENSITIVE, //!< The cell is insensitive.
	CELL_RENDERER_SORTED = GTK_CELL_RENDERER_SORTED, //!< The cell is in the sort column/row.
	CELL_RENDERER_FOCUSED = GTK_CELL_RENDERER_FOCUSED //!< The cell can be focused.
};

//! CellRendererStateField holds one or more values from the CellRendererState enum OR'd together.

typedef unsigned int CellRendererStateField;

//! @enum CellRendererMode
//! Specifies the editable mode of the CellRenderer.

enum CellRendererMode
{
	INERT = GTK_CELL_RENDERER_MODE_INERT, //!< The cell is neither activatable nor editable.
	ACTIVATABLE = GTK_CELL_RENDERER_MODE_ACTIVATABLE, //!< The cell can be activated.
	EDITABLE = GTK_CELL_RENDERER_MODE_EDITABLE //!< The cell can be edited.
};

//! @class CellRenderer cellrenderer.h inti/gtk/cellrenderer.h
//! @brief A GtkCellRenderer C++ wrapper class.
//!
//! The purpose of the cell renderers is to provide extensibility to the widget
//! and to allow multiple ways of rendering the same type of data. For example,
//! consider how to render a bool variable. Should you render it as a string of
//! "true" or "false", "On" or "Off", or should you render it as a checkbox?
//!
//! The TreeView widget uses columns and cell renderers to display a model. Cell
//! renderers are used to draw the data in the tree model in a certain way. There
//! are three cell renderers that come with Inti. They are the CellRendererText,
//! CellRendererPixbuf and CellRendererToggle. It is relatively easy to write a
//! custom renderer.

class CellRenderer : public Object
{
	friend class G::Object;
	friend class CellRendererClass;

	CellRenderer(const CellRenderer&);
	CellRenderer& operator=(const CellRenderer&);
	
protected:
//! @name Constructors
//! @{

	explicit CellRenderer(GtkCellRenderer *cell, bool reference = false);
	//!< Construct a new CellRenderer from an existing GtkCellRenderer.
	//!< @param cell A pointer to a GtkCellRenderer.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>cell</EM> can be a newly created GtkCellRenderer or an existing
	//!< GtkCellRenderer. (see G::Object::Object).

//! @}
//  Override these do_ methods when you want to change the default behaviour of the GtkCellRenderer.

	virtual void do_get_size(GtkWidget *widget, GdkRectangle *cell_area, int *x_offset, int *y_offset,
	                         int *width, int *height);

	virtual void do_render(GdkWindow *window, GtkWidget *widget, GdkRectangle *background_area,
	                       GdkRectangle *cell_area, GdkRectangle *expose_area, GtkCellRendererState flags);

	virtual bool do_activate(GdkEvent *event, GtkWidget *widget, const char *path, GdkRectangle *background_area,
	                         GdkRectangle *cell_area, GtkCellRendererState flags);

	virtual GtkCellEditable* do_start_editing(GdkEvent *event, GtkWidget *widget, const char *path,
	                                          GdkRectangle *background_area, GdkRectangle *cell_area,
						  GtkCellRendererState flags);
//  Properties

	typedef G::Property<CellRendererMode, int> ModePropertyType;
	typedef G::PropertyProxy<G::Object, ModePropertyType> ModePropertyProxy;
	static const ModePropertyType mode_property;

	typedef G::Property<bool> VisiblePropertyType;
	typedef G::PropertyProxy<G::Object, VisiblePropertyType> VisiblePropertyProxy;
	static const VisiblePropertyType visible_property;

	typedef G::Property<float> XAlignPropertyType;
	typedef G::PropertyProxy<G::Object, XAlignPropertyType> XAlignPropertyProxy;
	static const XAlignPropertyType xalign_property;

	typedef G::Property<float> YAlignPropertyType;
	typedef G::PropertyProxy<G::Object, YAlignPropertyType> YAlignPropertyProxy;
	static const YAlignPropertyType yalign_property;

	typedef G::Property<unsigned int> XPadPropertyType;
	typedef G::PropertyProxy<G::Object, XPadPropertyType> XPadPropertyProxy;
	static const XPadPropertyType xpad_property;

	typedef G::Property<unsigned int> YPadPropertyType;
	typedef G::PropertyProxy<G::Object, YPadPropertyType> YPadPropertyProxy;
	static const YPadPropertyType ypad_property;

	typedef G::Property<int> WidthPropertyType;
	typedef G::PropertyProxy<G::Object, WidthPropertyType> WidthPropertyProxy;
	static const WidthPropertyType width_property;

	typedef G::Property<int> HeightPropertyType;
	typedef G::PropertyProxy<G::Object, HeightPropertyType> HeightPropertyProxy;
	static const HeightPropertyType height_property;

	typedef G::Property<bool> IsExpanderPropertyType;
	typedef G::PropertyProxy<G::Object, IsExpanderPropertyType> IsExpanderPropertyProxy;
	static const IsExpanderPropertyType is_expander_property;

	typedef G::Property<bool> IsExpandedPropertyType;
	typedef G::PropertyProxy<G::Object, IsExpandedPropertyType> IsExpandedPropertyProxy;
	static const IsExpandedPropertyType is_expanded_property;

	typedef G::WritableProperty<String> CellBackgroundPropertyType;
	typedef G::PropertyProxy<G::Object, CellBackgroundPropertyType> CellBackgroundPropertyProxy;
	static const CellBackgroundPropertyType cell_background_property;

	typedef G::Property<Pointer<Gdk::Color> > CellBackgroundGdkPropertyType;
	typedef G::PropertyProxy<G::Object, CellBackgroundGdkPropertyType> CellBackgroundGdkPropertyProxy;
	static const CellBackgroundGdkPropertyType cell_background_gdk_property;

	typedef G::Property<bool> CellBackgroundSetPropertyType;
	typedef G::PropertyProxy<G::Object, CellBackgroundSetPropertyType> CellBackgroundSetPropertyProxy;
	static const CellBackgroundSetPropertyType cell_background_set_property;

public:
//! @name Constructors
//! @{

	virtual ~CellRenderer();
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	GtkCellRenderer* gtk_cell_renderer() const { return (GtkCellRenderer*)instance; }
	//!< Get a pointer to the GtkCellRenderer structure.

	GtkCellRendererClass* gtk_cell_renderer_class() const;
	//!< Get a pointer to the GtkCellRendererClass structure.

	operator GtkCellRenderer* () const;
	//!< Conversion operator; safely converts a CellRenderer to a GtkCellRenderer pointer.

	void get_fixed_size(int *width, int *height);
	//!< Fills in width and height with the appropriate size of cell.
	//!< @param width Location to fill in with the fixed width of the widget, or null.
	//!< @param height Location to fill in with the fixed height of the widget, or null.

	void get_size(const Widget& widget, const Gdk::Rectangle *cell_area,
	              int *x_offset, int *y_offset, int *width, int *height) const;
	//!< Obtains the width and height needed to render the cell and sets the x and y offets (if set).
	//!< @param widget The widget the renderer is rendering to.
	//!< @param cell_area The area a cell will be allocated, or null
	//!< @param x_offset Location to return x offset of cell relative to cell_area, or null.
	//!< @param y_offset Location to return y offset of cell relative to cell_area, or null.
	//!< @param width Location to return width needed to render a cell, or null.
	//!< @param height Location to return height needed to render a cell, or null.
	//!<
	//!< <BR>Used by view widgets to determine the appropriate size for the cell_area passed
	//!< to render(). If cell_area is not null, fills in the x and y offsets (if set) of the
	//!< cell relative to this location. Otherwise x_offset and y_offset are inorgred. Please
	//!< note that the values set in width and height, as well as those in x_offset and 
	//!< y_offset are inclusive of the xpad and ypad properties.

//! @}
//! @name Methods
//! @{

	void render(Gdk::Window& window, Widget& widget, const Gdk::Rectangle *background_area,
	            const Gdk::Rectangle *cell_area, const Gdk::Rectangle *expose_area, 
	            CellRendererStateField flags);
	//!< Invokes the virtual render function of the GtkCellRenderer.
	//!< @param window A GdkDrawable to draw to.
	//!< @param widget The widget owning window.
	//!< @param background_area Entire cell area (including tree expanders and maybe padding on the sides).
	//!< @param cell_area Area normally rendered by a cell renderer.
	//!< @param expose_area Area that actually needs updating.
	//!< @param flags Flags that affect the rendering.
	//!<
	//!< <BR>The three passed-in rectangles are areas of window. Most renderers will draw
	//!< within cell_area; the xalign, yalign, xpad, and ypad properties of the CellRenderer
	//!< should be honored with respect to cell_area. The background_area includes the blank
	//!< space around the cell, and also the area containing the tree expander; so the
	//!< background_area rectangles for all cells tile to cover the entire window. The
	//!< expose_area is a clip rectangle.

	bool activate(const Gdk::Event *event, Widget *widget, const String& path,
	              const Gdk::Rectangle *background_area, const Gdk::Rectangle *cell_area,
	              CellRendererStateField flags);
	//!< Passes an activate event to the cell renderer for possible processing.
	//!< @param event A Gdk::Event.
	//!< @param widget Widget that received the event.
	//!< @param path Widget-dependent string representation of the event location;
	//!<             e.g. for TreeView, a string representation of TreePath.
	//!< @param background_area Background area as passed to render.
	//!< @param cell_area Cell area as passed to render().
	//!< @param flags Render flags
	//!< @return <EM>true</EM> if the event was handled.
	//!<
	//!< <BR>Some cell renderers may use events; for example, CellRendererToggle
	//!< toggles when it gets a mouse click.

	Pointer<CellEditable> start_editing(const Gdk::Event *event, Widget *widget, const String& path,
	                                    const Gdk::Rectangle *background_area, const Gdk::Rectangle *cell_area,
	                                    CellRendererStateField  flags);
	//!< Passes an activate event to the cell renderer for possible processing.
	//!< @param event A Gdk::Event.
	//!< @param widget Widget that received the event.
	//!< @param path Widget-dependent string representation of the event location;
	//!<             e.g. for TreeView, a string representation of TreePath.
	//!< @param background_area Background area as passed to render.
	//!< @param cell_area Cell area as passed to render().
	//!< @param flags Render flags
	//!< @return A new CellEditable or null.

	void set_fixed_size(int width, int height);
	//!< Sets the renderer size to be explicit, independent of the properties set.
	//!< @param width The width of the cell renderer, or -1.
	//!< @param height The height of the cell renderer, or -1.

//! @}
//! @name Property Proxies
//! @{

	const ModePropertyProxy prop_mode()
	{
		return ModePropertyProxy(this, &mode_property);
	}
	//!< Editable mode of the CellRenderer (CellRendererMode : Read / Write).

	const VisiblePropertyProxy prop_visible()
	{
		return VisiblePropertyProxy(this, &visible_property);
	}
	//!< Display the cell (bool : Read / Write).

	const XAlignPropertyProxy prop_xalign()
	{
		return XAlignPropertyProxy(this, &xalign_property);
	}
	//!< The x-align (float : Read / Write).

	const YAlignPropertyProxy prop_yalign()
	{
		return YAlignPropertyProxy(this, &yalign_property);
	}
	//!< The y-align (float : Read / Write).

	const XPadPropertyProxy prop_xpad()
	{
		return XPadPropertyProxy(this, &xpad_property);
	}
	//!< The xpad (unsigned int : Read / Write).

	const YPadPropertyProxy prop_ypad()
	{
		return YPadPropertyProxy(this, &ypad_property);
	}
	//!< The ypad (unsigned int : Read / Write).

	const WidthPropertyProxy prop_width()
	{
		return WidthPropertyProxy(this, &width_property);
	}
	//!< The fixed width (int : Read / Write).

	const HeightPropertyProxy prop_height()
	{
		return HeightPropertyProxy(this, &height_property);
	}
	//!< The fixed height (int : Read / Write).

	const IsExpanderPropertyProxy prop_is_expander()
	{
		return IsExpanderPropertyProxy(this, &is_expander_property);
	}
	//!< The row has children (bool : Read / Write).

	const IsExpandedPropertyProxy prop_is_expanded()
	{
		return IsExpandedPropertyProxy(this, &is_expanded_property);
	}
	//!< The row is an expander row, and is expanded (bool : Read / Write).
	
	const CellBackgroundPropertyProxy prop_cell_background()
	{
		return CellBackgroundPropertyProxy(this, &cell_background_property);
	}
	//!< The cell background color as a string (String : Write).

	const CellBackgroundGdkPropertyProxy prop_cell_background_gdk()
	{
		return CellBackgroundGdkPropertyProxy(this, &cell_background_gdk_property);
	}
	//!< The cell background color as a Gdk::Color (Pointer<Gdk::Color> : Read / Write).

	const CellBackgroundSetPropertyProxy prop_cell_background_set()
	{
		return CellBackgroundSetPropertyProxy(this, &cell_background_set_property);
	}
	//!< Whether this tag affects the cell background color (bool : Read / Write).

};

//! @class CellRendererPixbuf cellrenderer.h inti/gtk/cellrenderer.h
//! @brief A GtkCellRendererPixbuf C++ wrapper class.

class CellRendererPixbuf : public CellRenderer
{
	friend class G::Object;
	friend class CellRendererClassPixbuf;

	CellRendererPixbuf(const CellRendererPixbuf&);
	CellRendererPixbuf& operator=(const CellRendererPixbuf&);

protected:
//! @name Constructors
//! @{

	explicit CellRendererPixbuf(GtkCellRendererPixbuf *cell, bool reference = false);
	//!< Construct a new CellRendererPixbuf from an existing GtkCellRendererPixbuf.
	//!< @param cell A pointer to a GtkCellRendererPixbuf.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>cell</EM> can be a newly created GtkCellRendererPixbuf or an existing
	//!< GtkCellRendererPixbuf. (see G::Object::Object).

//! @}
//  Properties

	typedef G::Property<Gdk::Pixbuf*, G::Object*> PixbufPropertyType;
	typedef G::PropertyProxy<G::Object, PixbufPropertyType> PixbufPropertyProxy;
	static const PixbufPropertyType pixbuf_property;

	typedef G::Property<Gdk::Pixbuf*, G::Object*> PixbufExpanderOpenPropertyType;
	typedef G::PropertyProxy<G::Object, PixbufExpanderOpenPropertyType> PixbufExpanderOpenPropertyProxy;
	static const PixbufExpanderOpenPropertyType pixbuf_expander_open_property;

	typedef G::Property<Gdk::Pixbuf*, G::Object*> PixbufExpanderClosedPropertyType;
	typedef G::PropertyProxy<G::Object, PixbufExpanderClosedPropertyType> PixbufExpanderClosedPropertyProxy;
	static const PixbufExpanderClosedPropertyType pixbuf_expander_closed_property;

	typedef G::Property<String> StockIdPropertyType;
	typedef G::PropertyProxy<G::Object, StockIdPropertyType> StockIdPropertyProxy;
	static const StockIdPropertyType stock_id_property;

	typedef G::Property<IconSize, int> StockSizePropertyType;
	typedef G::PropertyProxy<G::Object, StockSizePropertyType> StockSizePropertyProxy;
	static const StockSizePropertyType stock_size_property;

	typedef G::Property<String> StockDetailPropertyType;
	typedef G::PropertyProxy<G::Object, StockDetailPropertyType> StockDetailPropertyProxy;
	static const StockDetailPropertyType stock_detail_property;

public:
//! @name Constructors
//! @{

	CellRendererPixbuf();
	//!< Construct a new CellRendererPixbuf.
	//!< Adjust rendering parameters using object properties. Also, with TreeViewColumn, you 
	//!< can bind a property to a value in a TreeModel. For example, you can bind the "pixbuf"
	//!< property on the cell renderer to a pixbuf value in the model, thus rendering a 
	//!< different image in each row of the TreeView.

	virtual ~CellRendererPixbuf();
	//!< Destructor.
	
//! @}
//! @name Accessors
//! @{

	GtkCellRendererPixbuf* gtk_cell_renderer_pixbuf() const { return (GtkCellRendererPixbuf*)instance; }
	//!< Get a pointer to the GtkCellRendererPixbuf structure.

	GtkCellRendererPixbufClass* gtk_cell_renderer_pixbuf_class() const;
	//!< Get a pointer to the GtkCellRendererPixbufClass structure.

	operator GtkCellRendererPixbuf* () const;
	//!< Conversion operator; safely converts a CellRendererPixbuf to a GtkCellRendererPixbuf pointer.

//! @}
//! @name Property Proxies
//! @{

	const PixbufPropertyProxy prop_pixbuf()
	{
		return PixbufPropertyProxy(this, &pixbuf_property);
	}
	//!< The pixbuf to render (Gdk::Pixbuf* : Read / Write).

	const PixbufExpanderOpenPropertyProxy prop_pixbuf_expander_open()
	{
		return PixbufExpanderOpenPropertyProxy(this, &pixbuf_expander_open_property);
	}
	//!< Pixbuf for open expander (Gdk::Pixbuf* : Read / Write).

	const PixbufExpanderClosedPropertyProxy prop_pixbuf_expander_closed()
	{
		return PixbufExpanderClosedPropertyProxy(this, &pixbuf_expander_closed_property);
	}
	//!< Pixbuf for closed expander (Gdk::Pixbuf* : Read / Write).

	const StockIdPropertyProxy prop_stock_id()
	{
		return StockIdPropertyProxy(this, &stock_id_property);
	}
	//!< The stock ID of the stock icon to render (String : Read / Write).

	const StockSizePropertyProxy prop_stock_size()
	{
		return StockSizePropertyProxy(this, &stock_size_property);
	}
	//!< The size of the rendered icon (IconSize : Read / Write).

	const StockDetailPropertyProxy prop_stock_detail()
	{
		return StockDetailPropertyProxy(this, &stock_detail_property);
	}
	//!< The render detail to pass to the theme engine (String : Read / Write).

//!@}
};

//! @class CellRendererText cellrenderer.h inti/gtk/cellrenderer.h
//! @brief A GtkCellRendererText C++ wrapper class.

class CellRendererText : public CellRenderer
{
	friend class G::Object;
	friend class CellRendererTextClass;

	CellRendererText(const CellRendererText&);
	CellRendererText& operator=(const CellRendererText&);

protected:
//! @name Constructors
//! @{

	explicit CellRendererText(GtkCellRendererText *cell, bool reference = false);
	//!< Construct a new CellRendererText from an existing GtkCellRendererText.
	//!< @param cell A pointer to a GtkCellRendererText.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>cell</EM> can be a newly created GtkCellRendererText or an existing
	//!< GtkCellRendererText. (see G::Object::Object).

//! @}
//  Properties

	typedef G::Property<String> TextPropertyType;
	typedef G::PropertyProxy<G::Object, TextPropertyType> TextPropertyProxy;
	static const TextPropertyType text_property;

	typedef G::WritableProperty<String> MarkupPropertyType;
	typedef G::PropertyProxy<G::Object, MarkupPropertyType> MarkupPropertyProxy;
	static const MarkupPropertyType markup_property;

	typedef G::Property<Pointer<Pango::AttrList> > AttributePropertyType;
	typedef G::PropertyProxy<G::Object, AttributePropertyType> AttributePropertyProxy;
	static const AttributePropertyType attribute_property;

	typedef G::WritableProperty<String> BackgroundPropertyType;
	typedef G::PropertyProxy<G::Object, BackgroundPropertyType> BackgroundPropertyProxy;
	static const BackgroundPropertyType background_property;

	typedef G::Property<Pointer<Gdk::Color> > BackgroundGdkPropertyType;
	typedef G::PropertyProxy<G::Object, BackgroundGdkPropertyType> BackgroundGdkPropertyProxy;
	static const BackgroundGdkPropertyType background_gdk_property;

	typedef G::WritableProperty<String> ForegroundPropertyType;
	typedef G::PropertyProxy<G::Object, ForegroundPropertyType> ForegroundPropertyProxy;
	static const ForegroundPropertyType foreground_property;

	typedef G::Property<Pointer<Gdk::Color> > ForegroundGdkPropertyType;
	typedef G::PropertyProxy<G::Object, ForegroundGdkPropertyType> ForegroundGdkPropertyProxy;
	static const ForegroundGdkPropertyType foreground_gdk_property;

	typedef G::Property<bool> EditablePropertyType;
	typedef G::PropertyProxy<G::Object, EditablePropertyType> EditablePropertyProxy;
	static const EditablePropertyType editable_property;

	typedef G::Property<String> FontPropertyType;
	typedef G::PropertyProxy<G::Object, FontPropertyType> FontPropertyProxy;
	static const FontPropertyType font_property;

	typedef G::Property<Pointer<Pango::FontDescription> > FontDescPropertyType;
	typedef G::PropertyProxy<G::Object, FontDescPropertyType> FontDescPropertyProxy;
	static const FontDescPropertyType font_desc_property;

	typedef G::Property<String> FamilyPropertyType;
	typedef G::PropertyProxy<G::Object, FamilyPropertyType> FamilyPropertyProxy;
	static const FamilyPropertyType family_property;

	typedef G::Property<Pango::Style, int> StylePropertyType;
	typedef G::PropertyProxy<G::Object, StylePropertyType> StylePropertyProxy;
	static const StylePropertyType style_property;

	typedef G::Property<Pango::Variant, int> VariantPropertyType;
	typedef G::PropertyProxy<G::Object, VariantPropertyType> VariantPropertyProxy;
	static const VariantPropertyType variant_property;

	typedef G::Property<Pango::Weight, int> WeightPropertyType;
	typedef G::PropertyProxy<G::Object, WeightPropertyType> WeightPropertyProxy;
	static const WeightPropertyType weight_property;

	typedef G::Property<Pango::Stretch, int> StretchPropertyType;
	typedef G::PropertyProxy<G::Object, StretchPropertyType> StretchPropertyProxy;
	static const StretchPropertyType stretch_property;

	typedef G::Property<int> SizePropertyType;
	typedef G::PropertyProxy<G::Object, SizePropertyType> SizePropertyProxy;
	static const SizePropertyType size_property;

	typedef G::Property<double> SizePointsPropertyType;
	typedef G::PropertyProxy<G::Object, SizePointsPropertyType> SizePointsPropertyProxy;
	static const SizePointsPropertyType size_points_property;

	typedef G::Property<double> ScalePropertyType;
	typedef G::PropertyProxy<G::Object, ScalePropertyType> ScalePropertyProxy;
	static const ScalePropertyType scale_property;

	typedef G::Property<int> RisePropertyType;
	typedef G::PropertyProxy<G::Object, RisePropertyType> RisePropertyProxy;
	static const RisePropertyType rise_property;

	typedef G::Property<bool> StrikethroughPropertyType;
	typedef G::PropertyProxy<G::Object, StrikethroughPropertyType> StrikethroughPropertyProxy;
	static const StrikethroughPropertyType strikethrough_property;

	typedef G::Property<Pango::Underline, int> UnderlinePropertyType;
	typedef G::PropertyProxy<G::Object, UnderlinePropertyType> UnderlinePropertyProxy;
	static const UnderlinePropertyType underline_property;

	typedef G::Property<bool> BackgroundSetPropertyType;
	typedef G::PropertyProxy<G::Object, BackgroundSetPropertyType> BackgroundSetPropertyProxy;
	static const BackgroundSetPropertyType background_set_property;

	typedef G::Property<bool> ForegroundSetPropertyType;
	typedef G::PropertyProxy<G::Object, ForegroundSetPropertyType> ForegroundSetPropertyProxy;
	static const ForegroundSetPropertyType foreground_set_property;

	typedef G::Property<bool> EditableSetPropertyType;
	typedef G::PropertyProxy<G::Object, EditableSetPropertyType> EditableSetPropertyProxy;
	static const EditableSetPropertyType editable_set_property;

	typedef G::Property<bool> FamilySetPropertyType;
	typedef G::PropertyProxy<G::Object, FamilySetPropertyType> FamilySetPropertyProxy;
	static const FamilySetPropertyType family_set_property;

	typedef G::Property<bool> StyleSetPropertyType;
	typedef G::PropertyProxy<G::Object, StyleSetPropertyType> StyleSetPropertyProxy;
	static const StyleSetPropertyType style_set_property;

	typedef G::Property<bool> VariantSetPropertyType;
	typedef G::PropertyProxy<G::Object, VariantSetPropertyType> VariantSetPropertyProxy;
	static const VariantSetPropertyType variant_set_property;

	typedef G::Property<bool> WeightSetPropertyType;
	typedef G::PropertyProxy<G::Object, WeightSetPropertyType> WeightSetPropertyProxy;
	static const WeightSetPropertyType weight_set_property;

	typedef G::Property<bool> StretchSetPropertyType;
	typedef G::PropertyProxy<G::Object, StretchSetPropertyType> StretchSetPropertyProxy;
	static const StretchSetPropertyType stretch_set_property;

	typedef G::Property<bool> SizeSetPropertyType;
	typedef G::PropertyProxy<G::Object, SizeSetPropertyType> SizeSetPropertyProxy;
	static const SizeSetPropertyType size_set_property;

	typedef G::Property<bool> ScaleSetPropertyType;
	typedef G::PropertyProxy<G::Object, ScaleSetPropertyType> ScaleSetPropertyProxy;
	static const ScaleSetPropertyType scale_set_property;

	typedef G::Property<bool> RiseSetPropertyType;
	typedef G::PropertyProxy<G::Object, RiseSetPropertyType> RiseSetPropertyProxy;
	static const RiseSetPropertyType rise_set_property;

	typedef G::Property<bool> StrikethroughSetPropertyType;
	typedef G::PropertyProxy<G::Object, StrikethroughSetPropertyType> StrikethroughSetPropertyProxy;
	static const StrikethroughSetPropertyType strikethrough_set_property;

	typedef G::Property<bool> UnderlineSetPropertyType;
	typedef G::PropertyProxy<G::Object, UnderlineSetPropertyType> UnderlineSetPropertyProxy;
	static const UnderlineSetPropertyType underline_set_property;

//! @name Signal Handlers
//! @{

	virtual void on_edited(const String& path, const String& new_text);
	//!< Called when the user has finished editing the cell text.

//! @}
//  Signals

	typedef G::Signal2<void, const char*, const char*> EditedSignalType;
	typedef G::SignalProxy<TypeInstance, EditedSignalType> EditedSignalProxy;
	static const EditedSignalType edited_signal;

public:
//! @name Constructors
//! @{

	CellRendererText();
	//!< Construct a new CellRendererText. 
	//!< You can adjust how text is drawn using object properties. Also, with TreeViewColumn,
	//!< you can bind a property to a value in a TreeModel. For example, you can bind the 
	//!< "text" property on the cell renderer to a string value in the model, thus rendering
	//!< a different string in each row of the TreeView.

	virtual ~CellRendererText();
	//!< Destructor.
	
//! @}
//! @name Accessors
//! @{

	GtkCellRendererText* gtk_cell_renderer_text() const { return (GtkCellRendererText*)instance; }
	//!< Get a pointer to the GtkCellRendererText structure.

	GtkCellRendererTextClass* gtk_cell_renderer_text_class() const;
	//!< Get a pointer to the GtkCellRendererTextClass structure.

	operator GtkCellRendererText* () const;
	//!< Conversion operator; safely converts a CellRendererText to a GtkCellRendererText pointer.

//! @}
//! @name Methods
//! @{

 	void set_fixed_height_from_font(int number_of_rows);
	//!< Sets the height of a renderer to explicitly be determined by the "font" and
	//!< the "y_pad" property set on it.
	//!< @param number_of_rows Number of rows of text each cell renderer is allocated, or -1
	//!<
	//!< <BR>Further changes in these properties do not affect the height, so they must
	//!< be accompanied by a subsequent call to this function. Using this function is
	//!< unflexible, and should really only be used if calculating the size of a cell
	//!< is too slow (ie, a massive number of cells displayed). If number_of_rows is -1,
	//!< then the fixed height is unset, and the height is determined by the properties
	//!< again.

//! @}
//! @name Property Proxies
//! @{

	const TextPropertyProxy prop_text()
	{
		return TextPropertyProxy(this, &text_property);
	}
	//!< Text to render (String : Read / Write).

	const MarkupPropertyProxy prop_markup()
	{
		return MarkupPropertyProxy(this, &markup_property);
	}
	//!< Marked up text to render (String : Write).

	const AttributePropertyProxy prop_attribute()
	{
		return AttributePropertyProxy(this, &attribute_property);
	}
	//!< A list of style attributes to apply to the text of the renderer
	//!< (Pointer<Pango::AttrList> : Read / Write).

	const BackgroundPropertyProxy prop_background()
	{
		return BackgroundPropertyProxy(this, &background_property);
	}
	//!< Background color as a string (String : Write).

	const BackgroundGdkPropertyProxy prop_background_gdk()
	{
		return BackgroundGdkPropertyProxy(this, &background_gdk_property);
	}
	//!< Background color as a Gdk::Color (Pointer<Gdk::Color> : Read / Write).

	const ForegroundPropertyProxy prop_foreground()
	{
		return ForegroundPropertyProxy(this, &foreground_property);
	}
	//!< Foreground color as a string (String : Write).

	const ForegroundGdkPropertyProxy prop_foreground_gdk()
	{
		return ForegroundGdkPropertyProxy(this, &foreground_gdk_property);
	}
	//!< Foreground color as a Gdk::Color (Pointer<Gdk::Color> : Read / Write).

	const EditablePropertyProxy prop_editable()
	{
		return EditablePropertyProxy(this, &editable_property);
	}
	//!< Whether the text can be modified by the user (bool : Read / Write).

	const FontPropertyProxy prop_font()
	{
		return FontPropertyProxy(this, &font_property);
	}
	//!< Font description as a string (String : Read / Write).

	const FontDescPropertyProxy prop_font_desc()
	{
		return FontDescPropertyProxy(this, &font_desc_property);
	}
	//!< Font description as a Pango::FontDescription (Pointer<Pango::FontDescription> : Read / Write).

	const FamilyPropertyProxy prop_family()
	{
		return FamilyPropertyProxy(this, &family_property);
	}
	//!< Name of the font family, e.g. Sans, Helvetica, Times, Monospace (String : Read / Write).

	const StylePropertyProxy prop_style()
	{
		return StylePropertyProxy(this, &style_property);
	}
	//!< Font style (Pango::Style : Read / Write).

	const VariantPropertyProxy prop_variant()
	{
		return VariantPropertyProxy(this, &variant_property);
	}
	//!< Font variant (Pango::Variant : Read / Write).

	const WeightPropertyProxy prop_weight()
	{
		return WeightPropertyProxy(this, &weight_property);
	}
	//!< Font weight (int : Read / Write).

	const StretchPropertyProxy prop_stretch()
	{
		return StretchPropertyProxy(this, &stretch_property);
	}
	//!< Font stretch (Pango::Stretch : Read / Write).

	const SizePropertyProxy prop_size()
	{
		return SizePropertyProxy(this, &size_property);
	}
	//!< Font size (int : Read / Write).

	const SizePointsPropertyProxy prop_size_points()
	{
		return SizePointsPropertyProxy(this, &size_points_property);
	}
	//!< Font size in points (double : Read / Write).

	const ScalePropertyProxy prop_scale()
	{
		return ScalePropertyProxy(this, &scale_property);
	}
	//!< Font scaling factor (double : Read / Write).

	const RisePropertyProxy prop_rise()
	{
		return RisePropertyProxy(this, &rise_property);
	}
	//!< Offset of text above the baseline (below the baseline if rise is negative) (int : Read / Write).

	const StrikethroughPropertyProxy prop_strikethrough()
	{
		return StrikethroughPropertyProxy(this, &strikethrough_property);
	}
	//!< Whether to strike through the text (bool : Read / Write).

	const UnderlinePropertyProxy prop_underline()
	{
		return UnderlinePropertyProxy(this, &underline_property);
	}
	//!< Style of underline for this text (Pango::Underline : Read / Write).

	const BackgroundSetPropertyProxy prop_background_set()
	{
		return BackgroundSetPropertyProxy(this, &background_set_property);
	}
	//!< Whether this tag affects the background color (bool : Read / Write).

	const ForegroundSetPropertyProxy prop_foreground_set()
	{
		return ForegroundSetPropertyProxy(this, &foreground_set_property);
	}
	//!< Whether this tag affects the foreground color.

	const EditableSetPropertyProxy prop_editable_set()
	{
		return EditableSetPropertyProxy(this, &editable_set_property);
	}
	//!< Whether this tag affects text editability (bool : Read / Write).

	const FamilySetPropertyProxy prop_family_set()
	{
		return FamilySetPropertyProxy(this, &family_set_property);
	}
	//!< Whether this tag affects the font family (bool : Read / Write).

	const StyleSetPropertyProxy prop_style_set()
	{
		return StyleSetPropertyProxy(this, &style_set_property);
	}
	//!< Whether this tag affects the font style (bool : Read / Write).

	const VariantSetPropertyProxy prop_variant_set()
	{
		return VariantSetPropertyProxy(this, &variant_set_property);
	}
	//!< Whether this tag affects the font variant (bool : Read / Write).

	const WeightSetPropertyProxy prop_weight_set()
	{
		return WeightSetPropertyProxy(this, &weight_set_property);
	}
	//!< Whether this tag affects the font weight (bool : Read / Write).

	const StretchSetPropertyProxy prop_stretch_set()
	{
		return StretchSetPropertyProxy(this, &stretch_set_property);
	}
	//!< Whether this tag affects the font stretch (bool : Read / Write).

	const SizeSetPropertyProxy prop_size_set()
	{
		return SizeSetPropertyProxy(this, &size_set_property);
	}
	//!< Whether this tag affects the font size (bool : Read / Write).

	const ScaleSetPropertyProxy prop_scale_set()
	{
		return ScaleSetPropertyProxy(this, &scale_set_property);
	}
	//!< Whether this tag scales the font size by a factor (bool : Read / Write).

	const RiseSetPropertyProxy prop_rise_set()
	{
		return RiseSetPropertyProxy(this, &rise_set_property);
	}
	//!< Whether this tag scales the font size by a factor (bool : Read / Write).

	const StrikethroughSetPropertyProxy prop_strikethrough_set()
	{
		return StrikethroughSetPropertyProxy(this, &strikethrough_set_property);
	}
	//!< Whether this tag affects strikethrough (bool : Read / Write).

	const UnderlineSetPropertyProxy prop_underline_set()
	{
		return UnderlineSetPropertyProxy(this, &underline_set_property);
	}
	//!< Whether this tag affects underlining (bool : Read / Write).

//! @}
//! @name Signal Proxies
//! @{

	const EditedSignalProxy sig_edited()
	{
		return EditedSignalProxy(this, &edited_signal);
	}
	//!< Connect to the editing_done_signal; emitted when the user has finished editing the cell text.

//!, @}
};

//! @class CellRendererToggle cellrenderer.h inti/gtk/cellrenderer.h
//! @brief A GtkCellRendererToggle C++ wrapper class.

class CellRendererToggle : public CellRenderer
{
	friend class G::Object;
	friend class CellRendererToggleClass;

	CellRendererToggle(const CellRendererToggle&);
	CellRendererToggle& operator=(const CellRendererToggle&);

protected:
//! @name Constructors
//! @{

	explicit CellRendererToggle(GtkCellRendererToggle *cell, bool reference = false);
	//!< Construct a new CellRendererToggle from an existing GtkCellRendererToggle.
	//!< @param cell A pointer to a GtkCellRendererToggle.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>cell</EM> can be a newly created GtkCellRendererToggle or an existing
	//!< GtkCellRendererToggle. (see G::Object::Object).

//! @}
//  Properties

	typedef G::Property<bool> ActivePropertyType;
	typedef G::PropertyProxy<G::Object, ActivePropertyType> ActivePropertyProxy;
	static const ActivePropertyType active_property;

	typedef G::Property<bool> ActivatablePropertyType;
	typedef G::PropertyProxy<G::Object, ActivatablePropertyType> ActivatablePropertyProxy;
	static const ActivatablePropertyType activatable_property;

	typedef G::Property<bool> RadioPropertyType;
	typedef G::PropertyProxy<G::Object, RadioPropertyType> RadioPropertyProxy;
	static const RadioPropertyType radio_property;

	typedef G::Property<bool> InconsistentPropertyType;
	typedef G::PropertyProxy<G::Object, InconsistentPropertyType> InconsistentPropertyProxy;
	static const InconsistentPropertyType inconsistent_property;

//! @name Signal Handlers
//! @{

	virtual void on_toggled(const String& path);
	//!< Called when the state of the CellRenderer is changed.

//! @}
//  Signals

	typedef G::Signal1<void, const char*> ToggledSignalType;
	typedef G::SignalProxy<TypeInstance, ToggledSignalType> ToggledSignalProxy;
	static const ToggledSignalType toggled_signal;

public:
//! @name Constructors
//! @{

	CellRendererToggle();
	//!< Construct a new CellRendererToggle. 
	//!< Adjust rendering parameters using object properties. Also, with TreeViewColumn, 
	//!< you can bind a property to a value in a TreeModel. For example, you can bind 
	//!< the "active" property on the cell renderer to a bool value in the model, thus
	//!< causing the check button to reflect the state of the model.

	virtual ~CellRendererToggle();
	//!< Destructor.

//! @}
//! @name Accessors
//! @{

	GtkCellRendererToggle* gtk_cell_renderer_toggle() const { return (GtkCellRendererToggle*)instance; }
	//!< Get a pointer to the GtkCellRendererToggle structure.

	GtkCellRendererToggleClass* gtk_cell_renderer_toggle_class() const;
	//!< Get a pointer to the GtkCellRendererToggleClass structure.

	operator GtkCellRendererToggle* () const;
	//!< Conversion operator; safely converts a CellRendererToggle to a GtkCellRendererToggle pointer.

	bool get_active() const;
	//!< Returns true if the cell renderer is active (see set_active()).

	bool get_radio() const;
	//!< Returns true if we're rendering radio toggles rather than checkboxes.

//! @}
//! @name Methods
//! @{

	void set_active(bool setting);
	//!< Activates or deactivates a cell renderer.
	//!< @param setting  The value to set.
	
	void set_radio(bool radio);
	//!< If radio is <EM>true</EM>, the cell renderer renders a radio toggle; 
	//!< if <EM>false</EM> it renders a check toggle.
	//!< @param radio <EM>true</EM> to make the toggle look like a radio button.
	//!<
	//!< <BR>This can be set globally for the cell renderer, or changed just 
	//!< before rendering each cell in the model (for TreeView, you set up a
	//!< per-row setting using TreeViewColumn to associate model columns with
	//!< cell renderer properties).

//! @}
//! @name Property Proxies
//! @{

	const ActivePropertyProxy prop_active()
	{
		return ActivePropertyProxy(this, &active_property);
	}
	//!< The toggle state of the button (bool : Read / Write).

	const ActivatablePropertyProxy prop_activatable()
	{
		return ActivatablePropertyProxy(this, &activatable_property);
	}
	//!< The toggle button can be activated (bool : Read / Write).

	const RadioPropertyProxy prop_radio()
	{
		return RadioPropertyProxy(this, &radio_property);
	}
	//!< Draw the toggle button as a radio button (bool : Read / Write).

	const InconsistentPropertyProxy prop_inconsistent()
	{
		return InconsistentPropertyProxy(this, &inconsistent_property);
	}
	//!< The inconsistent state of the button (bool : Read / Write).

//! @}
//! @name Signal Proxies
//! @{

	const ToggledSignalProxy sig_toggled()
	{
		return ToggledSignalProxy(this, &toggled_signal);
	}
	//!< Connect to the toggled_signal; emitted when the state of the CellRenderer is changed.
	
//! @}
};

} // namespace Gtk

} // namespace Inti

#endif // INTI_GTK_CELL_RENDERER_H

