/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/atk/selection.h
//! @brief An AtkSelection C++ wrapper interface.
//!
//! Provides Selection, an interface implemented by container objects whose Atk::Object children can be selected.

#ifndef INTI_ATK_SELECTION_H
#define INTI_ATK_SELECTION_H

#ifndef INTI_G_TYPE_H
#include <inti/glib/type.h>
#endif

#ifndef __ATK_SELECTION_H__
#include <atk/atkselection.h>
#endif

namespace Inti {

namespace Atk {

class Object;
class SelectionIface;

//! @class Selection selection.h inti/atk/selection.h
//! @brief An AtkSelection C++ wrapper class.
//!
//! Selection should be implemented by UI components with children which are exposed by
//! Atk::Object::ref_accessible_child() and Atk::Object::get_n_accessible_children(), if
//! the use of the parent UI component ordinarily involves selection of one or more of 
//! the objects corresponding to those Atk::Object children - for example, selectable lists.
//!
//! Note that other types of "selection" (for instance text selection) are accomplished
//! a other ATK interfaces - Atk::Selection is limited to the selection/deselection of
//! children.

class Selection : public virtual G::TypeInterface
{
	friend class SelectionIface;

protected:
//! @name Constructors
//! @{

	virtual ~Selection() = 0;
	//!< Destructor.
	
//! @}
//  Override these do_ methods when you want to change the default behaviour of the AtkSelection.
	
	virtual bool do_add_selection(int i);

	virtual bool do_clear_selection();

	virtual AtkObject* do_ref_selection(int i);

	virtual int do_get_selection_count();

	virtual bool do_is_child_selected(int i);

	virtual bool do_remove_selection(int i);

	virtual bool do_select_all_selection();

//! @name Signal Handlers
//! @{

	virtual void on_selection_changed();
	//!< Called when an object's selection changes.

//! @}
//  Signals

	typedef G::Signal0<void> SelectionChangedSignalType;
	typedef G::SignalProxy<TypeInstance, SelectionChangedSignalType> SelectionChangedSignalProxy;
	static const SelectionChangedSignalType selection_changed_signal;

public:
//! @name Accessors
//! @{

	AtkSelection* atk_selection() const { return (AtkSelection*)instance; }
	//!< Get a pointer to the AtkSelection structure.

	AtkSelectionIface* atk_selection_iface() const;
	//!< Get a pointer to the AtkSelectionIface structure.

	operator AtkSelection* () const;
	//!< Conversion operator; safely converts a Selection to an AtkSelection pointer.

	int get_selection_count() const;
	//!< Gets the number of accessible children currently selected. 
	//!< @return An integer representing the number of items selected, or 0
	//!< if selection does not implement this interface.
	//!<
	//!< <BR><B>Note:</B> callers should not rely on null or on a zero value
	//!< for an indication of whether the Selection interface is implemented,
	//!< they should use is_a(ATK_TYPE_SELECTION).

	bool is_child_selected(int i) const;
	//!< Determines if the current child of this object is selected.
	//!< @param i An integer specifying the child index.
	//!< @return <EM>true</EM> if the child is selected, or <EM>false</EM> if selection
	//!< does not implement this interface.
	//!<
	//!< <BR><B>Note:</B> callers should not rely on null or on a zero value
	//!< for an indication of whether the Selection interface is implemented,
	//!< they should use is_a(ATK_TYPE_SELECTION).

//! @}
//! @name Methods
//! @{

	bool add_selection(int i);
	//!< Adds the specified accessible child of the object to the object's selection.
	//!< @param i An integer specifying the child index.
	//!< @return <EM>true</EM> if success, <EM>false</EM> otherwise.

	bool clear_selection();
	//!< Clears the selection in the object so that no children in the object are selected.
	//!< @return <EM>true</EM> if successful, <EM>false</EM> otherwise.

	Object* ref_selection(int i);
	//!< Gets a pointer to the accessible object representing the
	//!< specified selected child of the object. 
	//!< @param i An integer specifying the index in the selection set (that is, the ith
	//!<          selection as opposed to the ith child).
	//!< @return An Atk::bject representing the selected accessible , or null if selection
	//!< does not implement this interface.
	//!<
	//!< <BR><B>Note:</B> callers should not rely on null or on a zero value for indication
	//!< of whether the Selection interface is implemented, they should use is_a(ATK_TYPE_SELECTION).

	bool remove_selection(int i);
	//!< Removes the specified child of the object from the object's selection.
	//!< @param i An integer specifying the index in the selection set (that is, the ith 
	//!<          selection as opposed to the ith child).
	//!< @return <EM>true</EM> if success, <EM>false</EM> otherwise.

	bool select_all_selection();
	//!< Causes every child of the object to be selected if the object supports multiple selections.
	//!< @return <EM>true</EM> if success, <EM>false</EM> otherwise.

//! @}
//! @name Signal Proxies
//! @{

	const SelectionChangedSignalProxy sig_selection_changed()
	{
		return SelectionChangedSignalProxy(this, &selection_changed_signal);
	}
	//!< Connect to the selection_changed_signal; emitted when an object's selection changes.

//! @}
};

} // namespace Atk

} // namespace Inti

#endif // INTI_ATK_SELECTION_H

