/**
 * @file libgalago-gtk/galago-gtk-account-label.c Account label widget
 *
 * @Copyright (C) 2004-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gtk-account-label.h>
#include <libgalago-gtk/galago-gdk-pixbuf.h>
#include <libgalago-gtk/galago-gtk-private.h>
#include <gtk/gtk.h>

struct _GalagoGtkAccountLabelPriv
{
	GalagoAccount *account;

	char *username;

	gulong account_destroy_id;
	gulong account_display_name_id;
};

static void galago_gtk_account_label_class_init(GalagoGtkAccountLabelClass *klass);
static void galago_gtk_account_label_init(GalagoGtkAccountLabel *label);
static void galago_gtk_account_label_finalize(GObject *obj);
static void galago_gtk_account_label_destroy(GtkObject *obj);

static GtkLabelClass *parent_class = NULL;

GType
galago_gtk_account_label_get_type(void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GalagoGtkAccountLabelClass),
			NULL,
			NULL,
			(GClassInitFunc)galago_gtk_account_label_class_init,
			NULL,
			NULL,
			sizeof(GalagoGtkAccountLabel),
			0,
			(GInstanceInitFunc)galago_gtk_account_label_init
		};

		type = g_type_register_static(GTK_TYPE_LABEL,
									  "GalagoGtkAccountLabel", &info, 0);
	}

	return type;
}

static void
galago_gtk_account_label_class_init(GalagoGtkAccountLabelClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;

	parent_class = g_type_class_peek_parent(klass);

	gobject_class = G_OBJECT_CLASS(klass);
	object_class  = GTK_OBJECT_CLASS(klass);

	gobject_class->finalize = galago_gtk_account_label_finalize;

	object_class->destroy = galago_gtk_account_label_destroy;
}

static void
galago_gtk_account_label_init(GalagoGtkAccountLabel *label)
{
	label->priv = g_new0(GalagoGtkAccountLabelPriv, 1);
}

static void
galago_gtk_account_label_finalize(GObject *obj)
{
	GalagoGtkAccountLabel *label;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_ACCOUNT_LABEL(obj));

	label = GALAGO_GTK_ACCOUNT_LABEL(obj);

	g_free(label->priv);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
galago_gtk_account_label_destroy(GtkObject *obj)
{
	GalagoGtkAccountLabel *label;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_ACCOUNT_LABEL(obj));

	label = GALAGO_GTK_ACCOUNT_LABEL(obj);

	GALAGO_GTK_DISCONNECT_HANDLER(label->priv->account,
								  label->priv->account_destroy_id);
	GALAGO_GTK_DISCONNECT_HANDLER(label->priv->account,
								  label->priv->account_display_name_id);

	label->priv->account = NULL;

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(obj);
}

static void
account_destroy_cb(GalagoAccount *account, GalagoGtkAccountLabel *label)
{
	galago_gtk_account_label_set_account(label, NULL);
}

static void
account_display_name_changed_cb(GalagoAccount *account,
								GalagoGtkAccountLabel *label)
{
	gtk_label_set_text(GTK_LABEL(label),
					   galago_account_get_display_name(account));
}

GtkWidget *
galago_gtk_account_label_new(void)
{
	GalagoGtkAccountLabel *label;

	label = g_object_new(GALAGO_GTK_TYPE_ACCOUNT_LABEL, NULL);

	return GTK_WIDGET(label);
}

void
galago_gtk_account_label_set_account(GalagoGtkAccountLabel *label,
									 GalagoAccount *account)
{
	g_return_if_fail(label != NULL);
	g_return_if_fail(GALAGO_GTK_IS_ACCOUNT_LABEL(label));

	if (label->priv->account == account)
		return;

	GALAGO_GTK_DISCONNECT_HANDLER(label->priv->account,
								  label->priv->account_destroy_id);
	GALAGO_GTK_DISCONNECT_HANDLER(label->priv->account,
								  label->priv->account_display_name_id);

	label->priv->account = account;

	if (account != NULL)
	{
		label->priv->account_destroy_id =
			g_signal_connect(G_OBJECT(account), "destroy",
							 G_CALLBACK(account_destroy_cb), label);

		label->priv->account_display_name_id =
			g_signal_connect(G_OBJECT(account), "display-name-changed",
							 G_CALLBACK(account_display_name_changed_cb),
							 label);

		gtk_label_set_text(GTK_LABEL(label),
						   galago_account_get_display_name(account));
	}
	else
	{
		gtk_label_set_text(GTK_LABEL(label), NULL);
	}
}

GalagoAccount *
galago_gtk_account_label_get_account(const GalagoGtkAccountLabel *label)
{
	g_return_val_if_fail(label != NULL, NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_ACCOUNT_LABEL(label), NULL);

	return label->priv->account;
}
