/* IIWU Synth  A soundfont synthesizer
 *
 * Copyright (C)  2001 Peter Hanappe
 * Author: Peter Hanappe, peter@hanappe.com
 *
 * This file is part of the IIWU program. 
 * IIWU is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 *
 */
#include "iiwu_midi.h"


long debug_msec = 0;


/* ALSA */
#if ALSA_SUPPORT
iiwu_midi_driver_t* new_iiwu_alsa_midi_driver(iiwu_midi_handler_t* handler);
int delete_iiwu_alsa_midi_driver(iiwu_midi_driver_t* p);
int iiwu_alsa_midi_driver_join(iiwu_midi_driver_t* p);
int iiwu_alsa_midi_driver_status(iiwu_midi_driver_t* p);
#endif

/* OSS */
#if OSS_SUPPORT
iiwu_midi_driver_t* new_iiwu_oss_midi_driver(iiwu_midi_handler_t* handler);
int delete_iiwu_oss_midi_driver(iiwu_midi_driver_t* p);
int iiwu_oss_midi_driver_join(iiwu_midi_driver_t* p);
int iiwu_oss_midi_driver_status(iiwu_midi_driver_t* p);
#endif

/* Windows MIDI service */
#if WINMIDI_SUPPORT
iiwu_midi_driver_t* new_iiwu_win_midi_driver(iiwu_midi_handler_t* handler);
int delete_iiwu_win_midi_driver(iiwu_midi_driver_t* p);
int iiwu_win_midi_driver_join(iiwu_midi_driver_t* p);
int iiwu_win_midi_driver_status(iiwu_midi_driver_t* p);
#endif

struct iiwu_mdriver_definition_t iiwu_midi_drivers[] = {
#if OSS_SUPPORT
  { "oss", 
    "Open Sound System", 
    new_iiwu_oss_midi_driver, 
    delete_iiwu_oss_midi_driver, 
    iiwu_oss_midi_driver_join,
    iiwu_oss_midi_driver_status },
#endif
#if ALSA_SUPPORT
  { "alsa", 
    "Advanced Linux Sound Architecture, v0.9",
    new_iiwu_alsa_midi_driver, 
    delete_iiwu_alsa_midi_driver, 
    iiwu_alsa_midi_driver_join,
    iiwu_alsa_midi_driver_status },
#endif
#if WINMIDI_SUPPORT
  { "winmidi", 
    "Windows MIDI System",
    new_iiwu_win_midi_driver, 
    delete_iiwu_win_midi_driver, 
    iiwu_win_midi_driver_join,
    iiwu_win_midi_driver_status },
#endif
  { NULL, NULL, NULL, NULL, NULL, NULL }
};

#if OSS_SUPPORT
char* iiwu_default_midi_driver = "oss";
#elif ALSA_SUPPORT
char* iiwu_default_midi_driver = "alsa";
#elif WINMIDI_SUPPORT
char* iiwu_default_midi_driver = "winmidi";
#else
#error No MIDI driver specified
#endif


/* all outgoing user messages are stored in a global text buffer */
#define MIDI_MESSAGE_LENGTH 1024
char midi_message_buffer[MIDI_MESSAGE_LENGTH];


/***************************************************************
 *
 *                      MIDIFILE
 */
/*
 * new_iiwu_midi_file
 */
iiwu_midi_file* new_iiwu_midi_file(char* filename)
{
  iiwu_midi_file* mf;
  mf = IIWU_NEW(iiwu_midi_file);
  if (mf == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;
  }
  IIWU_MEMSET(mf, 0, sizeof(iiwu_midi_file));
  mf->c = -1;
  mf->running_status = -1;
  mf->fp = IIWU_FOPEN(filename, "rb");
  if (mf->fp == NULL) {
    IIWU_LOG(ERR, "Couldn't open the MIDI file");
    IIWU_FREE(mf);
    return NULL;    
  }
  if (iiwu_midi_file_read_mthd(mf) != IIWU_OK) {
    IIWU_FREE(mf);
    return NULL;
  }
  return mf;
}

/*
 * delete_iiwu_midi_file
 */
void delete_iiwu_midi_file(iiwu_midi_file* mf)
{
  if (mf == NULL) {
    return;
  }
  if (mf->fp != NULL) {
    IIWU_FCLOSE(mf->fp);
  }
  IIWU_FREE(mf);
  return;
}

/*
 * iiwu_midi_file_getc
 */
int iiwu_midi_file_getc(iiwu_midi_file* mf)
{
  unsigned char c;
  int n;
  if (mf->c >= 0) {
    c = mf->c;
    mf->c = -1;
  } else {
    n = IIWU_FREAD(&c, 1, 1, mf->fp);
    mf->trackpos++;
  }
  return (int) c;
}

/*
 * iiwu_midi_file_push
 */
int iiwu_midi_file_push(iiwu_midi_file* mf, int c)
{
  mf->c = c;
  return IIWU_OK;
}

/*
 * iiwu_midi_file_read
 */
int iiwu_midi_file_read(iiwu_midi_file* mf, void* buf, int len)
{
  int num = IIWU_FREAD(buf, 1, len, mf->fp);
  mf->trackpos += num;
#if DEBUG
  if (num != len) {
    IIWU_LOG(DBG, "Coulnd't read the requested number of bytes");
  }
#endif
  return (num != len)? IIWU_FAILED : IIWU_OK;
}

/*
 * iiwu_midi_file_skip
 */
int iiwu_midi_file_skip(iiwu_midi_file* mf, int skip)
{
  int err = IIWU_FSEEK(mf->fp, skip, SEEK_CUR);
  if (err) {
    IIWU_LOG(ERR, "Failed to seek position in file");
    return IIWU_FAILED;    
  }
  return IIWU_OK;
}

/*
 * iiwu_midi_file_read_mthd
 */
int iiwu_midi_file_read_mthd(iiwu_midi_file* mf)
{
  char mthd[15];
  if (iiwu_midi_file_read(mf, mthd, 14) != IIWU_OK) {
    return IIWU_FAILED;
  }
  if ((IIWU_STRNCMP(mthd, "MThd", 4) != 0) || (mthd[7] != 6) || (mthd[9] > 2)) {
    IIWU_LOG(ERR, "Doesn't look like a MIDI file: invalid MThd header");
    return IIWU_FAILED;
  }
  mf->type = mthd[9];
  mf->ntracks = (unsigned) mthd[11];
  mf->ntracks += (unsigned int) (mthd[10]) << 16;
  if((mthd[12]) < 0){
    mf->uses_smpte = 1;
    mf->smpte_fps = -mthd[12];
    mf->smpte_res = (unsigned) mthd[13];
    IIWU_LOG(ERR, "File uses SMPTE timing -- Not implemented yet");
    return IIWU_FAILED;
  } else {
    mf->uses_smpte = 0;
    mf->division = (mthd[12] << 8) | (mthd[13] & 0xff);
  }
  return IIWU_OK;
}

/*
 * iiwu_midi_file_load_tracks
 */
int iiwu_midi_file_load_tracks(iiwu_midi_file* mf, iiwu_player_t* player)
{
  int i;
  for (i = 0; i < mf->ntracks; i++) {
    if (iiwu_midi_file_read_track(mf, player) != IIWU_OK) {
      return IIWU_FAILED;
    }
  }
  return IIWU_OK;
}

/*
 * iiwu_isasciistring
 */
int iiwu_isasciistring(char* s)
{
  int i;
  int len = (int) IIWU_STRLEN(s);
  for (i = 0; i < len; i++) {
    if (!iiwu_isascii(s[i])) {
      return 0;
    }
  }
  return 1;
}

/*
 * iiwu_getlength
 */
long iiwu_getlength(unsigned char *s)
{
  long i = 0;
  i = s[3] | (s[2]<<8) | (s[1]<<16) | (s[0]<<24);
  return i;
}

/*
 * iiwu_midi_file_read_tracklen
 */
int iiwu_midi_file_read_tracklen(iiwu_midi_file* mf)
{
  unsigned char length[5];
  if (iiwu_midi_file_read(mf, length, 4) != IIWU_OK) {
    return IIWU_FAILED;
  }
  mf->tracklen = iiwu_getlength(length);
  mf->trackpos = 0;
  mf->eot = 0;
  return IIWU_OK;
}

/*
 * iiwu_midi_file_eot
 */
int iiwu_midi_file_eot(iiwu_midi_file* mf)
{
#if DEBUG
  if (mf->trackpos > mf->tracklen) {
    printf("track overrun: %d > %d\n", mf->trackpos, mf->tracklen);
  }
#endif
  return mf->eot || (mf->trackpos >= mf->tracklen);
}

/*
 * iiwu_midi_file_read_track
 */
int iiwu_midi_file_read_track(iiwu_midi_file* mf, iiwu_player_t* player)
{
  iiwu_track* track;
  unsigned char id[5], length[5];
  int found_track = 0;
  int skip;
  if (iiwu_midi_file_read(mf, id, 4) != IIWU_OK) {
    return IIWU_FAILED;
  }
  id[4]='\0';
  while (!found_track){
    if (iiwu_isasciistring((char*) id) == 0) {
      IIWU_LOG(ERR, "An non-ascii track header found, currupt file");
      return IIWU_FAILED;
    } else if (strcmp((char*) id, "MTrk") == 0) {
      found_track = 1;
      if (iiwu_midi_file_read_tracklen(mf) != IIWU_OK) {
	return IIWU_FAILED;
      }
      track = new_iiwu_track();
      if (track == NULL) {
	IIWU_LOG(ERR, "Out of memory");
	return IIWU_FAILED;
      }
      while (!iiwu_midi_file_eot(mf)) {
	if (iiwu_midi_file_read_event(mf, track) != IIWU_OK) {
	  return IIWU_FAILED;	  
	}
      }
      iiwu_player_add_track(player, track);
    } else {
      found_track = 0;
      if (iiwu_midi_file_read(mf, length, 4) != IIWU_OK) {
	return IIWU_FAILED;
      }
      skip = iiwu_getlength(length);
/*        fseek(mf->fp, skip, SEEK_CUR); */
      if (iiwu_midi_file_skip(mf, skip) != IIWU_OK) {
	return IIWU_FAILED;
      }
    }
  }
  if (feof(mf->fp)) {
    IIWU_LOG(ERR, "Unexpected end of file");
    return IIWU_FAILED;
  }
  return IIWU_OK;
}

/*
 * iiwu_midi_file_read_varlen
 */
int iiwu_midi_file_read_varlen(iiwu_midi_file* mf)
{ 
  int i;
  int c;
  mf->varlen = 0;
  for (i = 0;;i++) {
    if (i == 4) {
      IIWU_LOG(ERR, "Invalid variable length number");
      return IIWU_FAILED;
    }
    c = iiwu_midi_file_getc(mf);
    if (c < 0) {
      IIWU_LOG(ERR, "Unexpected end of file");
      return IIWU_FAILED;
    }
    if (c & 0x80){
      mf->varlen |= (int) (c & 0x7F);
      mf->varlen <<= 7;
    } else {
      mf->varlen += c;
      break;
    }    
  }
  return IIWU_OK;
}

/*
 * iiwu_midi_file_read_event
 */
int iiwu_midi_file_read_event(iiwu_midi_file* mf, iiwu_track* track)
{
  int dtime;
  int status;
  int type;
  int tempo;
  unsigned char metadata[1024];
  int nominator, denominator, clocks, notes, sf, mi;
  iiwu_midi_event_t* evt;
  int channel = 0;
  int param1 = 0;
  int param2 = 0;

  /* read the delta-time of the event */
  if (iiwu_midi_file_read_varlen(mf) != IIWU_OK) {
    return IIWU_FAILED;
  }
  dtime = mf->varlen;

  /* read the status byte */
  status = iiwu_midi_file_getc(mf);
  if (status < 0) {
    IIWU_LOG(ERR, "Unexpected end of file");
    return IIWU_FAILED;
  }

  /* not a valid status byte: use the running status instead */
  if ((status & 0x80) == 0) {
    if ((mf->running_status & 0x80) == 0) {
      IIWU_LOG(ERR, "Undefined status and invalid running status");
      return IIWU_FAILED;
    }
    iiwu_midi_file_push(mf, status);
    status = mf->running_status;
  } 

  /* check what message we have */
  if (status & 0x80) {
    mf->running_status = status;

    if ((status == MIDI_SYSEX) || (status == MIDI_EOX)) {     /* system exclusif */
      /*
       * Sysex messages are not handled yet
       */
      /* read the length of the message */
      if (iiwu_midi_file_read_varlen(mf) != IIWU_OK) {
	return IIWU_FAILED;
      }
      /* read the data of the message */
      if (iiwu_midi_file_read(mf, metadata, mf->varlen) != IIWU_OK) {
	return IIWU_FAILED;
      }
      return IIWU_OK;
      
    } else if (status == MIDI_META_EVENT) {             /* meta events */

      /* get the type of the meta message */
      type = iiwu_midi_file_getc(mf);
      if (type < 0) {
	IIWU_LOG(ERR, "Unexpected end of file");
	return IIWU_FAILED;
      }

      /* get the length of the data part */
      if (iiwu_midi_file_read_varlen(mf) != IIWU_OK) {
	return IIWU_FAILED;
      }
      if (mf->varlen > 1024) {
	IIWU_LOG(ERR, "Meta data buffer overflow");
	return IIWU_FAILED;
      }

      /* read the data */
      if (iiwu_midi_file_read(mf, metadata, mf->varlen) != IIWU_OK) {
	return IIWU_FAILED;
      }

      /* handle meta data */
      switch (type) {

      case MIDI_COPYRIGHT:
	metadata[mf->varlen] = 0;
	break;

      case MIDI_TRACK_NAME:
	metadata[mf->varlen] = 0;
	iiwu_track_set_name(track, metadata);
	break;

      case MIDI_INST_NAME:
	metadata[mf->varlen] = 0;
	break;

      case MIDI_LYRIC:
	break;

      case MIDI_MARKER:
	break;

      case MIDI_CUE_POINT:
	break; /* don't care much for text events */

      case MIDI_EOT:
	if (mf->varlen != 0) {
	  IIWU_LOG(ERR, "Invalid length for EndOfTrack event");
	  return IIWU_FAILED;
	}
	mf->eot = 1;
	break; 

      case MIDI_SET_TEMPO:
	if (mf->varlen != 3) {
	  IIWU_LOG(ERR, "Invalid length for SetTempo meta event");
	  return IIWU_FAILED;
	}
	tempo = (metadata[0] << 16) + (metadata[1] << 8) + metadata[2];
	evt = new_iiwu_midi_event();
	if (evt == NULL) {
	  IIWU_LOG(ERR, "Out of memory");
	  return IIWU_FAILED;
	}
	evt->dtime = dtime;
	evt->type = MIDI_SET_TEMPO;
	evt->channel = 0;
	evt->param1 = tempo;
	evt->param2 = 0;
	iiwu_track_add_event(track, evt);
	break; 

      case MIDI_SMPTE_OFFSET:
	if (mf->varlen != 5) {
	  IIWU_LOG(ERR, "Invalid length for SMPTE Offset meta event");
	  return IIWU_FAILED;
	}
	break; /* we don't use smtp */	

      case MIDI_TIME_SIGNATURE:
	if (mf->varlen != 4) {
	  IIWU_LOG(ERR, "Invalid length for TimeSignature meta event");
	  return IIWU_FAILED;
	}
	nominator = metadata[0];
	denominator = (int) pow(2.0, (double) metadata[1]);
	clocks = metadata[2];
	notes = metadata[3];

	IIWU_LOG(DBG, "signature=%d/%d, metronome=%d, 32nd-notes=%d\n", nominator, denominator, clocks, notes);

	break;

      case MIDI_KEY_SIGNATURE:
	if (mf->varlen != 2) {
	  IIWU_LOG(ERR, "Invalid length for KeySignature meta event");
	  return IIWU_FAILED;
	}
	sf = metadata[0];
	mi = metadata[1];
	break;

      case MIDI_SEQUENCER_EVENT:
	break;

      default:
	break;
      }
      return IIWU_OK;

    } else {                /* channel messages */

      type = status & 0xf0;
      channel = status & 0x0f;

      /* all channel message have at least 1 byte of associated data */
      if ((param1 = iiwu_midi_file_getc(mf)) < 0) {
	IIWU_LOG(ERR, "Unexpected end of file");
	return IIWU_FAILED;
      }

      switch (type) {

      case NOTE_ON:
	if ((param2 = iiwu_midi_file_getc(mf)) < 0) {
	  IIWU_LOG(ERR, "Unexpected end of file");
	  return IIWU_FAILED;
	}
	break;

      case NOTE_OFF:	
	if ((param2 = iiwu_midi_file_getc(mf)) < 0) {
	  IIWU_LOG(ERR, "Unexpected end of file");
	  return IIWU_FAILED;
	}
	break;

      case KEY_PRESSURE:
	if ((param2 = iiwu_midi_file_getc(mf)) < 0) {
	  IIWU_LOG(ERR, "Unexpected end of file");
	  return IIWU_FAILED;
	}
	break;

      case CONTROL_CHANGE:
	if ((param2 = iiwu_midi_file_getc(mf)) < 0) {
	  IIWU_LOG(ERR, "Unexpected end of file");
	  return IIWU_FAILED;
	}
	break;

      case PROGRAM_CHANGE:
	break;

      case CHANNEL_PRESSURE:
	break;

      case PITCH_BEND:
	if ((param2 = iiwu_midi_file_getc(mf)) < 0) {
	  IIWU_LOG(ERR, "Unexpected end of file");
	  return IIWU_FAILED;
	}
	param1 = ((param1 & 0x7f) << 7) | (param2 & 0x7f);
	param2 = 0;
	break;

      default:
	/* Can't possibly happen !? */
	IIWU_LOG(ERR, "Unrecognized MIDI event");
	return IIWU_FAILED;      
      }
      evt = new_iiwu_midi_event();
      if (evt == NULL) {
	IIWU_LOG(ERR, "Out of memory");
	return IIWU_FAILED;
      }
      evt->dtime = dtime;
      evt->type = type;
      evt->channel = channel;
      evt->param1 = param1;
      evt->param2 = param2;
      iiwu_track_add_event(track, evt);
    }
  }
  return IIWU_OK;
}

/*
 * iiwu_midi_file_get_division
 */
int iiwu_midi_file_get_division(iiwu_midi_file* midifile)
{
  return midifile->division;
}

/******************************************************
 *
 *     iiwu_track
 */

/*
 * new_iiwu_midi_event
 */
iiwu_midi_event_t* new_iiwu_midi_event()
{
  iiwu_midi_event_t* evt;
  evt = IIWU_NEW(iiwu_midi_event_t);
  if (evt == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;
  }
  evt->dtime = 0;
  evt->type = 0;
  evt->channel = 0;
  evt->param1 = 0;
  evt->param2 = 0;
  evt->next = NULL;
  return evt;
}

/*
 * delete_iiwu_midi_event
 */
int delete_iiwu_midi_event(iiwu_midi_event_t* evt)
{
  if (evt->next != NULL) {
    delete_iiwu_midi_event(evt->next);
  }
  IIWU_FREE(evt);
  return IIWU_OK;
}

/*
 * iiwu_midi_event_get_type
 */
int iiwu_midi_event_get_type(iiwu_midi_event_t* evt)
{
  return evt->type;
}

/*
 * iiwu_midi_event_set_type
 */
int iiwu_midi_event_set_type(iiwu_midi_event_t* evt, int type)
{
  evt->type = type;
  return IIWU_OK;
}

/*
 * iiwu_midi_event_get_channel
 */
int iiwu_midi_event_get_channel(iiwu_midi_event_t* evt)
{
  return evt->channel;
}

/*
 * iiwu_midi_event_set_channel
 */
int iiwu_midi_event_set_channel(iiwu_midi_event_t* evt, int chan)
{
  evt->channel = chan;
  return IIWU_OK;
}

/*
 * iiwu_midi_event_get_param1
 */
int iiwu_midi_event_get_param1(iiwu_midi_event_t* evt)
{
  return evt->param1;
}

/*
 * iiwu_midi_event_set_param1
 */
int iiwu_midi_event_set_param1(iiwu_midi_event_t* evt, int v)
{
  evt->param1 = v;
  return IIWU_OK;
}

/*
 * iiwu_midi_event_get_param2
 */
int iiwu_midi_event_get_param2(iiwu_midi_event_t* evt)
{
  return evt->param2;
}

/*
 * iiwu_midi_event_set_param2
 */
int iiwu_midi_event_set_param2(iiwu_midi_event_t* evt, int v)
{
  evt->param2 = v;
  return IIWU_OK;
}

/******************************************************
 *
 *     iiwu_track
 */

/*
 * new_iiwu_track
 */
iiwu_track* new_iiwu_track()
{
  iiwu_track* track;
  track = IIWU_NEW(iiwu_track);
  if (track == NULL) {
    return NULL;
  }
  track->name = NULL;
  track->first = NULL;
  track->cur = NULL;
  track->last = NULL;
  track->ticks = 0;
  return track;
}

/*
 * delete_iiwu_track
 */
int delete_iiwu_track(iiwu_track* track)
{
  if (track->name != NULL) {
    IIWU_FREE(track->name);
  }
  if (track->first != NULL) {
    delete_iiwu_midi_event(track->first);
  }
  IIWU_FREE(track);
  return IIWU_OK;
}

/*
 * iiwu_track_set_name
 */
int iiwu_track_set_name(iiwu_track* track, char* name)
{
  int len;
  if (track->name != NULL) {
    IIWU_FREE(track->name);
  }
  if (name == NULL) {
    track->name = NULL;
    return IIWU_OK;  
  }
  len = IIWU_STRLEN(name);
  track->name = IIWU_MALLOC(len + 1);
  if (track->name == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return IIWU_FAILED;
  }
  IIWU_STRCPY(track->name, name);
  return IIWU_OK;  
}

/*
 * iiwu_track_get_name
 */
char* iiwu_track_get_name(iiwu_track* track)
{
  return track->name;
}

/*
 * iiwu_track_get_duration
 */
int iiwu_track_get_duration(iiwu_track* track)
{
  int time = 0;
  iiwu_midi_event_t* evt = track->first;
  while (evt != NULL) {
    time += evt->dtime;
    evt = evt->next;
  }
  return time;
}

/*
 * iiwu_track_count_events
 */
int iiwu_track_count_events(iiwu_track* track, int* on, int* off)
{
  iiwu_midi_event_t* evt = track->first;
  while (evt != NULL) {
    if (evt->type == NOTE_ON) {
      (*on)++;
    } else if (evt->type == NOTE_OFF) {
      (*off)++;
    }
    evt = evt->next;
  }
  return IIWU_OK;
}

/*
 * iiwu_track_add_event
 */
int iiwu_track_add_event(iiwu_track* track, iiwu_midi_event_t* evt)
{
  evt->next = NULL;
  if (track->first == NULL) {
    track->first = evt;
    track->cur = evt;
    track->last = evt;
  } else {
    track->last->next = evt;
    track->last = evt;
  }
  return IIWU_OK;
}

/*
 * iiwu_track_first_event
 */
iiwu_midi_event_t* iiwu_track_first_event(iiwu_track* track)
{
  track->cur = track->first;
  return track->cur;
}

/*
 * iiwu_track_next_event
 */
iiwu_midi_event_t* iiwu_track_next_event(iiwu_track* track)
{
  if (track->cur != NULL) {
    track->cur = track->cur->next;
  }
  return track->cur;
}

/*
 * iiwu_track_reset
 */
int iiwu_track_reset(iiwu_track* track)
{
  track->ticks = 0;
  track->cur = track->first;
  return IIWU_OK;
}

/*
 * iiwu_track_send_events
 */
int iiwu_track_send_events(iiwu_track* track, 
			   iiwu_synth_t* synth,
			   iiwu_player_t* player,
			   long ticks)
{
  int status = IIWU_OK;
  iiwu_midi_event_t* event;
  while (1) {

    event = track->cur;
    if (event == NULL) {
      return status;
    }

    if (track->ticks + event->dtime > ticks) {
      return status;
    }

    track->ticks += event->dtime;
    status = iiwu_midi_send_event(synth, player, event);
    iiwu_track_next_event(track);
  }
}

/******************************************************
 *
 *     iiwu_player
 */
/*
 * new_iiwu_player
 */
iiwu_player_t* new_iiwu_player(iiwu_synth_t* synth)
{
  int i;
  iiwu_player_t* player;
  player = IIWU_NEW(iiwu_player_t);
  if (player == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;
  }
  player->status = IIWU_PLAYER_READY;
  player->loop = 0;
  player->ntracks = 0;
  for (i = 0; i < MAX_NUMBER_OF_TRACKS; i++) {
    player->track[i] = NULL;
  }
  player->synth = synth;
  player->timer = NULL;
  player->division = 0;
  player->send_program_change = 1;
  player->ticks_passed = 0;
  player->msec_passed = 0;
  player->miditempo = 480000;
  player->deltatime = 4.0;
  return player;
}

/*
 * delete_iiwu_player
 */
int delete_iiwu_player(iiwu_player_t* player)
{
  int i;
  if (player == NULL) {
    return IIWU_OK;
  }
  iiwu_player_stop(player);
  for (i = 0; i < MAX_NUMBER_OF_TRACKS; i++) {
    if (player->track[i] != NULL) {
      delete_iiwu_track(player->track[i]);
    }
  }
  IIWU_FREE(player);
  return IIWU_OK;
}

/*
 * iiwu_player_add_track
 */
int iiwu_player_add_track(iiwu_player_t* player, iiwu_track* track)
{
  if (player->ntracks < MAX_NUMBER_OF_TRACKS) {
    player->track[player->ntracks++] = track;
    return IIWU_OK;
  } else {
    return IIWU_FAILED;
  }
}

/*
 * iiwu_player_count_tracks
 */
int iiwu_player_count_tracks(iiwu_player_t* player)
{
  return player->ntracks;
}

/*
 * iiwu_player_get_track
 */
iiwu_track* iiwu_player_get_track(iiwu_player_t* player, int i)
{
  if ((i >= 0) && (i < MAX_NUMBER_OF_TRACKS)) {
    return player->track[i];
  } else {
    return NULL;
  }
}

/*
 * iiwu_player_load
 */
int iiwu_player_load(iiwu_player_t* player, char *filename)
{
  iiwu_midi_file* midifile;
  midifile = new_iiwu_midi_file(filename); 
  if (midifile == NULL) {
    return IIWU_FAILED;
  }
  player->division = iiwu_midi_file_get_division(midifile);

  IIWU_LOG(DBG, "quarter note division=%d\n", player->division);

  if (iiwu_midi_file_load_tracks(midifile, player) != IIWU_OK){
    return IIWU_FAILED;
  }
  delete_iiwu_midi_file(midifile);
  return IIWU_OK;  
}

/*
 * iiwu_player_callback
 */
int iiwu_player_callback(void* data, long msec)
{
  int i;
  long ticks;
  int status = IIWU_PLAYER_DONE;
  iiwu_player_t* player;
  iiwu_synth_t* synth;
  player = (iiwu_player_t*) data;
  synth = player->synth;

  debug_msec = msec;

  ticks = (long) ((double) msec / player->deltatime);
  for (i = 0; i < player->ntracks; i++) {
    if (!iiwu_track_eot(player->track[i])) {
      status = IIWU_PLAYER_PLAYING;
      if (iiwu_track_send_events(player->track[i], synth, player, ticks) != IIWU_OK) {
	/* */
      }
    }
  }
  player->status = status;
  player->ticks_passed = ticks;
  player->msec_passed = msec;
  return (player->status == IIWU_PLAYER_DONE)? 0 : 1;
}

/*
 * iiwu_player_play
 */
int iiwu_player_play(iiwu_player_t* player)
{
  int i;
  if (player->status == IIWU_PLAYER_PLAYING) {
    return IIWU_OK;
  }
  player->ticks_passed = 0;
  player->msec_passed = 0;
  player->status = IIWU_PLAYER_PLAYING;
  for (i = 0; i < player->ntracks; i++) {
    if (player->track[i] != NULL) {
      iiwu_track_reset(player->track[i]);
    }
  }
  player->timer = new_iiwu_timer((int) player->deltatime, iiwu_player_callback, (void*) player);
  if (player->timer == NULL) {
    return IIWU_FAILED;
  }
  return IIWU_OK;
}

/*
 * iiwu_player_stop
 */
int iiwu_player_stop(iiwu_player_t* player)
{
  if (player->timer != NULL) {
    delete_iiwu_timer(player->timer);
  }
  player->status = IIWU_PLAYER_DONE;
  player->timer = NULL;
  return IIWU_OK;
}

/*
 * iiwu_player_set_loop
 */
int iiwu_player_set_loop(iiwu_player_t* player, int loop)
{
  player->loop = loop;
  return IIWU_OK;
}

/*
 *  iiwu_player_set_midi_tempo
 */
int iiwu_player_set_midi_tempo(iiwu_player_t* player, int tempo)
{
  player->miditempo = tempo;
  player->deltatime = (double) tempo / player->division / 1000.0; /* in milliseconds */

  IIWU_LOG(DBG,"tempo=%d\n", tempo);
  IIWU_LOG(DBG,"tick time=%f msec\n", player->deltatime);

  return IIWU_OK;
}

/*
 *  iiwu_player_set_bpm
 */
int iiwu_player_set_bpm(iiwu_player_t* player, int bpm)
{
  return iiwu_player_set_midi_tempo(player, (int)((double) 60 * 1e6 / bpm));
}

/*
 *  iiwu_player_join
 */
int iiwu_player_join(iiwu_player_t* player)
{
  return iiwu_timer_join(player->timer);
}

/************************************************************************
 *       MIDI PARSER
 *
 */

#define IS_SYSTEM_EVENT(_c)  ((_c & 0xf0) == (unsigned char)0xf0)
#define IS_CHANNEL_EVENT(_c)  (_c & 0x80)
#define PARAM_CLEAR 0x80

/*
 * new_iiwu_midi_parser
 */
iiwu_midi_parser_t* new_iiwu_midi_parser()
{
  iiwu_midi_parser_t* parser;
  parser = IIWU_NEW(iiwu_midi_parser_t);
  if (parser == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;
  }
  parser->status = 0;
  parser->channel = 0; 
  parser->running_status = 0;
  parser->p1 = PARAM_CLEAR;
  parser->p2 = 0;
  return parser;
}

/*
 * delete_iiwu_midi_parser
 */
int delete_iiwu_midi_parser(iiwu_midi_parser_t* parser)
{
  IIWU_FREE(parser);
  return IIWU_OK;
}

/*
 * iiwu_midi_parser_parse
 */
iiwu_midi_event_t* iiwu_midi_parser_parse(iiwu_midi_parser_t* parser, unsigned char c)
{
  /* if we're waiting for a new event, we expect either :
     - a system event (system exclusive, system common, system real-time),
     - a channel event, or 
     - an event using the running status.
  */
  if (parser->status == 0) {
    if (IS_SYSTEM_EVENT(c)) {
      /* just set the status and return. no post processing done (yet). */
      parser->status = c;
      parser->running_status = c;
      switch (parser->status) {
      case MIDI_SYSEX:
      case MIDI_TIME_CODE:
      case MIDI_SONG_POSITION:
      case MIDI_SONG_SELECT:
	/* not handled yet but more data expected */
	return NULL;

      case MIDI_EOX:
      case MIDI_SYNC:
      case MIDI_START:
      case MIDI_CONTINUE:
      case MIDI_STOP:
      case MIDI_ACTIVE_SENSING:
      case MIDI_SYSTEM_RESET:
      case MIDI_TUNE_REQUEST:
	/* not handled yet and no more data expected */
	parser->status = 0;
	return NULL;
      default: /* error! */
	parser->status = 0;
	return NULL;
      }
    } else if (IS_CHANNEL_EVENT(c)) {
      /* just set the status and return. no post-processing needed. */
      parser->status = c & 0xf0;
      parser->channel = c & 0x0f;
      parser->running_status = c;
      return NULL;
    } else {
      /* now we now the status. post-processing the data. */
      if (IS_SYSTEM_EVENT(parser->running_status)) {
	parser->status = parser->running_status;
      } else {
	parser->status = parser->running_status & 0xf0;
	parser->channel = parser->running_status & 0x0f;      
      }
    }
  }

  switch (parser->status) {
  case NOTE_OFF:
  case NOTE_ON:
  case KEY_PRESSURE: 
  case CONTROL_CHANGE:
    if (parser->p1 == PARAM_CLEAR) {
      parser->p1 = c;
      return NULL;
    } else {
      parser->p2 = c;
      goto return_event;
    }

  case PITCH_BEND:
    if (parser->p1 == PARAM_CLEAR) {
      parser->p1 = c;
      return NULL;
    } else {
      parser->p1 = ((c & 0x7f) << 7) | (parser->p1 & 0x7f);
      parser->p2 = 0;
      goto return_event;
    }

  case PROGRAM_CHANGE:
  case CHANNEL_PRESSURE:
    parser->p1 = c;
    goto return_event;

  case MIDI_SYSEX:
    /* not handled yet. just wait till a MIDI_EOX comes along */
    return NULL;

  case MIDI_SONG_POSITION:
    if (parser->p1 == PARAM_CLEAR) {
      return NULL;
    } else {
      parser->p1 = PARAM_CLEAR;
      parser->status = 0;
      return NULL;
    }
    
  case MIDI_SONG_SELECT:
    parser->status = 0;
    return NULL;

  default:
    return NULL;
  }

 return_event:
  /* avoid copying! */
  parser->event.type = parser->status;
  parser->event.channel = parser->channel;
  parser->event.param1 = parser->p1;
  parser->event.param2 = parser->p2;
  parser->p1 = PARAM_CLEAR;
  parser->status = 0;
  return &parser->event;
}

/************************************************************************
 *       iiwu_midi_send_event
 *
 * This is a utility function that doesn't really belong to any class
 * or structure. It is called by iiwu_midi_track and iiwu_midi_device.
 */
int iiwu_midi_send_event(iiwu_synth_t* synth, iiwu_player_t* player, iiwu_midi_event_t* event)
{
  switch (event->type) {
  case NOTE_ON:
    if (iiwu_synth_noteon(synth, event->channel, event->param1, event->param2) != IIWU_OK) {
      return IIWU_FAILED;
    }
    break;
  case NOTE_OFF:
    if (iiwu_synth_noteoff(synth, event->channel, event->param1) != IIWU_OK) {
      return IIWU_FAILED;
    }
    break;
  case CONTROL_CHANGE:
    if (iiwu_synth_cc(synth, event->channel, event->param1, event->param2) != IIWU_OK) {
      return IIWU_FAILED;
    }
    break;
  case MIDI_SET_TEMPO:
    if (player != NULL) {
      if (iiwu_player_set_midi_tempo(player, event->param1) != IIWU_OK) {
	return IIWU_FAILED;
      }
    }
    break;
  case PROGRAM_CHANGE:
    if (iiwu_synth_program_change(synth, event->channel, event->param1) != IIWU_OK) {
      return IIWU_FAILED;
    }
  case PITCH_BEND:
    if (iiwu_synth_pitch_bend(synth, event->channel, event->param1) != IIWU_OK) {
      return IIWU_FAILED;
    }
    break;
  default:
    break;
  }
  return IIWU_OK;
}

/************************************************************************
 *       MIDI HANDLER
 *
 */

/*
 * new_iiwu_midi_handler
 */
iiwu_midi_handler_t* new_iiwu_midi_handler(iiwu_synth_t* synth, char* driver, char* device)
{
  int i;
  iiwu_midi_handler_t* handler;

  /* create the handler and clear it */
  handler = IIWU_NEW(iiwu_midi_handler_t);
  if (handler == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;
  }
  IIWU_MEMSET(handler, 0, sizeof(iiwu_midi_handler_t));

  handler->synth = synth;

  /* copy the device name */
  if (device != NULL) {
    handler->device_name = IIWU_MALLOC(IIWU_STRLEN(device) + 1);
    if (handler->device_name == NULL) {
      IIWU_LOG(ERR, "Out of memory");
      IIWU_FREE(handler);
      return NULL;
    }
    IIWU_STRCPY(handler->device_name, device);
  }

  /* choose the default driver if none is specified */
  if (driver == NULL) {
    driver = iiwu_default_midi_driver;
  }

  /* copy the driver name */
  handler->driver_name = IIWU_MALLOC(IIWU_STRLEN(driver) + 1);
  if (handler->driver_name == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    IIWU_FREE(handler);
    return NULL;
  }
  IIWU_STRCPY(handler->driver_name, driver);

  /* find the driver definition */
  i = 0;
  while (iiwu_midi_drivers[i].name != NULL) {
    if (IIWU_STRCMP(driver, iiwu_midi_drivers[i].name) == 0) {
      IIWU_LOG(DBG, "Using %s MIDI driver\n", iiwu_midi_drivers[i].name);
      handler->new_driver = iiwu_midi_drivers[i].new_driver;
      handler->delete_driver = iiwu_midi_drivers[i].delete_driver;
      handler->join_driver = iiwu_midi_drivers[i].join_driver;
      handler->status_driver = iiwu_midi_drivers[i].status_driver;
      break;
    }
    i++;
  }

  /* make sure we found a driver */
  if ((handler->new_driver == NULL) 
      || (handler->delete_driver == NULL) 
      || (handler->join_driver == NULL)) {
    IIWU_LOG(ERR, "Couldn't find the requested MIDI driver");
    goto error_recovery;
  }

  /* create the low-level MIDI driver */
  handler->driver = handler->new_driver(handler);
  if (handler->driver == NULL) {
    goto error_recovery;
  }
  return handler;

 error_recovery:
    IIWU_FREE(handler);
    return NULL;
}

/*
 * delete_iiwu_midi_handler
 */
int delete_iiwu_midi_handler(iiwu_midi_handler_t* handler)
{
  if (handler == NULL) {
    return IIWU_OK;
  }
  if (handler->driver != NULL) {
    handler->delete_driver(handler->driver);
  }
  IIWU_FREE(handler);
  return IIWU_OK;
}

/*
 * iiwu_midi_handler_set_device_name
 */
void iiwu_midi_handler_set_device_name(iiwu_midi_handler_t* handler, char* name)
{
  if (handler->device_name != NULL) {
    IIWU_FREE(handler->device_name);
    handler->device_name = NULL;
  }
  if (name == NULL) {
    return;
  }
  handler->device_name = IIWU_MALLOC(IIWU_STRLEN(name) + 1);
  if (handler->device_name == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return;
  }
  IIWU_STRCPY(handler->device_name, name);
}

/*
 * iiwu_midi_handler_join
 */
int iiwu_midi_handler_join(iiwu_midi_handler_t* handler)
{
  return handler->join_driver(handler->driver);
}

/*
 * iiwu_midi_handler_send_event
 */
int iiwu_midi_handler_send_event(iiwu_midi_handler_t* handler, iiwu_midi_event_t* event)
{
  iiwu_synth_t* synth = handler->synth;
  switch (event->type) {
  case NOTE_ON:
    handler->noteon_count++;
    if (iiwu_synth_noteon(synth, event->channel, event->param1, event->param2) != IIWU_OK) {
      handler->error_count++;
      return IIWU_FAILED;
    }
    break;
  case NOTE_OFF:
    handler->noteoff_count++;
    if (iiwu_synth_noteoff(synth, event->channel, event->param1) != IIWU_OK) {
      handler->error_count++;
      return IIWU_FAILED;
    }
    break;
  case CONTROL_CHANGE:
    if (iiwu_synth_cc(synth, event->channel, event->param1, event->param2) != IIWU_OK) {
      handler->error_count++;
      return IIWU_FAILED;
    }
    break;
  case MIDI_SET_TEMPO:
    break;
  case PROGRAM_CHANGE:
    handler->prog_count++;
    if (iiwu_synth_program_change(synth, event->channel, event->param1) != IIWU_OK) {
      handler->error_count++;
      return IIWU_FAILED;
    }
  case PITCH_BEND:
    handler->pbend_count++;
    if (iiwu_synth_pitch_bend(synth, event->channel, event->param1) != IIWU_OK) {
      handler->error_count++;
      return IIWU_FAILED;
    }
    break;
  default:
    break;
  }
  return IIWU_OK;
}


char* iiwu_midi_handler_get_driver_name(iiwu_midi_handler_t* handler)
{
  return handler->driver_name;
}

char* iiwu_midi_handler_get_device_name(iiwu_midi_handler_t* handler)
{
  return handler->device_name;
}

char* iiwu_midi_handler_get_status(iiwu_midi_handler_t* handler)
{
  char* s;
  int status = handler->status_driver(handler->driver);
  switch (status) {
  case IIWU_MIDI_READY: 
    s = "ready"; 
    break;
  case IIWU_MIDI_LISTENING: 
    s = "listening"; 
    break;
  case IIWU_MIDI_DONE: 
    s = "done"; 
    break;
  default:
    s = "unknown";
  }
  sprintf(midi_message_buffer, "%s", s);
  return midi_message_buffer;  
}

int iiwu_midi_handler_get_event_count(iiwu_midi_handler_t* handler, int type)
{
  switch (type) {
  case 0:
    return handler->error_count;
  case NOTE_ON:
    return handler->noteon_count;
  case NOTE_OFF:
    return handler->noteoff_count;
  case PROGRAM_CHANGE:
    return handler->prog_count;
  case PITCH_BEND:
    return handler->pbend_count;
  default:
    return 0;
  }
}



#if HAVE_LIBPTHREAD

#include <pthread.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>

long iiwu_curtime(void);

/************************************************************************
 *       PTHREAD TIMERS
 *
 */
struct _iiwu_timer 
{
  long msec;
  iiwu_timer_callback callback;
  void* data;
  pthread_t thread;
  int cont;
};

long iiwu_curtime()
{
  struct timeval now;
  gettimeofday(&now, NULL);
  return now.tv_sec * 1000 + now.tv_usec / 1000;
}

void* iiwu_timer_start(void *data)
{
  int count = 0;
  int cont = 1;
  long start;
  long delay;
  iiwu_timer* timer;
  timer = (iiwu_timer*) data;

  /* keep track of the start time for absolute positioning */
  start = iiwu_curtime();

  while (cont) {

    /* do whatever we have to do */
    cont = (*timer->callback)(timer->data, iiwu_curtime() - start);

    count++;

    /* to avoid incremental time errors, I calculate the delay between
       two callbacks bringing in the "absolute" time (count *
       timer->msec) */
    delay = (count * timer->msec) - (iiwu_curtime() - start);
    if (delay > 0) {
      usleep(delay * 1000);
    }

    cont &= timer->cont;
  }

  IIWU_LOG(DBG, "Player thread finished");
  pthread_exit(NULL);
  return NULL;
}

iiwu_timer* new_iiwu_timer(int msec, iiwu_timer_callback callback, void* data)
{
  iiwu_timer* timer = IIWU_NEW(iiwu_timer);
  if (timer == NULL) {
    IIWU_LOG(ERR, "Out of memory");     
    return NULL;
  }
  timer->msec = msec;
  timer->callback = callback;
  timer->data = data;
  timer->cont = 1;
  if (pthread_create(&timer->thread, NULL, iiwu_timer_start, (void*) timer)) {
    IIWU_LOG(ERR, "Failed to create the timer thread");
    IIWU_FREE(timer);
    return NULL;
  }
  return timer;
}

int delete_iiwu_timer(iiwu_timer* timer)
{
  timer->cont = 0;
  pthread_join(timer->thread, NULL);
  IIWU_LOG(DBG, "Joined player thread");
  IIWU_FREE(timer);
  return IIWU_OK;
}

int iiwu_timer_join(iiwu_timer* timer)
{
  int err = pthread_join(timer->thread, NULL);
  IIWU_LOG(DBG, "Joined player thread");
  return (err == 0)? IIWU_OK : IIWU_FAILED;
}

#elif defined(WIN32)


/************************************************************************
 *   WIN32 TIMER
 *
 */

#include <windows.h>

#define USE_WAITABLE_TIMER   0

struct _iiwu_timer 
{
  long msec;
  iiwu_timer_callback callback;
  void* data;
#if USE_WAITABLE_TIMER
  HANDLE wtimer; 
#endif
  HANDLE thread;
  DWORD thread_id;
  int cont;
};

static int iiwu_timer_count = 0;
DWORD WINAPI iiwu_timer_run(LPVOID data);

#if USE_WAITABLE_TIMER
VOID CALLBACK iiwu_timer_handle(LPVOID data, DWORD low, DWORD hi); 
#endif

#define  iiwu_curtime GetTickCount

iiwu_timer* new_iiwu_timer(int msec, iiwu_timer_callback callback, void* data)
{
  iiwu_timer* timer = IIWU_NEW(iiwu_timer);
  if (timer == NULL) {
    IIWU_LOG(ERR, "Out of memory");     
    return NULL;
  }
  timer->cont = 1;
  timer->msec = msec;
  timer->callback = callback;
  timer->data = data;
#if USE_WAITABLE_TIMER
  sprintf(s, "iiwu_timer_%i", iiwu_timer_count++);
  timer->wtimer = CreateWaitableTimer(NULL, FALSE, s);
  if (timer->wtimer == NULL) {
    IIWU_LOG(ERR, "Couldn't create timer object");     
    return NULL;
  }
#endif
  timer->thread = CreateThread(NULL, 0, iiwu_timer_run, (LPVOID) timer, 0, &timer->thread_id);
  if (timer->thread == NULL) {
    IIWU_LOG(ERR, "Couldn't create timer thread");     
    return NULL;
  }
  SetThreadPriority(timer->thread, THREAD_PRIORITY_TIME_CRITICAL);
  return timer;
}

#if USE_WAITABLE_TIMER
DWORD WINAPI iiwu_timer_run(LPVOID data)
{
  DWORD wait_result;
  __int64 due_time;
  iiwu_timer* timer;
  long start;
  int count = 0;
  timer = (iiwu_timer*) data;
  due_time = -1;
  SetWaitableTimer(timer->wtimer, (const LARGE_INTEGER*) &due_time, timer->msec, 
		     iiwu_timer_handle, (LPVOID) timer, FALSE);  
  start = iiwu_curtime();
  while (timer->cont) {
    wait_result = WaitForSingleObject(timer->wtimer, 2 * timer->msec);
    if (wait_result == WAIT_ABANDONED) {
      printf("WaitForSingleObject returned WAIT_ABANDONED\n");
    } else if (wait_result == WAIT_TIMEOUT) {
      LOG_ERR("WaitForSingleObject returned WAIT_TIMEOUT\n");
    } else if (wait_result == WAIT_OBJECT_0) {
      LOG_ERR("WaitForSingleObject returned WAIT_OBJECT_0\n");
    }
    printf("timer count = %d\n", count++);
  }
  CancelWaitableTimer(timer->wtimer);
  ExitThread(0);
  return 0;
}
#else
DWORD WINAPI iiwu_timer_run(LPVOID data)
{
  int count = 0;
  int cont = 1;
  long start;
  long delay;
  iiwu_timer* timer;
  timer = (iiwu_timer*) data;

  if ((timer == NULL) || (timer->callback == NULL)) {
    return 0;
  }

  /* keep track of the start time for absolute positioning */
  start = iiwu_curtime();

  while (cont) {

    /* do whatever we have to do */
    cont = (*timer->callback)(timer->data, iiwu_curtime() - start);

    count++;

    /* to avoid incremental time errors, I calculate the delay between
       two callbacks bringing in the "absolute" time (count *
       timer->msec) */
    delay = (count * timer->msec) - (iiwu_curtime() - start);
    if (delay > 0) {
      Sleep(delay);
    }

    cont &= timer->cont;
  }

  IIWU_LOG(DBG, "Player thread finished");
  ExitThread(0);
  return 0;
}

#endif

#if USE_WAITABLE_TIMER
VOID CALLBACK iiwu_timer_handle(LPVOID data, DWORD low, DWORD hi)
{
  iiwu_timer* timer;
  timer = (iiwu_timer*) data;
  printf("iiwu_timer_handle called\n");       
}
#endif

iiwu_status delete_iiwu_timer(iiwu_timer* timer)
{
  DWORD wait_result;
  timer->cont = 0;
  wait_result = WaitForSingleObject(timer->thread, 10 * timer->msec);
  return (wait_result == WAIT_OBJECT_0)? IIWU_OK : IIWU_FAILED;
}

int iiwu_timer_join(iiwu_timer* timer)
{
  DWORD wait_result;
  wait_result = WaitForSingleObject(timer->thread, INFINITE);
  return (wait_result == WAIT_OBJECT_0)? IIWU_OK : IIWU_FAILED;
}

#endif
