package rtrlib

import (
	"encoding/binary"
	"github.com/stretchr/testify/assert"
	"net"
	"testing"
)

func GenerateRoas(size uint32, offset uint32) []ROA {
	roas := make([]ROA, size)
	for i := uint32(0); i < size; i++ {
		ipFinal := make([]byte, 4)
		binary.BigEndian.PutUint32(ipFinal, i+offset)
		roas[i] = ROA{
			Prefix: net.IPNet{
				IP:   net.IP(append([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, ipFinal...)),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    64496,
		}
	}
	return roas
}

func BaseBench(base int, multiplier int) {
	benchSize1 := base * multiplier
	newRoas := GenerateRoas(uint32(benchSize1), uint32(0))
	benchSize2 := base
	prevRoas := GenerateRoas(uint32(benchSize2), uint32(benchSize1-benchSize2/2))
	ComputeDiff(newRoas, prevRoas)
}

func BenchmarkComputeDiff1000x10(b *testing.B) {
	BaseBench(1000, 10)
}

func BenchmarkComputeDiff10000x10(b *testing.B) {
	BaseBench(10000, 10)
}

func BenchmarkComputeDiff100000x1(b *testing.B) {
	BaseBench(100000, 1)
}

func TestComputeDiff(t *testing.T) {
	newRoas := []ROA{
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65003,
		},
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65002,
		},
	}
	prevRoas := []ROA{
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65001,
		},
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65002,
		},
	}
	added, removed, unchanged := ComputeDiff(newRoas, prevRoas)
	assert.Len(t, added, 1)
	assert.Len(t, removed, 1)
	assert.Len(t, unchanged, 1)
	assert.Equal(t, added[0].ASN, uint32(65003))
	assert.Equal(t, removed[0].ASN, uint32(65001))
	assert.Equal(t, unchanged[0].ASN, uint32(65002))
}

func TestApplyDiff(t *testing.T) {
	diff := []ROA{
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65003,
			Flags:  FLAG_ADDED,
		},
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65002,
			Flags:  FLAG_REMOVED,
		},
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65004,
			Flags:  FLAG_REMOVED,
		},
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65006,
			Flags:  FLAG_REMOVED,
		},
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65007,
			Flags:  FLAG_ADDED,
		},
	}
	prevRoas := []ROA{
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65001,
			Flags:  FLAG_ADDED,
		},
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65002,
			Flags:  FLAG_ADDED,
		},
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65005,
			Flags:  FLAG_REMOVED,
		},
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65006,
			Flags:  FLAG_REMOVED,
		},
		ROA{
			Prefix: net.IPNet{
				IP:   net.IP([]byte{0xfd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7}),
				Mask: net.IPMask([]byte{0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}),
			},
			MaxLen: 128,
			ASN:    65007,
			Flags:  FLAG_REMOVED,
		},
	}
	roas := ApplyDiff(diff, prevRoas)

	assert.Len(t, roas, 6)
	assert.Equal(t, roas[0].ASN, uint32(65001))
	assert.Equal(t, roas[0].Flags, uint8(FLAG_ADDED))
	assert.Equal(t, roas[1].ASN, uint32(65005))
	assert.Equal(t, roas[1].Flags, uint8(FLAG_REMOVED))
	assert.Equal(t, roas[2].ASN, uint32(65003))
	assert.Equal(t, roas[2].Flags, uint8(FLAG_ADDED))
	assert.Equal(t, roas[3].ASN, uint32(65004))
	assert.Equal(t, roas[3].Flags, uint8(FLAG_REMOVED))
	assert.Equal(t, roas[4].ASN, uint32(65006))
	assert.Equal(t, roas[4].Flags, uint8(FLAG_REMOVED))
	assert.Equal(t, roas[5].ASN, uint32(65007))
	assert.Equal(t, roas[5].Flags, uint8(FLAG_ADDED))
}
