package nvd

import (
	"reflect"
	"testing"

	c "github.com/kotakanbe/go-cve-dictionary/config"
)

func TestNvdFeedURLToYear(t *testing.T) {
	var tests = []struct {
		in   string
		year int
		xml  bool
	}{
		{
			in:   "https://nvd.nist.gov/feeds/json/cve/1.0/nvdcve-1.0-2018.json.gz",
			year: 2018,
			xml:  false,
		},
		{
			in:   "https://nvd.nist.gov/feeds/json/cve/1.0/nvdcve-1.0-recent.json.gz",
			year: c.Latest,
			xml:  false,
		},
		{
			in:   "https://nvd.nist.gov/feeds/json/cve/1.0/nvdcve-1.0-modified.json.gz",
			year: c.Latest,
			xml:  false,
		},
		{
			in:   "https://nvd.nist.gov/feeds/xml/cve/nvdcve-2.0-2018.xml.gz",
			year: 2018,
			xml:  true,
		},
		{
			in:   "https://nvd.nist.gov/feeds/xml/cve/nvdcve-2.0-recent.xml.gz",
			year: c.Latest,
			xml:  true,
		},
		{
			in:   "https://nvd.nist.gov/feeds/xml/cve/nvdcve-2.0-modified.xml.gz",
			year: c.Latest,
			xml:  true,
		},
	}

	for i, tt := range tests {
		y, xml, err := nvdFeedURLToYear(tt.in)
		if err != nil {
			t.Errorf("[%d] err: %s", i, err)
		}
		if y != tt.year {
			t.Errorf("[%d] expected: %v\n  actual: %v\n", i, y, tt.year)
		}
		if xml != tt.xml {
			t.Errorf("[%d] expected: %v\n  actual: %v\n", i, y, tt.xml)
		}
	}
}

func TestMakeNvdMetaURL(t *testing.T) {
	var tests = []struct {
		year int
		xml  bool
		url  []string
	}{
		{
			year: 2018,
			xml:  false,
			url:  []string{"https://nvd.nist.gov/feeds/json/cve/1.0/nvdcve-1.0-2018.meta"},
		},
		{
			year: c.Latest,
			xml:  false,
			url: []string{
				"https://nvd.nist.gov/feeds/json/cve/1.0/nvdcve-1.0-modified.meta",
				"https://nvd.nist.gov/feeds/json/cve/1.0/nvdcve-1.0-recent.meta",
			},
		},
		{
			year: 2018,
			xml:  true,
			url:  []string{"https://nvd.nist.gov/feeds/xml/cve/nvdcve-2.0-2018.meta"},
		},
		{
			year: c.Latest,
			xml:  true,
			url: []string{
				"https://nvd.nist.gov/feeds/xml/cve/nvdcve-2.0-modified.meta",
				"https://nvd.nist.gov/feeds/xml/cve/nvdcve-2.0-recent.meta",
			},
		},
	}

	for i, tt := range tests {
		url := MakeNvdMetaURLs(tt.year, tt.xml)
		if !reflect.DeepEqual(url, tt.url) {
			t.Errorf("[%d] expected: %v\n  actual: %v\n", i, tt.url, url)
		}
	}
}
