/* vi:set ts=8 sts=0 sw=8:
 * $Id: main.c,v 1.3 2001/01/23 17:29:27 kahn Exp kahn $
 *
 * Copyright (C) 2000 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <time.h>
#include <gnome.h>
#include <glade/glade.h>

#include "main.h"
#include "win.h"
#include "prefs.h"
#include "appinfo.h"


/*
 * The main structure which holds information about the "application" itself.
 */
struct ef_app_s {
	GSList *	win;		/* list of windows */
	GtkWidget **	prefs;		/* preferences dialog widgets */
	GtkWidget *	about;		/* about dialog */
};
typedef struct ef_app_s ef_app_t;

static ef_app_t ef_app;


/*** PRIVATE FUNCTION PROTOTYPES ***/
static void ef_show_info(void);
static void ef_show_vers(void);
static void print_version(void);
static void print_appinfo(void);

/* make these variables global to fend off strict ANSI-C compiler warnings */
static gboolean print_info = FALSE;
static gboolean print_vers = FALSE;
#ifdef ENABLE_DEBUG
static char *dbgstr = NULL;
#endif
static const struct poptOption gnopts[] = {
	{
		"version",
		'v',
		POPT_ARG_NONE,
		&print_vers,
		0,
		N_("Print version information"),
		NULL
	},
	{
		"info",
		'i',
		POPT_ARG_NONE,
		&print_info,
		0,
		N_("Print compile/link options used"),
		NULL
	},
#ifdef ENABLE_DEBUG
	{
		"debug",
		'd',
		POPT_ARG_STRING,
		&dbgstr,
		0,
		N_("Set debug flag to FLAGS (see debug.h)"),
		N_("FLAGS")
	},
#endif
	{ NULL }
};

/*** PUBLIC FUNCTION DEFINTIONS ***/
int
main(int argc, char *argv[])
{
	win_t *w;
	poptContext ctx;

#ifdef ENABLE_NLS
	bindtextdomain(PACKAGE, PACKAGE_LOCALE_DIR);
	textdomain(PACKAGE);
#endif

	/*
	 * Disable the stupid GNOME crash dialog.  All it ever does it cause
	 * people to send in useless bug reports to bugs.gnome.org.
	 */
	(void)putenv("GNOME_DISABLE_CRASH_DIALOG=1");

	gnome_init_with_popt_table(PACKAGE, VERSION, argc, argv, gnopts,
				   0, &ctx);
	if (print_vers)
		ef_show_vers();	/* does not return */
	if (print_info)
		ef_show_info();	/* does not return */
#ifdef ENABLE_DEBUG
	if (dbgstr) {
		if (dbgstr[0] == '0')
			sscanf(dbgstr, "%x", &dbg_flags);
		else    
			sscanf(dbgstr, "%d", &dbg_flags);
	}
#endif
	glade_gnome_init();

	srand((unsigned int)time(NULL));

	prefs_init();
	w = win_new();
	ef_win_add(w);

	gtk_main();
#ifdef ENABLE_DEBUG
	APPDBG_ALWAYS(("%s: clean exit.\n", PACKAGE));
#endif
	return 0;
} /* main */


/*
 * Return pointer to array of widgets for prefs dialog.
 */
GtkWidget **
ef_prefs_wgts_get(void)
{
	return ef_app.prefs;
} /* ef_prefs_wgts_get */


/*
 * Sets array of widgets for prefs dialog.
 */
void
ef_prefs_wgts_set(GtkWidget **wgts)
{
	g_free(ef_app.prefs);
	ef_app.prefs = wgts;
} /* ef_prefs_wgts_set */


/*
 * Callback for menu item "File->Exit"
 */
void
ef_quit_cb(GtkWidget *wgt, gpointer cbdata)
{
	GSList *wp;
	win_t *w;

	/*
	 * Look through all the windows and mark them for closing.  If any of
	 * them have a search in progress, don't close that window right away.
	 * Instead, mark that window for closing, and close it once we abort
	 * the search.
	 *
	 * Note that we could have just done an exit() here, but that's kind
	 * of abrupt and if we ever create temporary files or other things, it
	 * will require cleanup.
	 */
	wp = ef_app.win;
	while (wp) {
		w = (win_t *)wp->data;
		wp = wp->next;

		if (w->is_searching) {
			w->close_window = TRUE;
			w->stop_search = TRUE;
		} else
			win_close_cb(NULL, w);
	}
} /* ef_quit_cb */


/*
 * Adds a new window to the window list.
 */
void
ef_win_add(win_t *w)
{
	ef_app.win = g_slist_append(ef_app.win, w);
} /* ef_win_add */


/*
 * Removes a window to the window list.
 */
void
ef_win_remove(win_t *w)
{
	GSList *wp;

	gtk_timeout_remove(w->timeout_id);
	wp = g_slist_find(ef_app.win, w);
	ef_app.win = g_slist_remove_link(ef_app.win, wp);
	g_free(w->mainwgts);
	g_free(w->reswgts);
	g_free(w->advwgts);
	g_free(w->filewgts);
	g_free(w->dirwgts);
	g_free(w->pcmdwgts);
	g_free(w->finfowgts);
	g_free(w->lastmsg);
	g_free(w);
	g_slist_free_1(wp);

	if (g_slist_length(ef_app.win) < 1)
		gtk_main_quit();
} /* ef_win_remove */


/*
 * Updates all prefs settings for all windows.
 */
void
ef_windows_update_all_prefs(void)
{
	GSList *wp;

	for (wp = ef_app.win; wp; wp = wp->next)
		win_prefs_set((win_t *)wp->data);
} /* ef_windows_update_all_prefs */


/*** PRIVATE FUNCTION DEFINITIONS ***/
static void
print_version(void)
{
	printf("%s %s (%s)\n", PACKAGE, VERSION, APP_DATE);
	printf("Compiled by %s@%s on %s\n", build_user, build_host, build_date);
} /* print_version */


static void
print_appinfo(void)
{
	printf("Compile/Link info:\n");
	printf("\tGTK version   = %s\n", gtkver);
	printf("\tGNOME version = %s\n", gnomever);
	printf("\tCC            = %s\n", compiler);
	printf("\tGNOME_CFLAGS  = %s\n", gno_cflags);
	printf("\tGNOME_LDFLAGS = %s\n", gno_ldflags);
	printf("\tGNOME_LIBS    = %s\n", gno_libs);
	printf("\tCFLAGS        = %s\n", cflags);
	printf("\tINCLUDES      = %s\n", includes);
	printf("\tLDFLAGS       = %s\n", ldflags);
	printf("\tLIBS          = %s\n", libs);
	printf("\tUNAME         = %s\n", unamenfo);
} /* print_appinfo */


static void
ef_show_vers(void)
{
	print_version();
	exit(0);
} /* ef_show_vers */


static void
ef_show_info(void)
{
	print_version();
	print_appinfo();
	exit(0);
} /* ef_show_info */


/* the end */
