/*
 * Copyright (C) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define NDEBUG

#include "compiler.h"
CODE16;

#include "assert.h"
#include "stdio.h"

#include "main.h"
#include "io.h"
#include "const.h"
#include "var.h"
#include "ptrace.h"

#include "vesa.h"

/* FIXME VOSSI */
#define PA_VIDEO_MEMORY_HIGH	0x40000000

#define VBE_SUCCESS	0x004f
#define VBE_ERROR	0x0000

/* Mode attributes (VBEModeInfoBlock) */
#define VBE_MA_HWSUPPORT	0x0001	/* mode supported by HW */
#define VBE_MA_BIOSTTY		0x0004	/* tty functions supported by BIOS */
#define VBE_MA_COLOR		0x0008  /* color or monochrome mode */
#define VBE_MA_GRAPHICS		0x0010	/* graphics or text mode */
#define VBE_MA_VGACOMP		0x0020	/* VGA compatible mode */
#define VBE_MA_VGAWIN		0x0040	/* VGA compatible windowed memory mode*/
#define VBE_MA_LINEARFB		0x0080	/* linear frame buffer mode */
#define VBE_MA_DSCAN		0x0100	/* double scan mode available */
#define VBE_MA_INTERLACE	0x0200	/* interlaced mode available */
#define VBE_MA_TRIPBUF		0x0400	/* triple buffering support */
#define VBE_MA_STEREO		0x0800	/* stereo display support */
#define VBE_MA_DUALSA		0x1000	/* dual display start address support */

typedef struct {
	char VbeSignature[4];		/* 0000 */
	unsigned short VbeVersion;	/* 0004 */
	unsigned short OemStringOff;	/* 0006 */
	unsigned short OemStringSeg;	/* 0008 */
	unsigned short Capabilities0;	/* 0010 */
	unsigned short Capabilities1;	/* 0012 */
	unsigned short VideoModeOff;	/* 0014 */
	unsigned short VideoModeSeg;	/* 0016 */
	unsigned short TotalMemory;	/* 0018 */
	unsigned short OemSoftwareRev;	/* 0020 */
	unsigned short OemVendorNameOff;/* 0022 */
	unsigned short OemVendorNameSeg;/* 0024 */
	unsigned short OemProductNameOff;/* 0026 */
	unsigned short OemProductNameSeg;/* 0028 */
	unsigned short OemProductRevOff;/* 0030 */
	unsigned short OemProductRevSeg;/* 0032 */
#if 0
	char Reserved[222];		/* 0034 */
	char OemData[256];		/* 0256 */
#endif
} PACKED VBEInfoBlock;

/* sizes of ModeInfoBlock structure depending on VBE version */
#define VBE_MIBSIZE_1_0	18
#define VBE_MIBSIZE_1_2	40
#define VBE_MIBSIZE_2_0	50
#define VBE_MIBSIZE_3_0 253
/* values look odd - perhaps wrong? FIXME VOSSI */

typedef struct {
  unsigned short	ModeAttributes;		/* 0000 */
  unsigned char		WinAAttributes;		/* 0002 */
  unsigned char		WinBAttributes;		/* 0003 */
  unsigned short	WinGranularity;		/* 0004 */
  unsigned short	WinSize;		/* 0006 */
  unsigned short	WinASegment;		/* 0008 */
  unsigned short	WinBSegment;		/* 0010 */
  unsigned long		WinFuncPtr;		/* 0012 */
  unsigned short	BytesPerScanLine;	/* 0016 */

  /* Mandatory for VBE >= 1.2 */
  unsigned short	XResolution;		/* 0018 */
  unsigned short	YResolution;		/* 0020 */
  unsigned char		XCharSize;		/* 0022 */
  unsigned char		YCharSize;		/* 0023 */
  unsigned char		NumberOfPlanes;		/* 0024 */
  unsigned char		BitsPerPixel;		/* 0025 */
  unsigned char		NumberOfBanks;		/* 0026 */
  unsigned char		MemoryModel;		/* 0027 */
  unsigned char		BankSize;		/* 0028 */
  unsigned char		NumberOfImagePages;	/* 0029 */
  unsigned char		Reserved;		/* 0030 */

  /* Direct color fields */
  unsigned char		RedMaskSize;		/* 0031 */
  unsigned char		RedFieldPosition;	/* 0032 */
  unsigned char		GreenMaskSize;		/* 0033 */
  unsigned char		GreenFieldPosition;	/* 0034 */
  unsigned char		BlueMaskSize;		/* 0035 */
  unsigned char		BlueFieldPosition;	/* 0036 */
  unsigned char		RsvdMaskSize;		/* 0037 */
  unsigned char		RsvdFieldPosition;	/* 0038 */
  unsigned char		DirectColorModeInfo;	/* 0039 */

  /* Mandatory for VBE >= 2.0 */
  unsigned long		PhysBasePtr;		/* 0040 */
  unsigned long		Reserved1;		/* 0044 */
  unsigned short	Reserved2;		/* 0048 */

  /* Mandatory for VBE >= 3.0 */
  unsigned short	LinBytesPerScanLine;	/* 0050 */
  unsigned char		BnkNumberOfImagePages;	/* 0052 */
  unsigned char		LinNumberOfImagePages;	/* 0053 */
  unsigned char		LinRedMaskSize;		/* 0054 */
  unsigned char		LinRedFieldPosition;	/* 0055 */
  unsigned char		LinGreenMaskSize;	/* 0056 */
  unsigned char		LinGreenFieldPosition;	/* 0057 */
  unsigned char		LinBlueMaskSize;	/* 0058 */
  unsigned char		LinBlueMaskPosition;	/* 0059 */
  unsigned char		LinRsvdMaskSize;	/* 0060 */
  unsigned char		LinRsvdMaskPosition;	/* 0061 */
  unsigned long		MaxPixelClock;		/* 0062 */

  /* char		Reserved3[189];		   0064 */
} PACKED VBEModeInfoBlock;

static CONST unsigned short modelist[] = {
	0x0101, 0x0103, 0x0105, 0x0107,		/* 8-bit */
	0x0111, 0x0114, 0x0117, 0x011A,		/* 16-bit */
	0x0112, 0x0115, 0x0118, 0x011B,		/* 32-bit */
	0xffff
};

static CONST VBEInfoBlock iblock = {
	{ 'V','E','S','A' },
	0x0300,		/* 3.0 */
	0x0000, 0x0000,	/* OemString */
	0x0000, 0x0000,	/* Capabilities */
	0x0000, 0x0000,	/* modelist */
	0x0100,		/* 256 * 64k = 16 Mb */
	0x0000,		/* OemSoftwareRev */
	0x0000, 0x0000,	/* OemVendorName */
	0x0000, 0x0000,	/* OemProductName */
	0x0000, 0x0000	/* OemProductRev */
};

static CONST VBEModeInfoBlock miblock [] = {
	/* Mode: 0x0101: 640 x 480 x 8 */
	{
		0x039f, 7, 0, 64, 64, 0xa000, 0x0000, 0, 640, 
		640, 480, 8, 16, 1, 8, 1, 4, 0, 11, 1, 
		0, 0, 0, 0, 0, 0, 0, 0, 0, PA_VIDEO_MEMORY_HIGH, 0, 0, 
		640, 11, 11, 0, 0, 0, 0, 0, 0, 0, 0, 229500000
	},
	/* Mode: 0x0103: 800 x 600 x 8 */
	{
		0x039f, 7, 0, 64, 64, 0xa000, 0x0000, 0, 800, 
		800, 600, 8, 16, 1, 8, 1, 4, 0, 7, 1, 
		0, 0, 0, 0, 0, 0, 0, 0, 0, PA_VIDEO_MEMORY_HIGH, 0, 0, 
		800, 7, 7, 0, 0, 0, 0, 0, 0, 0, 0, 229500000
	},
	/* Mode: 0x0105: 1024 x 768 x 8 */
	{
		0x039f, 7, 0, 64, 64, 0xa000, 0x0000, 0, 1024, 
		1024, 768, 8, 16, 1, 8, 1, 4, 0, 4, 1, 
		0, 0, 0, 0, 0, 0, 0, 0, 0, PA_VIDEO_MEMORY_HIGH, 0, 0, 
		1024, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 229500000
	},
	/* Mode: 0x0107: 1280 x 1024 x 8 */
	{
		0x039f, 7, 0, 64, 64, 0xa000, 0x0000, 0, 1280, 
		1280, 1024, 8, 16, 1, 8, 1, 4, 0, 2, 1, 
		0, 0, 0, 0, 0, 0, 0, 0, 0, PA_VIDEO_MEMORY_HIGH, 0, 0, 
		1280, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 229500000
	},

	/* Mode: 0x0111: 640 x 480 x 16 */
	{
		0x039f, 7, 0, 64, 64, 0xa000, 0x0000, 0, 1280, 
		640, 480, 8, 16, 1, 16, 1, 6, 0, 5, 1, 
		5, 11, 6, 5, 5, 0, 0, 0, 0, PA_VIDEO_MEMORY_HIGH, 0, 0, 
		1280, 5, 5, 5, 11, 6, 5, 5, 0, 0, 0, 229500000
	},
	/* Mode: 0x0114: 800 x 600 x 16 */
	{
		0x039f, 7, 0, 64, 64, 0xa000, 0x0000, 0, 1600, 
		800, 600, 8, 16, 1, 16, 1, 6, 0, 5, 1, 
		5, 11, 6, 5, 5, 0, 0, 0, 0, PA_VIDEO_MEMORY_HIGH, 0, 0, 
		1600, 5, 5, 5, 11, 6, 5, 5, 0, 0, 0, 229500000
	},
	/* Mode: 0x0117: 1024 x 768 x 16 */
	{
		0x039f, 7, 0, 64, 64, 0xa000, 0x0000, 0, 2048, 
		1024, 768, 8, 16, 1, 16, 1, 6, 0, 5, 1, 
		5, 11, 6, 5, 5, 0, 0, 0, 0, PA_VIDEO_MEMORY_HIGH, 0, 0, 
		2048, 5, 5, 5, 11, 6, 5, 5, 0, 0, 0, 229500000
	},
	/* Mode: 0x011A: 1280 x 1024 x 16 */
	{
		0x039f, 7, 0, 64, 64, 0xa000, 0x0000, 0, 2560, 
		1280, 1024, 8, 16, 1, 16, 1, 6, 0, 5, 1, 
		5, 11, 6, 5, 5, 0, 0, 0, 0, PA_VIDEO_MEMORY_HIGH, 0, 0, 
		2560, 5, 5, 5, 11, 6, 5, 5, 0, 0, 0, 229500000
	},

	/* Mode: 0x0112: 640 x 480 x 32 */
	{
		0x039f, 7, 0, 64, 64, 0xa000, 0x0000, 0, 2560, 
		640, 480, 8, 16, 1, 32, 1, 6, 0, 2, 1, 
		8, 16, 8, 8, 8, 0, 8, 24, 0, PA_VIDEO_MEMORY_HIGH, 0, 0, 
		2560, 2, 2, 8, 16, 8, 8, 8, 0, 8, 24, 229500000
	},
	/* Mode: 0x0115: 800 x 600 x 32 */
	{
		0x039f, 7, 0, 64, 64, 0xa000, 0x0000, 0, 3200, 
		800, 600, 8, 16, 1, 32, 1, 6, 0, 2, 1, 
		8, 16, 8, 8, 8, 0, 8, 24, 0, PA_VIDEO_MEMORY_HIGH, 0, 0, 
		3200, 2, 2, 8, 16, 8, 8, 8, 0, 8, 24, 229500000
	},
	/* Mode: 0x0118: 1024 x 768 x 32 */
	{
		0x039f, 7, 0, 64, 64, 0xa000, 0x0000, 0, 4096, 
		1024, 768, 8, 16, 1, 32, 1, 6, 0, 2, 1, 
		8, 16, 8, 8, 8, 0, 8, 24, 0, PA_VIDEO_MEMORY_HIGH, 0, 0, 
		4096, 2, 2, 8, 16, 8, 8, 8, 0, 8, 24, 229500000
	},
	/* Mode: 0x011B: 1280 x 1024 x 32 */
	{
		0x039f, 7, 0, 64, 64, 0xa000, 0x0000, 0, 5120, 
		1280, 1024, 8, 16, 1, 32, 1, 6, 0, 2, 1, 
		8, 16, 8, 8, 8, 0, 8, 24, 0, PA_VIDEO_MEMORY_HIGH, 0, 0, 
		5120, 2, 2, 8, 16, 8, 8, 8, 0, 8, 24, 229500000
	}
};

static CONST char edidblock[128] = {
	/* EDID Header */
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
	/* Vendor Information */
	0x05, 0xb4, /* Manufacturer ID */
	0x30, 0x55, /* Product Code */
	0xbb, 0x02, 0x00, 0x00, /* Serial No. */
	0x29, 0x09, /* Week, Year-1990 */
	/* EDID Structure Versioning */
	0x01, 0x01, /* Version, Revision */
	/* Basic Display Parameters */
	0x1e, /* Video Input Definition: analog */
	36, 27, /* X,Y Size in cm */
	0x78, 0xe8, /* Gamma, Supported Features */
	/* Color Characteristics */
	0x07, 0xb3, 0xa0, 0x57, 0x4a, 0x9a, 0x26, 0x11, 0x48, 0x4f,
	/* Established Timings */
	0xbc, 0xe7, 0x80,
	/* Standard Timing Identification */
	/* First byte
	 *   Horizontal resolution.  Multiply by 8, then add 248 for actual value.
	 * Second byte
	 *   bit 7-6: Aspect ratio.  00=16:10, 01=4:3, 10=5:4, 11=16:9
	 *   bit 5-0: Vertical frequeny.  Add 60 to get actual value.
	 */
	0xa9, 0x4f, /* 1600x1200 75 Hz */
	0xa9, 0x4a, /* 1600x1200 70 Hz */
	0x61, 0x59, /* 1024x768 85 Hz */
	0x31, 0x59, /* 640x480 85 Hz */
	0x45, 0x59, /* 800x600 85 Hz */
	0x31, 0x4a, /* 640x480 70 Hz */
	0xa9, 0x45, /* 1600x1200 65 Hz */
	0x01, 0x01, /* unused */
	/* Detailed Timing Description #1 */
	0x1a, 0x4f, 0x40, 0x30, 0x62, 0xb0, 0x32, 0x40, 0x40, 0xc0,
	0x13, 0x00, 0x8e, 0x2b, 0x11, 0x00, 0x00, 0x1e,
	/* Detailed Timing Description #2 */
	0xd4, 0x49, 0x40, 0x30, 0x62, 0xb0, 0x32, 0x40, 0x40, 0xc0,
	0x13, 0x00, 0x8e, 0x2b, 0x11, 0x00, 0x00, 0x1e,
	/* Detailed Timing Description #3 */
	0xea, 0x24, 0x00, 0x60, 0x41, 0x00, 0x28, 0x30, 0x30, 0x60,
	0x13, 0x00, 0x8e, 0x2b, 0x11, 0x00, 0x00, 0x1e,
	/* Detailed Timing Description #4 */
	0x40, 0x1f, 0x00, 0x30, 0x41, 0x00, 0x24, 0x30, 0x20, 0x60,
	0x33, 0x00, 0x8e, 0x2b, 0x11, 0x00, 0x00, 0x02,
	/* Extension Flag */
	0x00,
	/* Checksum: 1-byte sum of all 128 bytes in block must be 0 */
	0xc3
};

/*
 * VESA Function 00: Return VBE Controller Information
 *
 * In:	AX	= 4f00h
 * 	ES:DI	= Ptr to VBEInfoBlock struct
 *
 * Out:	AX	= VBE return status
 */
static void
bios_10_4f00(struct regs *regs)
{
	VBEInfoBlock *vbe_ib = (VBEInfoBlock *) (unsigned long) DI;
	CONST unsigned char *src;
	unsigned short i;
	unsigned short modelistoff;
	unsigned short modelistseg;

	/* copy info block */
	src = (CONST unsigned char *) &iblock;
	for (i=0; i < sizeof(VBEInfoBlock); i++) { 
		put_byte(ES, (unsigned char *) (DI + i),
				get_byte(get_cs(), src + i));
	} 

	modelistoff = (unsigned short) (unsigned long) &modelist;
	modelistseg = get_cs();

	put_word(ES, &vbe_ib->VideoModeOff, modelistoff);
	put_word(ES, &vbe_ib->VideoModeSeg, modelistseg);

	AX = VBE_SUCCESS;	/* Success */
}

/*
 * VESA Function 01: Return VBE mode information
 *
 * In:	AX	= 4f01h
 * 	CX	= mode number
 *	ES:DI	= ptr to VBEModeInfoBlock struct
 *
 * Out:	AX	= VBE return status
 */
static void
bios_10_4f01(struct regs *regs)
{
	unsigned short index;
	unsigned short i;
	CONST unsigned char *src;

	for (index = 0; const_get(modelist[index]) != 0xffff; index++) {
		if (const_get(modelist[index]) == CX) {
			/* found */
			/* copy mode information structure */
			src = (CONST unsigned char *) &miblock[index];
			for (i = 0; i < sizeof(VBEModeInfoBlock); i++)
				put_byte(ES, (unsigned char *) (DI + i),
						get_byte(get_cs(), src + i));

			AX = VBE_SUCCESS;	/* Success */
			return;
		}
	}

	/* not found */
	AX = VBE_ERROR;	/* Error */
	return;
}

/*
 * VESA Function 02: Set VBE mode
 *
 * In:	AX	= 4f02h
 *	BX	= VBE mode
		bits:
 *		0-8	mode number
 *		9-10	reserved (must be 0)
 *		11	refresh rate (0=current,1=user)
 *		12-13	reserved (must be 0)
 *		14	frame buffer model (0=windowed,1=linear)
 *		15	clear display memory (0=yes,1=no)
 * 	ES:DI	= ptr to VBECRTCInfoBlock struct
 *
 * Out:	AX	= VBE return status
 */
static void
bios_10_4f02(struct regs *regs)
{
	unsigned short index;

	/* special: non VESA modes */
	if (BL <= 0x13 && ! (BH & 0x01)) {
		/* FIXME VOSSI */
		outw(BL, 0x03de);
		outw(0, 0x03dc);
		set_mode(BL);
		AX = VBE_SUCCESS;	/* Success */
		return;
	}

	for (index = 0; ; index++) {
		if (const_get(modelist[index]) == 0xffff) {
			/*
			 * No such mode.
			 */
			AX = VBE_ERROR;
			break;
		}
		if (const_get(modelist[index]) == (BX & 0x1ff)) {
			/*
			 * Mode found.
			 */
			unsigned short pages;
			unsigned short page;

			var_put(vbe_mode, BX);

			/* FIXME VOSSI */
			outw(BX & 0x1ff, 0x03de);

			if (((BX >> 15) & 1) == 0) {
				/*
				 * Clear display memory.
				 */
				/* calculate necessary number of pages - FIXME MARCEL */
				pages = ((unsigned long) const_get(miblock[index].XResolution)
					* (unsigned long) const_get(miblock[index].YResolution)
					* (unsigned long) const_get(miblock[index].BitsPerPixel)
					/ 8 + 65535UL) / 65536UL;

				for (page = 0; page < pages; page++) {
					unsigned long i;

					outw(page, 0x03dc);
					/* assume 64k pages - FIXME MARCEL */
					for (i = 0; i < 0x10000UL; i += 2) {
						put_word(0xa000, (unsigned short *) i, 0);
					}
				}
			}

			AX = VBE_SUCCESS;	/* Success */
			break;
		}
	}
}

/*
 * VESA Function 03: Get current video mode
 *
 * In:	-
 *
 * Out:	AX	= VBE return status
 * 	BX	= bit 14: linear mode
 * 		  bit 13: accelerated mode
 * 		  bit 12-0: video mode
 */
static void
bios_10_4f03(struct regs *regs)
{
	BX = var_get(vbe_mode);
	
	AX = VBE_SUCCESS;
}

/*
 * VESA Function 04: Save/restore state
 *
 * In:	DL	= 0: Get size of state buffer
 * 		  1: Save video state
 * 		  2: Restore video state
 *	ES:BX	= Pointer to video state buffer
 *	CX	= bit3: SuperVGA register state
 *		  bit2: DAC state
 *		  bit1: BIOS data state
 *		  bit0: Video hardware state
 *
 * Out:	BX	= # of 64byte blocks needed for saving state
 * 	AX	= VBE return status
 */
static void
bios_10_4f04(struct regs *regs)
{
	if (DL == 0) {
		/* Get size of state buffer. */
		BX = 1;
	} else if (DL == 1) {
		/* Save state to buffer. */
		/* FIXME VOSSI */
		dprintf("WARNING: VESA save state called.\n");
	} else if (DL == 2) {
		/* Restore state from buffer. */
		/* FIXME VOSSI */
		dprintf("WARNING: VESA restore state called.\n");
	}

	AX = VBE_SUCCESS;
}

/*
 * VESA Function 09: Get/set DAC entries
 *
 * In:	BL	= Bit7: Set palette during vertical retrace
 * 		  Bit6-0: 0: Set primary palette
 * 		          1: Get primary palette
 * 		          2: Set secondary palette
 * 		          3: Get secondary palette
 *	CX	= # of entries to change
 *	DX	= first palette index
 *	ES:DI	= array of colors
 *
 * Out: AX	= VBE return status
 */
static void
bios_10_4f09(struct regs *regs)
{
	unsigned short vert;
	unsigned short op;

	vert = (BL >> 7) & 1;
	op = BL & ~(1 << 7);

	if (op == 0) {
		/* FIXME VOSSI */
		dprintf("WARNING: VESA set primary DAC entries called.\n");

	} else if (op == 1) {
		/* FIXME VOSSI */
		dprintf("WARNING: VESA get primary DAC entries called.\n");

	} else if (op == 2) {
		/* FIXME VOSSI */
		dprintf("WARNING: VESA set secondary DAC entries called.\n");

	} else if (op == 3) {
		/* FIXME VOSSI */
		dprintf("WARNING: VESA get secondary DAC entries called.\n");

	} else {
		dprintf("WARNING: VESA BIOS: Unknown 4f09 subfunction %d.\n",
				BL);
		AX = VBE_ERROR;
		return;
	}

	AX = VBE_SUCCESS;
}

/*
 * VESA Function 05: Display window control
 *
 * In:	AX	= 4f05h
 *	BH	= 00h	set memory window
 *		= 01h	get memory window
 *	BL	= window #
 *	DX	= window # in window granularity units (only if BH==0)
 *
 * Out:	AX	= VBE return status
 *	DX	= window # in window granularity units (only if BH==1)
 */
static void
bios_10_4f05(struct regs *regs)
{
	if (BH == 0x00) {
		/*
		 * Set memory window.
		 */
		outw(DX, 0x03dc);

	} else {
		/*
		 * Get memory window.
		 */
		assert(0);	/* FIXME MARCEL */
	}

	AX = VBE_SUCCESS;
}

/*
 * VESA Function 0a: Return VBE protected mode interface
 *
 * In:	AX	= 4f0ah
 *	BL	= 00h
 *
 * Out:	AX	= VBE return status
 *	ES	= real mode segment of table
 *	DI	= offset of table
 *	CX	= length of table
 */
static void
bios_10_4f0a(struct regs *regs)
{
	AX = VBE_ERROR;	/* No protected mode interface - FIXME VOSSI */
}

/*
 * VESA Function 15: Report DDC Capabilities / Read EDID
 *
 * In:	AX	= 4f15h
 *	BL	= 00h	report DDC capabilities
 *		= 01h	read EDID sub-function
 *	CX	= controller unit (=0)
 *	ES:DI	= NULL (if BL==0)
 *		= ptr to EDID block (if BL==1)
 *
 * Out:	AX	= VBE return status
 *	BH	= time to transfer 1 EDID block (if BL==0)
 *	BL	= DDC level supported (if BL==0)
 */
static void
bios_10_4f15(struct regs *regs)
{
	unsigned char *src;
	unsigned short i;

	if (BL == 0x00) {	/* report DDC capabilities */
		BH = 0;
		BL = 0x02;	/* DDC 2 supported, no screen blank during data transfer */
		AX = VBE_SUCCESS;
	} else if (BL == 0x01) {	/* get EDID info */
		src = (unsigned char *)&edidblock;
		for (i = 0; i < sizeof(edidblock); i++) { 
			put_byte(ES, (unsigned char *) (DI + i),
				get_byte(get_cs(), (unsigned char *)(src + i)));
		} 
		AX = VBE_SUCCESS;
	} else
		AX = VBE_ERROR;
}

/*
 * VESA dispatcher
 */
void
bios_10_4fxx(struct regs *regs)
{
	if (AL == 0x00) {
		/* Return VBE controller information */
		bios_10_4f00(regs);

	} else if (AL == 0x01) {
		/* Get mode information structure */
		bios_10_4f01(regs);

	} else if (AL == 0x02) {
		/* VESA BIOS mode set call */
		bios_10_4f02(regs);

	} else if (AL == 0x03) {
		/* VESA BIOS mode get call */
		bios_10_4f03(regs);

	} else if (AL == 0x04) {
		/* VESA BIOS save/restore state call */
		bios_10_4f04(regs);

	} else if (AL == 0x05) {
		/* display window control - FIXME MARCEL */
		bios_10_4f05(regs);

	} else if (AL == 0x09) {
		/* VESA BIOS get/set DAC entries */
		bios_10_4f09(regs);

	} else if (AX == 0x0a) {
		/* get protected mode interface informations */
		bios_10_4f0a(regs);

	} else if (AL == 0x15) {
		/* report DDC capabilities / read EDID */
		bios_10_4f15(regs);

	} else {
		dprintf("int $0x%02x: unknown sub-function 0x%04x\n", 0x10, AX);

		AX = VBE_ERROR;
	}
}

/*
 * Dummy functions only implemented by cirrus' vga bios
 */
void
cirrus_10_12xx_extensions(struct regs *regs)
{
	/* do nothing */
}
void
cirrus_extension_init(void)
{
	/* do nothing */
}
void
cirrus_set_mode(unsigned int mode)
{
        dprintf("UMVGA-BIOS: unknown set mode %02x\n", mode);
}
