/*
 * Copyright (C) 2007-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __GLUE_SCHEDULER_H_INCLUDED
#define __GLUE_SCHEDULER_H_INCLUDED

#include <stdio.h> /* FILE */

#include "config.h"
#include "glue-main.h"

struct process {
	union {
		struct {
			void *sp; /* Must be first! Hardcoded in glue-scheduler.c! */
			int state; /* 0: sleeping; others: running */
			uint64_t inst_cnt;
			uint64_t inst_limit;
			uint64_t inst_hz;
			void *tos;
			void (*f)(void *);
		};
		char stack[0x10000];
	};
};

/** Init a new process.
 *  @param proc process control block.
 *  @param func function to be called in the newly created process context.
 *  @param param parameter passed to func on process startup.
 */
extern void
sched_process_init(struct process *proc, void (*func)(void *), void *param);

/** Return to scheduler.
 */
extern void
sched_to_scheduler(void);

/** Remove current process from run queue.
 */
extern void
sched_sleep(void);

/** Add process to run queue.
 *  @param p process to be added to run queue.
 */
extern void
sched_wakeup(struct process *p);

/** Delay execution for some time.
 *  @param delay delay time.
 */
extern void
sched_delay(unsigned long long delay);


/* glue-*-specific functions */


/** Set a function to be called after all processes ran and before the first
 *  process will run again. Different actions are executed based on the return
 *  value (see enum sched_action). list will point to the readylist
 *  of the scheduler, i.e. all processes on the list will be executed in
 *  the next round.
 *  @param func The function to be called.
 */
void
sched_set_timing_callback(int (*func)(struct process *list));

/** Start scheduling processes
 */
void
sched_go(void);

/** Initialize the scheduler
 */
void
sched_create(void);

/** De-initialize the scheduler
 */
void
sched_destroy(void);

/** Suspend scheduling status to a file
 *  @param fSched the file to write to
 */
void
sched_do_susped(FILE *fSched);

/** Resume scheduling status from a file
 *  @param fSched the file to read from
 */
int
sched_do_resume(FILE *fSched);

/**
 * Abort execution of all scheduled processes.
 */
void
sched_abort_processes(void);

#endif
