/*
 * Copyright (C) 2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <gtk/gtk.h>

#include "glue.h"
#include "glue-gui-gtk-sec-div-knob.h"
#include "glue-gui-gtk-control-knob.h"
#include "glue-gui-gtk-volt-div-knob.h"
#include "glue-gui-gtk-plus-minus-button.h"
#include "glue-gui-gtk-text-control.h"
#include "glue-gui-gtk.h"
#include "glue-gui-recorder.h"
#include "glue-gui-gtk-monitor.h"
#include "glue-gui-gtk-potentiometer.h"
#include "glue-gui-gtk-button.h"

#include "red_pitaya_gui_gtk.h"

#define COMP_(x)	red_pitaya_ ## x

#define SCREEN_HEIGHT 800
#define NUMBER_ANALOG_CHANNELS 5
#define NUMBER_DIGITAL_CHANNELS 5
#define NUMBER_SEC_DIV_ENTRIES 10
#define NUMBER_VOLT_DIV_ENTRIES 4
#define NUMBER_AMP_DIV_ENTRIES 4
#define NUMBER_SEC_DIV_ENTRIES_2 28
#define NUMBER_SAMPLE_RATE_ENTRIES 4

struct cpssp {
	GtkWidget *screen;
	void *recorder;
	GtkWidget *sec_div_knob;
	GtkWidget *sample_rate_knob;
	GtkWidget *trigger_poti;
	GtkWidget *channel_select_combo_box;
	GtkWidget *start_button;
	GtkWidget *stop_button;
	GtkWidget *scroll_right_button;
	GtkWidget *scroll_left_button;
	GtkWidget *export_button;
	GtkWidget *offset_button;
	GtkWidget *volt_div_button;
	GtkWidget *amp_div_button;
	GtkWidget *sec_div_button;
	GtkWidget *trigger_channel_knob;
	GtkWidget *trigger_edge_knob;
	GtkWidget *trigger_level_text_control;
	struct sig_integer *port_sample_rate_gui_sim;
	struct sig_integer *port_sec_div_gui_sim;
	struct sig_integer *port_sec_div_sim_gui;
	struct sig_integer *port_volt_div_gui_sim;
	struct sig_integer *port_volt_div_sim_gui;
	struct sig_integer *port_amp_div_gui_sim;
	struct sig_integer *port_amp_div_sim_gui;
	struct sig_integer *port_channel_select;
	struct sig_integer *port_trigger_sim_gui;
	struct sig_integer *port_trigger_gui_sim;
	struct sig_integer *port_vert_offset_sim_gui;
	struct sig_integer *port_vert_offset_gui_sim;
	struct sig_std_logic *port_start;
	struct sig_std_logic *port_stop;
	struct sig_std_logic *port_scroll_left;
	struct sig_std_logic *port_scroll_right;
	struct sig_std_logic *port_export;
	struct sig_integer *port_trigger_channel_gui_sim;
	struct sig_integer *port_trigger_channel_sim_gui;
	struct sig_integer *port_trigger_edge_gui_sim;
	struct sig_integer *port_trigger_edge_sim_gui;
	struct sig_integer *port_trigger_level_gui_sim;
	struct sig_integer *port_trigger_level_sim_gui;
};

static void
COMP_(gui_gtk_pixel_set)(
	void *_cpssp,
	unsigned int x,
	unsigned int y,
	uint8_t r,
	uint8_t g,
	uint8_t b
)
{
	struct cpssp *cpssp = _cpssp;

	gui_gtk_monitor_pixel_set(GUI_GTK_MONITOR(cpssp->screen), x, y, r, g, b);
	gui_recorder_pixel_set(cpssp->recorder, x, y, r, g, b);
}

static void
COMP_(gui_gtk_size_set)(void *_cpssp, unsigned int width, unsigned int height)
{
	struct cpssp *cpssp = _cpssp;

	gui_recorder_size_set(cpssp->recorder, width, height);
	gui_gtk_monitor_size_set(GUI_GTK_MONITOR(cpssp->screen), width, height);
}

static void
COMP_(gui_gtk_sync)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	gui_gtk_monitor_sync(GUI_GTK_MONITOR(cpssp->screen));
}

static void
sec_div_button_value_changed_sim(void *_cpssp, int val)
{
	/* val in ns */
	struct cpssp *cpssp = _cpssp;
	char val_to_string[15];
	char suffix[3];
	unsigned int val_u;
	val_u = (unsigned int) val;
	GtkWidget *sec_div_button;

	sec_div_button = cpssp->sec_div_button;

	if (1e9 < val_u) {
		val_u /= 1e9;
		suffix[0] = 's';
		suffix[1] = '\0';
	} else if (1e6 < val_u) {
		val_u /= 1e6;
		suffix[0] = 'm';
		suffix[1] = 's';
		suffix[2] = '\0';
	} else if (1e3 < val_u) {
		val_u /= 1e3;
		suffix[0] = 'u';
		suffix[1] = 's';
		suffix[2] = '\0';
	} else {
		suffix[0] = 'n';
		suffix[1] = 's';
		suffix[2] = '\0';
	}
	sprintf(val_to_string, "%u ", val_u);
	strcat(val_to_string, suffix);

	gui_gtk_plus_minus_button_set_value(sec_div_button, val_to_string);
}

static void
volt_div_button_value_changed_sim(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;
	GtkWidget *button;
	char val_to_string[10];
	char suffix[3];

	button = cpssp->volt_div_button;

	sprintf(val_to_string, "%d ", val);
	suffix[0] = 'm';
	suffix[1] = 'V';
	suffix[2] = '\0';
	strcat(val_to_string, suffix);

	gui_gtk_plus_minus_button_set_value(button, val_to_string);

}

static void
amp_div_button_value_changed_sim(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;
	GtkWidget *button;
	char val_to_string[10];
	char suffix[3];

	button = cpssp->amp_div_button;

	sprintf(val_to_string, "%d ", val);
	suffix[0] = 'm';
	suffix[1] = 'A';
	suffix[2] = '\0';
	strcat(val_to_string, suffix);

	gui_gtk_plus_minus_button_set_value(button, val_to_string);
}

static void
trigger_poti_value_changed_sim(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	gui_gtk_potentiometer_adj_sim(cpssp->trigger_poti, val);

}

static void
trigger_poti_value_changed(GtkWidget *w, void *_cpssp)
{

	struct cpssp *cpssp = _cpssp;
	GuiGtkPotentiometer *poti;
	int val;

	poti = (GuiGtkPotentiometer*)w;
	val = (int) poti->adj->value;
	sig_integer_set(cpssp->port_trigger_gui_sim, cpssp, val);
}

static void
trigger_channel_value_changed_sim(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;
	GuiGtkControlKnob *knob;
	char channel_name[10];

	knob = (GuiGtkControlKnob*) cpssp->trigger_channel_knob;
	if (val <= NUMBER_ANALOG_CHANNELS) {
		sprintf(channel_name, "CH%d\0", val);
	} else {
		sprintf(channel_name, "D%d\0", val - NUMBER_ANALOG_CHANNELS);
	}
	gui_gtk_control_knob_set_value(knob, channel_name);
}

static void
trigger_edge_value_changed_sim(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;
	GuiGtkControlKnob *knob;

	knob = (GuiGtkControlKnob*)cpssp->trigger_edge_knob;
	if (val == 0) {
		gui_gtk_control_knob_set_value(knob, "NE");
	} else {
		gui_gtk_control_knob_set_value(knob, "PE");
	}
}

static void
trigger_level_value_changed_sim(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;
	char level_string[10];

	sprintf(level_string, "%d", val);
	gui_gtk_text_control_set_value(cpssp->trigger_level_text_control,
			level_string);
}

static void
sample_rate_value_changed_sim(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;
	GuiGtkControlKnob *knob;
	char val_to_string[10];

	knob = (GuiGtkControlKnob*) cpssp->sample_rate_knob;

	if (1000*1000*1000 < val ) {
		/* GHz */
		sprintf(val_to_string, "%d GHz", val / (1000*1000*1000));
	} else if (1000*1000 < val) {
		/* MHz */
		sprintf(val_to_string, "%d MHz", val / (1000*1000));
	} else if (1000 < val) {
		/* kHz */
		sprintf(val_to_string, "%d KHz", val / (1000));
	}

	gui_gtk_control_knob_set_value(knob, val_to_string);
	/* TODO */
}

static void
vertical_offset_value_changed_sim(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;
	GuiGtkPlusMinusButton *button;
	char val_to_string[10];

	button = (GuiGtkPlusMinusButton *) cpssp->offset_button;

	sprintf(val_to_string, "%d", val);

	gui_gtk_plus_minus_button_set_value(button, val_to_string);
}

static void
volt_div_button_value_changed(GtkWidget *w, void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
	GuiGtkPlusMinusButton *plus_minus_button;

	plus_minus_button = (GuiGtkPlusMinusButton*) w;

	char *s = plus_minus_button->items[plus_minus_button->current_item];
	char *end;
	long val = strtol(s, &end, 10);
	if (end	!= s) {
		sig_integer_set(cpssp->port_volt_div_gui_sim, _cpssp, (int) val);
	}
}

static void
amp_div_button_value_changed(GtkWidget *w, void *_cpssp)
{
	GuiGtkPlusMinusButton *plus_minus_button;
	struct cpssp *cpssp;
	plus_minus_button = (GuiGtkPlusMinusButton*)w;
	cpssp = _cpssp;

	char *s = plus_minus_button->items[plus_minus_button->current_item];
	char *end;
	long val = strtol(s, &end, 10);
	if (end	!= s) {
		sig_integer_set(cpssp->port_amp_div_gui_sim, _cpssp, (int)val);
	}

}
static void
sec_div_button_value_changed(GtkWidget *w, void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
	GuiGtkPlusMinusButton *plus_minus_button;

	plus_minus_button = (GuiGtkPlusMinusButton*) w;
	char *s = plus_minus_button->items[plus_minus_button->current_item];
	unsigned int value;
	int length = strlen(s);
	char number[length];
	char unit[length];

	memset(number, '\0', length);
	memset(unit, '\0', length);
	int char_pos = 0;
	int unit_pos = 0;
	char c;

	while (char_pos < length) {
		c = s[char_pos];
		if ('0' <= c && c <= '9') {
			number[char_pos] = c;
		} else if (c == ' ') {

		} else {
			unit[unit_pos++] = c;
		}
		char_pos++;
	}

	char *end;
	long val = strtol(number, &end, 10);
	if (end	!= number) {
		value = (unsigned int) val;
		/* values are always transmitted in ns */
		if (strcmp(unit, "s") == 0) {
			value *= 1e9;
		} else if (strcmp(unit, "ms") == 0) {
			value *= 1e6;
		} else if (strcmp(unit, "us") == 0) {
			value *= 1e3;
		} else if (strcmp(unit, "ns") == 0) {
			/* TODO potential problem with overflow */
		}
		sig_integer_set(cpssp->port_sec_div_gui_sim, _cpssp, value);
	}
}

static void
vertical_offset_button_value_changed(GtkWidget *w, void *_cpssp)
{
	GuiGtkPlusMinusButton *plus_minus_button;
	struct cpssp *cpssp;
	plus_minus_button = (GuiGtkPlusMinusButton*)w;
	cpssp = _cpssp;

	char *s = plus_minus_button->items[plus_minus_button->current_item];
	char *end;
	long val = strtol(s, &end, 10);
	if (end	!= s) {
		sig_integer_set(cpssp->port_vert_offset_gui_sim, _cpssp, (int)val);
	}
}

static void
channel_select_combo_box_value_changed(GtkWidget *w, void *_cpssp)
{
	GuiGtkControlKnob *channel_select_combo_box = (GuiGtkControlKnob*) w;
	struct cpssp *cpssp = _cpssp;
	char *val_text;
	long val;
	char *end;

	val_text = gtk_combo_box_text_get_active_text(
			GTK_COMBO_BOX_TEXT(channel_select_combo_box->combo_box));
	if (val_text[0] == 'C'
	 && val_text[1] == 'H') {
		val = strtol(val_text + 2, &end, 10);
		sig_integer_set(cpssp->port_channel_select, cpssp, (int) val);

	} else if (val_text[0] == 'D') {
		val = strtol(val_text + 1, &end, 10);
		sig_integer_set(cpssp->port_channel_select, cpssp,
				(int) val + NUMBER_ANALOG_CHANNELS);
	}
}

static void
trigger_edge_knob_value_changed(GtkWidget *w, void *_cpssp)
{
	GuiGtkControlKnob *trigger_edge_knob = (GuiGtkControlKnob*) w;
	struct cpssp *cpssp = _cpssp;
	char *val_text;

	val_text = gtk_combo_box_text_get_active_text(
			GTK_COMBO_BOX_TEXT(trigger_edge_knob->combo_box));
	if (val_text[0] == 'P'
	 && val_text[1] == 'E') {
		sig_integer_set(cpssp->port_trigger_edge_gui_sim, cpssp, 1);

	} else if (val_text[0] == 'N'
		&& val_text[1] == 'E') {
		sig_integer_set(cpssp->port_trigger_edge_gui_sim, cpssp, 0);
	}
}

static void
trigger_channel_knob_value_changed(GtkWidget *w, void *_cpssp)
{
	GuiGtkControlKnob *trigger_channel_knob = (GuiGtkControlKnob *) w;
	struct cpssp *cpssp = _cpssp;
	char *val_text;
	long val;
	char *end;

	val_text = gtk_combo_box_text_get_active_text(
			GTK_COMBO_BOX_TEXT(trigger_channel_knob->combo_box));
	if (val_text[0] == 'C'
	 && val_text[1] == 'H') {
		val = strtol(val_text + 2, &end, 10);
		sig_integer_set(cpssp->port_trigger_channel_gui_sim, cpssp,
				(int) val - 1);

	} else if (val_text[0] == 'D') {
		val = strtol(val_text + 1, &end, 10);
		sig_integer_set(cpssp->port_trigger_channel_gui_sim, cpssp,
				(int)val + NUMBER_ANALOG_CHANNELS - 1);
	}
}

static void
trigger_level_text_control_value_changed(GtkWidget *w, void *_cpssp)
{
	GuiGtkTextControl *text_control = (GuiGtkTextControl *) w;
	struct cpssp *cpssp = _cpssp;
	char *level_string;
	int level;
	char *end;

	level_string = (char*) gtk_entry_get_text(GTK_ENTRY(text_control->entry));
	level = (int) strtol(level_string, &end, 10);
	if (level_string != end
	 && strlen(end) == 0) {
		sig_integer_set(cpssp->port_trigger_level_gui_sim, cpssp, level);
		gui_gtk_text_control_set_value(text_control, level_string);
	} else {
		gui_gtk_text_control_set_value(text_control, "Error");
	}
}


static void
sample_rate_knob_value_changed(GtkWidget *w, void *_cpssp)
{
	GuiGtkControlKnob *sample_rate_knob = (GuiGtkControlKnob *) w;
	struct cpssp *cpssp = _cpssp;
	char *val_text;

	val_text = gtk_combo_box_text_get_active_text(
			GTK_COMBO_BOX_TEXT(sample_rate_knob->combo_box));

	int length = strlen(val_text);
	char number[length];
	char unit[length];
	memset(number, '\0', length);
	memset(unit, '\0', length);
	int char_pos = 0;
	int unit_pos = 0;
	char c;
	while (char_pos < length) {
		c = val_text[char_pos];
		if ('0' <= c && c <= '9') {
			number[char_pos] = c;
		} else if (c == ' ') {

		} else {
			unit[unit_pos++] = c;
		}
		char_pos++;
	}

	char *end;
	long val = strtol(number, &end, 10);
	unsigned int value;
	if (end	!= number) {
		value = (unsigned int)val;
		/* values are always transmitted in Hz */
		if (strcmp(unit, "GHz") == 0) {
			/* currently not needed */
			value *= 1e9;
		}
		else if (strcmp(unit, "MHz") == 0) {
			value *= 1e6;
		} else if (strcmp(unit, "kHz") == 0) {
			value *= 1e3;
		} else if (strcmp(unit, "Hz") == 0) {
			/* do nothing */
		}
		sig_integer_set(cpssp->port_sample_rate_gui_sim, _cpssp, value);
	}
}


static void
scroll_left_button_clicked(GtkWidget *w, void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	sig_std_logic_set(cpssp->port_scroll_left, cpssp, SIG_STD_LOGIC_1);
	sig_std_logic_set(cpssp->port_scroll_left, cpssp, SIG_STD_LOGIC_0);
}

static void
scroll_right_button_clicked(GtkWidget *w, void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	sig_std_logic_set(cpssp->port_scroll_right, cpssp, SIG_STD_LOGIC_1);
	sig_std_logic_set(cpssp->port_scroll_right, cpssp, SIG_STD_LOGIC_0);
}

static void
start_button_clicked(GtkWidget *w, void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	sig_std_logic_set(cpssp->port_start, cpssp, SIG_STD_LOGIC_1);
	sig_std_logic_set(cpssp->port_start, cpssp, SIG_STD_LOGIC_0);
}

static void
stop_button_clicked(GtkWidget *w, void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	sig_std_logic_set(cpssp->port_stop, cpssp, SIG_STD_LOGIC_1);
	sig_std_logic_set(cpssp->port_stop, cpssp, SIG_STD_LOGIC_0);
}

static void
export_button_clicked(GtkWidget *w, void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	sig_std_logic_set(cpssp->port_export, cpssp, SIG_STD_LOGIC_1);
	sig_std_logic_set(cpssp->port_export, cpssp, SIG_STD_LOGIC_0);
}

static void
monitor_gui_screenshot_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	if (val == 0) {
		return;
	}

	gui_recorder_screenshot(cpssp->recorder);
}

#if 0
static void
monitor_gui_rec_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	gui_recorder_rec_set(cpssp->recorder, val);
	gui_gtk_monitor_rec_set(GUI_GTK_MONITOR(cpssp->screen), (gboolean) val);
	gui_gtk_flush();
}
#endif

/*
 * GUI Callbacks
 */
static void
monitor_gui_recording_on_event(GtkWidget *w, gpointer _cpssp)
{
	struct cpssp * cpssp = _cpssp;

	gui_recorder_rec_set(cpssp->recorder, 1);
}

static void
monitor_gui_recording_off_event(GtkWidget *w, gpointer _cpssp)
{
	struct cpssp * cpssp = _cpssp;

	gui_recorder_rec_set(cpssp->recorder, 0);
}

static void
monitor_gui_screenshot_pressed(GtkWidget *w, gpointer _cpssp)
{
	struct cpssp * cpssp = _cpssp;

	monitor_gui_screenshot_set(cpssp, 1);
}

static void
monitor_gui_screenshot_released(GtkWidget *w, gpointer _cpssp)
{
	struct cpssp * cpssp = _cpssp;

	monitor_gui_screenshot_set(cpssp, 0);
}

void *
COMP_(gui_gtk_create)(
	unsigned int page,
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_voltage_1,
	struct sig_std_logic *port_voltage_2,
	struct sig_std_logic *port_voltage_3,
	struct sig_std_logic *port_voltage_4,
	struct sig_std_logic *port_voltage_5,
	struct sig_std_logic *port_voltage_6,
	struct sig_std_logic *port_voltage_7,
	struct sig_std_logic *port_voltage_8,
	struct sig_std_logic *port_voltage_9,
	struct sig_std_logic *port_voltage_10,
	struct sig_std_logic *port_digi_1,
	struct sig_std_logic *port_digi_2,
	struct sig_std_logic *port_digi_3,
	struct sig_std_logic *port_digi_4,
	struct sig_std_logic *port_digi_5,
	struct sig_std_logic *port_digi_6,
	struct sig_std_logic *port_digi_7,
	struct sig_std_logic *port_digi_8,
	struct sig_std_logic *port_digi_9,
	struct sig_std_logic *port_digi_10,
	struct sig_integer *port_sample_rate_gui_sim,
	struct sig_integer *port_sample_rate_sim_gui,
	struct sig_integer *port_sec_div_gui_sim,
	struct sig_integer *port_sec_div_sim_gui,
	struct sig_integer *port_volt_div_gui_sim,
	struct sig_integer *port_volt_div_sim_gui,
	struct sig_integer *port_channel_select,
	struct sig_integer *port_trigger_sim_gui,
	struct sig_integer *port_trigger_gui_sim,
	struct sig_integer *port_vert_offset_sim_gui,
	struct sig_integer *port_vert_offset_gui_sim,
	struct sig_opt_rgb *port_monitor,
	struct sig_std_logic *port_vcc,
	struct sig_std_logic *port_gnd,
	struct sig_std_logic *port_start,
	struct sig_std_logic *port_stop,
	struct sig_std_logic *port_scroll_left,
	struct sig_std_logic *port_scroll_right,
	struct sig_std_logic *port_export,
	struct sig_integer *port_trigger_channel_gui_sim,
	struct sig_integer *port_trigger_channel_sim_gui,
	struct sig_integer *port_trigger_edge_gui_sim,
	struct sig_integer *port_trigger_edge_sim_gui,
	struct sig_integer *port_trigger_level_gui_sim,
	struct sig_integer *port_trigger_level_sim_gui,
	struct sig_integer *port_amp_div_gui_sim,
	struct sig_integer *port_amp_div_sim_gui
	)
{
	static const char *volt_div_entries[NUMBER_VOLT_DIV_ENTRIES] = {
		"100 mV", "1000 mV", "2500 mV", "5000 mV"
	};
	static const char *amp_div_entries[NUMBER_AMP_DIV_ENTRIES] = {
		"100 mA", "1000 mA", "2500 mA", "5000 mA"
	};
	static const char *sample_rate_entries[NUMBER_SAMPLE_RATE_ENTRIES] = {
		"125 MHz", "2 MHz", "1 MHz", "500 kHz"
	};
	static const char *channel_names[NUMBER_ANALOG_CHANNELS + NUMBER_DIGITAL_CHANNELS] = {
		"CH1", "CH2", "CH3", "CH4", "CH5", "D1", "D2", "D3", "D4", "D5"
	};
	static const char *trigger_edges[2] = {
		"PE", "NE"
	};
	static const char *sec_div_values[NUMBER_SEC_DIV_ENTRIES_2] = {
		"4 ns",
		"10 ns", "20 ns", "50 ns",
		"100 ns", "200 ns", "500 ns",
		"1 us", "2 us", "5 us",
		"10 us", "20 us", "50 us",
		"100 us", "200 us", "500 us",
		"1 ms", "2 ms", "5 ms",
		"10 ms", "20 ms", "50 ms",
		"100 ms", "200 ms", "500 ms",
		"1 s", "2 s", "4 s"
	};

	struct cpssp *cpssp;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *scrollbox;

	static const struct sig_opt_rgb_funcs opt_video_funcs = {
		.pixel_set = COMP_(gui_gtk_pixel_set),
		.size_set = COMP_(gui_gtk_size_set),
		.sync = COMP_(gui_gtk_sync),
	};
	static const struct sig_integer_funcs sec_div_sim_gui_func = {
		.set = sec_div_button_value_changed_sim,
	};
	static const struct sig_integer_funcs volt_div_sim_gui_func = {
		.set = volt_div_button_value_changed_sim,
	};
	static const struct sig_integer_funcs amp_div_sim_gui_func = {
		.set = amp_div_button_value_changed_sim,
	};
	static const struct sig_integer_funcs trigger_sim_gui_func = {
		.set = trigger_poti_value_changed_sim,
	};
	static const struct sig_integer_funcs vert_offset_sim_gui_func = {
		.set = vertical_offset_value_changed_sim,
	};
	static const struct sig_integer_funcs sample_rate_sim_gui_func = {
		.set = sample_rate_value_changed_sim,
	};
	static const struct sig_integer_funcs trigger_channel_sim_gui_func = {
		.set = trigger_channel_value_changed_sim,
	};
	static const struct sig_integer_funcs trigger_edge_sim_gui_func = {
		.set = trigger_edge_value_changed_sim,
	};
	static const struct sig_integer_funcs trigger_level_sim_gui_func = {
		.set = trigger_level_value_changed_sim,
	};

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	/* assign ports */
	cpssp->port_sample_rate_gui_sim = port_sample_rate_gui_sim;
	cpssp->port_sec_div_gui_sim = port_sec_div_gui_sim;
	cpssp->port_volt_div_gui_sim = port_volt_div_gui_sim;
	cpssp->port_amp_div_gui_sim = port_amp_div_gui_sim;
	cpssp->port_channel_select = port_channel_select;
	cpssp->port_trigger_sim_gui = port_trigger_sim_gui;
	cpssp->port_trigger_gui_sim = port_trigger_gui_sim;
	cpssp->port_vert_offset_sim_gui = port_vert_offset_sim_gui;
	cpssp->port_vert_offset_gui_sim = port_vert_offset_gui_sim;
	cpssp->port_start = port_start;
	cpssp->port_stop = port_stop;
	cpssp->port_scroll_left = port_scroll_left;
	cpssp->port_scroll_right = port_scroll_right;
	cpssp->port_export = port_export;
	cpssp->port_trigger_channel_gui_sim = port_trigger_channel_gui_sim;
	cpssp->port_trigger_channel_sim_gui = port_trigger_channel_sim_gui;
	cpssp->port_trigger_edge_gui_sim = port_trigger_edge_gui_sim;
	cpssp->port_trigger_edge_sim_gui = port_trigger_edge_sim_gui;
	cpssp->port_trigger_level_gui_sim = port_trigger_level_gui_sim;
	cpssp->port_trigger_level_sim_gui = port_trigger_level_sim_gui;
	/* create sub-components */
	cpssp->recorder = gui_recorder_create(1600, 1200);
	cpssp->screen = gui_gtk_monitor_new("VGA", 1600, 1200);
	cpssp->sample_rate_knob =
		gui_gtk_control_knob_new("sampling rate",
			NUMBER_SAMPLE_RATE_ENTRIES,
			sample_rate_entries);
	cpssp->trigger_channel_knob =
		gui_gtk_control_knob_new("trigger channel",
			NUMBER_ANALOG_CHANNELS + NUMBER_DIGITAL_CHANNELS,
			channel_names);
	cpssp->trigger_edge_knob =
		gui_gtk_control_knob_new("trigger edge", 2,trigger_edges);
	cpssp->trigger_level_text_control =
		gui_gtk_text_control_new("trigger level");
	cpssp->trigger_poti = gui_gtk_potentiometer_new("trigger\t\t");
	cpssp->start_button = gui_gtk_button_new("START");
	cpssp->stop_button = gui_gtk_button_new("STOP");
	cpssp->scroll_left_button = gui_gtk_button_new("<<");
	cpssp->scroll_right_button = gui_gtk_button_new(">>");
	cpssp->export_button = gui_gtk_button_new("EXPORT");
	int i;
	int item_count = 0;
	char *offset_values[SCREEN_HEIGHT/5 + 1];
	for (i = 0; i < SCREEN_HEIGHT + 1; i+=5) {
		char buf[5];
		memset(buf, '\0', 5);

		sprintf(buf, "%d", i - (SCREEN_HEIGHT/2));
		offset_values[item_count++] = strdup(buf);
	}
	cpssp->offset_button = gui_gtk_plus_minus_button_new("Offset",
 						SCREEN_HEIGHT/5 + 1,
						(const char**)offset_values);
	cpssp->sec_div_button = gui_gtk_plus_minus_button_new("ms_div",
						NUMBER_SEC_DIV_ENTRIES_2,
						sec_div_values);
	cpssp->volt_div_button = gui_gtk_plus_minus_button_new("volt_div",
						NUMBER_VOLT_DIV_ENTRIES,
						volt_div_entries);
	cpssp->amp_div_button = gui_gtk_plus_minus_button_new("amp_div",
						NUMBER_AMP_DIV_ENTRIES,
						amp_div_entries);
	cpssp->channel_select_combo_box =
			gui_gtk_control_knob_new("channels\t\t",
			NUMBER_ANALOG_CHANNELS + NUMBER_DIGITAL_CHANNELS,
			channel_names);
	/* connect callback functions */
	g_signal_connect(G_OBJECT(cpssp->sample_rate_knob),
			"value_changed_gui",
			G_CALLBACK(sample_rate_knob_value_changed),
			cpssp);

	g_signal_connect(G_OBJECT(cpssp->offset_button),
			"value_changed_gui",
			G_CALLBACK(vertical_offset_button_value_changed),
			cpssp);

	g_signal_connect(G_OBJECT(cpssp->sec_div_button),
			"value_changed_gui",
			G_CALLBACK(sec_div_button_value_changed),
			cpssp);

	g_signal_connect(G_OBJECT(cpssp->trigger_poti),
			"value_changed_gui",
			G_CALLBACK(trigger_poti_value_changed),
			cpssp);

	g_signal_connect(G_OBJECT(cpssp->volt_div_button),
			"value_changed_gui",
			G_CALLBACK(volt_div_button_value_changed),
 			cpssp);

	g_signal_connect(G_OBJECT(cpssp->amp_div_button),
		"value_changed_gui",
		G_CALLBACK(amp_div_button_value_changed),
		cpssp);

	g_signal_connect(G_OBJECT(cpssp->channel_select_combo_box),
			"value_changed_gui",
			G_CALLBACK(channel_select_combo_box_value_changed),
			cpssp);

	g_signal_connect(G_OBJECT(cpssp->trigger_channel_knob),
			"value_changed_gui",
			G_CALLBACK(trigger_channel_knob_value_changed),
			cpssp);

	g_signal_connect(G_OBJECT(cpssp->trigger_edge_knob),
			"value_changed_gui",
			G_CALLBACK(trigger_edge_knob_value_changed),
			cpssp);
	g_signal_connect(G_OBJECT(cpssp->trigger_level_text_control),
			"value_changed_gui",
			G_CALLBACK(trigger_level_text_control_value_changed),
			cpssp);
	g_signal_connect(G_OBJECT(cpssp->screen),
			"recording-on",
			G_CALLBACK(monitor_gui_recording_on_event),
			cpssp);

	g_signal_connect(G_OBJECT(cpssp->screen),
			"recording-off",
			G_CALLBACK(monitor_gui_recording_off_event),
			cpssp);

	g_signal_connect(G_OBJECT(cpssp->screen),
			"screenshot-pressed",
			G_CALLBACK(monitor_gui_screenshot_pressed),
			cpssp);

	g_signal_connect(G_OBJECT(cpssp->screen),
			"screenshot-released",
			G_CALLBACK(monitor_gui_screenshot_released),
			cpssp);

	g_signal_connect(G_OBJECT(cpssp->start_button),
			"button-pressed",
			G_CALLBACK(start_button_clicked), cpssp);

	g_signal_connect(G_OBJECT(cpssp->stop_button),
			"button-pressed",
			G_CALLBACK(stop_button_clicked), cpssp);

	g_signal_connect(G_OBJECT(cpssp->scroll_left_button),
			"button-pressed",
			G_CALLBACK(scroll_left_button_clicked), cpssp);

	g_signal_connect(G_OBJECT(cpssp->scroll_right_button),
			"button-pressed",
			G_CALLBACK(scroll_right_button_clicked), cpssp);

	g_signal_connect(G_OBJECT(cpssp->export_button),
			"button-pressed",
			G_CALLBACK(export_button_clicked), cpssp);

	/* connect ports */
	sig_integer_connect_out(port_sample_rate_gui_sim, cpssp, 0);
	sig_integer_connect_out(port_sec_div_gui_sim, cpssp, 0);
	sig_integer_connect_out(port_vert_offset_gui_sim, cpssp, 0);
	sig_integer_connect_out(port_trigger_gui_sim, cpssp, 0);
	sig_integer_connect_out(port_volt_div_gui_sim, cpssp, 0);
	sig_integer_connect_out(port_amp_div_gui_sim, cpssp, 0);
	sig_integer_connect_out(port_channel_select, cpssp, 0);
	sig_integer_connect_in(port_sample_rate_sim_gui,
				cpssp,
				&sample_rate_sim_gui_func);
	sig_integer_connect_in(port_sec_div_sim_gui,
				cpssp,
				&sec_div_sim_gui_func);
	sig_integer_connect_in(port_volt_div_sim_gui,
				cpssp,
				&volt_div_sim_gui_func);
	sig_integer_connect_in(port_amp_div_sim_gui,
				cpssp,
				&amp_div_sim_gui_func);
	sig_integer_connect_in(port_trigger_sim_gui,
				cpssp,
				&trigger_sim_gui_func);
	sig_integer_connect_in(port_vert_offset_sim_gui,
				cpssp,
				&vert_offset_sim_gui_func);
	sig_integer_connect_in(port_trigger_channel_sim_gui,
				cpssp,
				&trigger_channel_sim_gui_func);
	sig_integer_connect_in(port_trigger_edge_sim_gui,
				cpssp,
				&trigger_edge_sim_gui_func);
	sig_integer_connect_in(port_trigger_level_sim_gui,
				cpssp,
				&trigger_level_sim_gui_func);
	sig_integer_connect_out(port_trigger_channel_gui_sim,
				cpssp,
				0);
	sig_integer_connect_out(port_trigger_edge_gui_sim,
				cpssp,
				0);
	sig_integer_connect_out(port_trigger_level_gui_sim,
				cpssp,
				0);

	sig_std_logic_connect_out(port_start, cpssp, SIG_STD_LOGIC_0);
	sig_std_logic_connect_out(port_stop, cpssp, SIG_STD_LOGIC_0);
	sig_std_logic_connect_out(port_scroll_left, cpssp, SIG_STD_LOGIC_0);
	sig_std_logic_connect_out(port_scroll_right, cpssp, SIG_STD_LOGIC_0);
	sig_std_logic_connect_out(port_export, cpssp, SIG_STD_LOGIC_0);
	/* get initial values*/
	/*sec_div_knob_value_changed(cpssp->sec_div_knob, cpssp);*/
	sample_rate_knob_value_changed(cpssp->sample_rate_knob, cpssp);
	vertical_offset_button_value_changed(cpssp->offset_button,cpssp);
	/* visually organize sub-components */
	vbox = gtk_vbox_new(FALSE, 1);
	hbox = gtk_hbox_new(FALSE, 1);
	scrollbox = gtk_hbox_new(FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), cpssp->sample_rate_knob, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), cpssp->trigger_channel_knob, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), cpssp->trigger_edge_knob, FALSE, FALSE, 1);
	/*gtk_box_pack_start(GTK_BOX(vbox), cpssp->trigger_level_text_control, FALSE, FALSE, 1);*/
	gtk_box_pack_start(GTK_BOX(vbox), cpssp->channel_select_combo_box, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), cpssp->volt_div_button, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), cpssp->amp_div_button, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), cpssp->sec_div_button, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), cpssp->offset_button, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), cpssp->start_button, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), cpssp->stop_button, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), cpssp->export_button, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(scrollbox), cpssp->scroll_left_button, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(scrollbox), cpssp->scroll_right_button, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), scrollbox, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 1);
	gtk_box_pack_start(GTK_BOX(hbox), cpssp->screen, TRUE, TRUE, 1);

	gtk_widget_show(cpssp->screen);
	gtk_widget_show(cpssp->start_button);
	gtk_widget_show(cpssp->stop_button);
	gtk_widget_show(cpssp->scroll_left_button);
	gtk_widget_show(cpssp->scroll_right_button);
	gtk_widget_show(cpssp->export_button);
	gtk_widget_show(cpssp->offset_button);
	gtk_widget_show(cpssp->sec_div_button);
	gtk_widget_show(cpssp->volt_div_button);
	gtk_widget_show(cpssp->amp_div_button);
	gtk_widget_show(cpssp->trigger_channel_knob);
	gtk_widget_show(cpssp->trigger_edge_knob);
	/*gtk_widget_show(cpssp->trigger_level_text_control);*/
	gtk_widget_show(scrollbox);
	gtk_widget_show(vbox);
	gtk_widget_show(hbox);
	gui_gtk_monitor_sync(GUI_GTK_MONITOR(cpssp->screen));
	gui_gtk_monitor_grab_focus(GUI_GTK_MONITOR(cpssp->screen));
	gui_gtk_comp_add(page, "COMP", name, hbox, TRUE, TRUE, NULL);

	sig_opt_rgb_connect(port_monitor, cpssp, &opt_video_funcs);
	channel_select_combo_box_value_changed(cpssp->channel_select_combo_box, cpssp);
	trigger_channel_knob_value_changed(cpssp->trigger_channel_knob, cpssp);
	trigger_edge_knob_value_changed(cpssp->trigger_edge_knob, cpssp);
	return cpssp;

}

void
COMP_(gui_gtk_destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	gui_gtk_plus_minus_button_destroy(cpssp->sec_div_button);
	gui_gtk_plus_minus_button_destroy(cpssp->offset_button);
	gui_gtk_plus_minus_button_destroy(cpssp->volt_div_button);
	gui_gtk_plus_minus_button_destroy(cpssp->amp_div_button);
	gui_gtk_control_knob_destroy(cpssp->sample_rate_knob);
	gui_gtk_control_knob_destroy(cpssp->channel_select_combo_box);

	free(cpssp);
}

void
COMP_(gui_gtk_suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	gui_gtk_monitor_suspend(cpssp->screen, fp);
}

void
COMP_(gui_gtk_resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	gui_gtk_monitor_resume(cpssp->screen, fp);
}
