/* Copyright (C) 2005 to 2007 and 2009 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

*/

#include <c++-gtk-utils/lib_defs.h>

#include <c++-gtk-utils/io_watch.h>
#include <c++-gtk-utils/emitter.h>
#include <c++-gtk-utils/thread.h>


struct WatchSource {
  GSource source;
  GPollFD poll_fd;
  GIOCondition io_condition;
  Cgu::SafeEmitterArg<bool&>* emitter_p;
};

extern "C" {
  gboolean io_watch_prepare_func(GSource*, gint*);
  gboolean io_watch_check_func(GSource*);
  gboolean io_watch_dispatch_func(GSource*, GSourceFunc, void*);
  void io_watch_finalize_func(GSource*);
}

namespace { // callbacks for internal use only

gboolean io_watch_prepare_func(GSource*, gint* timeout_p) {

  *timeout_p = -1;
  return false; // we want the file descriptor to be polled
}

gboolean io_watch_check_func(GSource* source_p) {
  // reinterpret_cast<>() is guaranteed to give the correct result here as the
  // address of WatchSource::source must be the same as the address of the
  // instance of the WatchSource struct of which it is the first member as
  // both are PODSs
  WatchSource* watch_source_p = reinterpret_cast<WatchSource*>(source_p);

  // what we have got
  gushort poll_condition = watch_source_p->poll_fd.revents;
  // what we are looking for
  gushort watch_condition = watch_source_p->io_condition;

  // return true if we have what we are looking for
  return (poll_condition & watch_condition); 
}

gboolean io_watch_dispatch_func(GSource* source_p, GSourceFunc, void*) {
  // reinterpret_cast<>() is guaranteed to give the correct result here as the
  // address of WatchSource::source must be the same as the address of the
  // instance of the WatchSource struct of which it is the first member as
  // both are PODSs
  WatchSource* watch_source_p = reinterpret_cast<WatchSource*>(source_p);

  // we are not interested in the GSourceFunc argument here as we have never
  // called g_source_set_callback()
  bool keep_source = true;
  bool connected;
  // provide a CancelBlock here to make this function NPTL friendly,
  // as we have a catch-all without rethrowing
  Cgu::Thread::CancelBlock b;
  try {
    connected = watch_source_p->emitter_p->test_emit(keep_source);
  }
  // we can't propagate exceptions from functions with C linkage.
  catch (...) {
    g_critical("Exception thrown in io_watch_dispatch_func()\n");
    return true;
  }
  return (keep_source && connected);
}

void io_watch_finalize_func(GSource* source_p) {

  // reinterpret_cast<>() is guaranteed to give the correct result here as the
  // address of WatchSource::source must be the same as the address of the
  // instance of the WatchSource struct of which it is the first member as
  // both are PODSs
  WatchSource* watch_source_p = reinterpret_cast<WatchSource*>(source_p);
  
  delete watch_source_p->emitter_p;
  watch_source_p->emitter_p = 0;
}

GSourceFuncs io_watch_source_funcs = {
  io_watch_prepare_func,
  io_watch_check_func,
  io_watch_dispatch_func,
  io_watch_finalize_func
};

} // anonymous namespace

namespace Cgu {

guint start_iowatch(int fd, const Callback::CallbackArg<bool&>* cb,
		    GIOCondition io_condition, gint priority,
		    GMainContext* context_p) {

  // context_p has a default value of NULL which will create the watch
  // in the default program main context

  Callback::SafeFunctorArg<bool&> f(cb);   // take ownership

  GSource* source_p = g_source_new(&io_watch_source_funcs, sizeof(WatchSource));
  // reinterpret_cast<>() is guaranteed to give the correct result here as the
  // address of WatchSource::source must be the same as the address of the
  // instance of the WatchSource struct of which it is the first member as
  // both are PODSs
  WatchSource* watch_source_p = reinterpret_cast<WatchSource*>(source_p);
  watch_source_p->poll_fd.fd = fd;
  watch_source_p->poll_fd.events = io_condition;
  watch_source_p->poll_fd.revents = 0;
  watch_source_p->io_condition = io_condition;
  watch_source_p->emitter_p = 0;
  try {
    watch_source_p->emitter_p = new SafeEmitterArg<bool&>;
    watch_source_p->emitter_p->connect(f);
  }
  catch (...) {
    delete watch_source_p->emitter_p; // either NULL or object allocated
    g_source_unref(source_p);
    throw;
  }

  g_source_set_priority(source_p, priority);

  // connect the source object to its polling object  
  g_source_add_poll(source_p, &watch_source_p->poll_fd);

  // attach the source to the relevant main context
  guint id = g_source_attach(source_p, context_p);

  // g_source_attach() will add a reference count to the GSource object
  // so we unreference it here so that the callback returning false or
  // calling g_source_remove() on the return value of this function will
  // finalize/destroy the GSource object
  g_source_unref(source_p);

  return id;
}

guint start_iowatch(int fd, const Callback::CallbackArg<bool&>* cb, Releaser& r,
		    GIOCondition io_condition, gint priority,
		    GMainContext* context_p) {

  // context_p has a default value of NULL which will create the watch
  // in the default program main context

  Callback::SafeFunctorArg<bool&> f(cb);   // take ownership

  GSource* source_p = g_source_new(&io_watch_source_funcs, sizeof(WatchSource));
  // reinterpret_cast<>() is guaranteed to give the correct result here as the
  // address of WatchSource::source must be the same as the address of the
  // instance of the WatchSource struct of which it is the first member as
  // both are PODSs
  WatchSource* watch_source_p = reinterpret_cast<WatchSource*>(source_p);
  watch_source_p->poll_fd.fd = fd;
  watch_source_p->poll_fd.events = io_condition;
  watch_source_p->poll_fd.revents = 0;
  watch_source_p->io_condition = io_condition;
  watch_source_p->emitter_p = 0;
  try {
    watch_source_p->emitter_p = new SafeEmitterArg<bool&>;
    watch_source_p->emitter_p->connect(f, r);
  }
  catch (...) {
    delete watch_source_p->emitter_p; // either NULL or object allocated
    g_source_unref(source_p);
    throw;
  }

  g_source_set_priority(source_p, priority);

  // connect the source object to its polling object  
  g_source_add_poll(source_p, &watch_source_p->poll_fd);

  // attach the source to the relevant main context
  guint id = g_source_attach(source_p, context_p);

  // g_source_attach() will add a reference count to the GSource object
  // so we unreference it here so that the callback returning false or
  // calling g_source_remove() on the return value of this function will
  // finalize/destroy the GSource object
  g_source_unref(source_p);

  return id;
}

} // namespace Cgu
