/* Copyright (C) 2005 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the src/utils sub-directory);
   if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

*/


#include "thread.h"
#include "sem_sync.h"

struct ThreadArgs {
  const sigc::slot<void>* slot_p;
  SemSync* sem_p;
};

namespace {
void* thread_func(void* arg) {

  // this is executing in the new thread
  ThreadArgs* thread_args_p = static_cast<ThreadArgs*>(arg);
  sigc::slot<void> callback = *(thread_args_p->slot_p);
  thread_args_p->sem_p->post();
  // at this point the pointers in pthread_args will cease to be valid
  // but that's OK as we now hold our own local copy of the slot and
  // we have finished with the semaphore
  callback();
  return 0;
}
} // anonymous namespace

namespace Thread {

std::auto_ptr<Thread> Thread::start(const sigc::slot<void>& cb, bool joinable) {

  // set joinable attribute
  int detach_state;
  if (joinable) detach_state = PTHREAD_CREATE_JOINABLE;
  else detach_state = PTHREAD_CREATE_DETACHED;
  pthread_attr_t attr;
  pthread_attr_init(&attr);
  pthread_attr_setdetachstate(&attr, detach_state);

  Thread* instance_p;
  try {
    instance_p = new Thread;
  }
  catch (...) {
    pthread_attr_destroy(&attr);
    throw;
  }
  std::auto_ptr<Thread> return_val(instance_p);

  ThreadArgs thread_args;
  SemSync sem;
  thread_args.slot_p = &cb;
  thread_args.sem_p = &sem;

  pthread_t thread;
  if (!pthread_create(&thread, &attr, ::thread_func, &thread_args)) {
    instance_p->thread = thread;
    sem.wait();
  }
  else {
    return_val.reset();
  }
  pthread_attr_destroy(&attr);
  return return_val;
}

CancelBlock::CancelBlock(bool blocking) {
  // the default value of blocking is true
  if (blocking) block();
  else unblock();
}

} // namespace Thread
