/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	VCD document and view class
 *
 *	by Tony Sideris	(11:57AM Dec 05, 2001)
 *================================================*/
#include "arson.h"

#include <qabstractlayout.h>
#include <qtoolbutton.h>
#include <qfileinfo.h>
#include <qlineedit.h>
#include <qlabel.h>
#include <qcheckbox.h>
#include <qlayout.h>
#include <qframe.h>
#include <qcombobox.h>

#include <kfiledialog.h>
#include <kstddirs.h>
#include <klocale.h>
#include <kmessagebox.h>

#include "tempfile.h"
#include "vcddoc.h"
#include "listwnd.h"
#include "process.h"
#include "mainwnd.h"

#define ARSONVCD_TEMPBASE		"vcd"

enum {
	VCDTYPE_VCD1 = 0,
	VCDTYPE_VCD2,
	VCDTYPE_SVCD,
	VCDTYPE_HQVCD,
	_VCDTYPE_MAX,
};

struct {
	QString desc;			//	Description
	const char *flag;		//	Command line switch to vcdxgen

}	g_vcd_types[_VCDTYPE_MAX] = {

	{ I18N_NOOP("VCD 1.1"), "vcd11" },
	{ I18N_NOOP("VCD 2.0 (default)"), "vcd2" },
	{ I18N_NOOP("SVCD"), "svcd" },
	{ I18N_NOOP("HQVCD"), "hqvcd" },	//	WTF is that ?!?
};

/*========================================================*/

class vcdProgress : public ArsonImgCreateProgress
{
public:
	vcdProgress (QWidget *parent, ArsonVcdDoc *pDoc);

private:
	virtual ArsonProcessMgr *createProcessMgr (void);
	virtual void processOpts (ArsonProcessOpts &opts);

	ArsonVcdDoc *m_pDoc;
	QComboBox *m_pVcdType;
};

void vcdProgress::processOpts (ArsonProcessOpts &opts)
{
	ArsonImgCreateProgress::processOpts(opts);

	opts.addString("type", g_vcd_types[m_pVcdType->currentItem()].flag);
}

ArsonProcessMgr *vcdProgress::createProcessMgr (void)
{
	return m_pDoc->createCdWriter(ui());
}

/*========================================================*/

vcdProgress::vcdProgress (QWidget *parent, ArsonVcdDoc *pDoc)
	: ArsonImgCreateProgress(parent, "vcddlg"), m_pDoc(pDoc)
{
	QLabel *pv = new QLabel(i18n("VCD Type:"), ctrlParent());

	pv->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Fixed));
	m_pVcdType = new ArsonProgressCombobox(ctrlParent(), "vcdtype");

	for (int index = 0; index < _VCDTYPE_MAX; ++index)
		m_pVcdType->insertItem(i18n(g_vcd_types[index].desc));

	m_pVcdType->setCurrentItem(VCDTYPE_VCD2);

	layoutRow() << pv << m_pVcdType;

	showLabelEdit();
	setIsoLabel(pDoc->rootText());
}

/*========================================================*/
/*	VCD document class implementation
 *========================================================*/

ArsonVcdDoc::ArsonVcdDoc (QWidget *parent, const char *name)
	: ArsonFileListDoc(parent, name)
{
	setRootItemInfo(
		new ArsonRootItemInfo(
			i18n("VideoCD"),
			"label",
			"cdwriter_unmount"));

	setRenameColumn(0);
}

/*========================================================*/

QString ArsonVcdDoc::propDocType (void) const
{
	return "vcd";
}

/*========================================================*/
/*	Factory functions
 *========================================================*/

class vcdListWnd : public ArsonFileListWnd
{
public:
	vcdListWnd (ArsonFileListDoc *parent)
		: ArsonFileListWnd(parent) { }

private:
	virtual void rename (QListViewItem *ptr, int col)
	{
		//	Only allow the root item to be edited
		if (!ptr->parent() && !col)
			ArsonFileListWnd::rename(ptr, col);
	}
};

ArsonListWnd *ArsonVcdDoc::createListWnd (void)
{
	return new vcdListWnd(this);
}

/*========================================================*/

ArsonProcessMgr *ArsonVcdDoc::createCdWriter (ArsonProcessUI *pUI)
{
	return new ArsonVcdWriter(pUI, this);
}

void ArsonVcdDoc::buildFileFilter (ArsonFileFilter &filter)
{
	filter.addFilter(i18n("*.mpg *.mpeg *.mpe|MPEG Files"));
}

ArsonProgress *ArsonVcdDoc::createProgress (QWidget *parent)
{
	return new vcdProgress(parent, this);
}

/*========================================================*/

void ArsonVcdDoc::addItem (ArsonListItem *ptr, ArsonLvPos *pp)
{
	KMessageBox::information(ArsonFrame::getFrame(),
		i18n("arson will create VCD image from the given MPEG files, but these files must already be in VCD format. arson performs no conversion on MPEG files. Continue?"),
		i18n("Warning"), "vcd-warning");

	ArsonFileListDoc::addItem(ptr, pp);
}

/*========================================================*/

namespace arson {
	class vcdFileItem : public ArsonFileListFileItem
	{
	public:
		vcdFileItem (const KURL &url) : ArsonFileListFileItem(url) { }

		virtual void refresh (QListViewItem *pi, ArsonDocWidget *pd)
		{
			static QPixmap pm;

			if (pm.isNull())
				pm = loadIcon("video");

			pi->setPixmap(0, pm);

			ArsonFileListFileItem::refresh(pi, pd);
		}
	};
};

ArsonFileListItem *ArsonVcdDoc::createFileItem (const KURL &url) const
{
	return new arson::vcdFileItem(url);
}

/*========================================================*/

void ArsonVcdDoc::create (void)
{
	static ArsonListHeader vcd_headers[] = {
		ArsonListHeader(i18n("Filename"), 85),
		ArsonListHeader(i18n("Size"), 15),
	};

	ArsonFileListDoc::create();

	listWnd()->setListHeaders(vcd_headers,
		ARRSIZE(vcd_headers));
}

/*========================================================*/
/*	VCD process manager class impl
 *========================================================*/

ArsonVcdWriter::ArsonVcdWriter (ArsonProcessUI *pUI, ArsonVcdDoc *pDoc)
	: ArsonCueWriter(pUI), m_pDoc(pDoc),
	m_bConverted(false), m_filebase(QString::null)
{
	//	Nothing...
}

ArsonVcdWriter::~ArsonVcdWriter (void)
{
	QFileInfo fi (QFile::encodeName(m_filebase + ".toc"));
	QDir dir = fi.dir();

	//	Remove all the temporary files created by vcdxbuild
	if (dir.exists())
	{
		const QString filter = fi.baseName() + "*";
		ArsonFileCleanup() << dir.entryList(filter);
	}
#ifdef ARSONDBG
	else { Trace("ERROR: dir doesn't exist! (%s)\n", (const char *) m_filebase); }
#endif	//	ARSONDBG
}

/*========================================================*/

class vcdVisitor : public ArsonFileListVisiter
{
public:
	vcdVisitor (ArsonVcdxbuildProcess *pb) : m_ptr(pb) { }

	virtual bool visitFile (QListViewItem *pi, ArsonFileListFileItem *pf)
	{
		m_ptr->addFile(pf->local());
		return true;
	}

private:
	ArsonVcdxbuildProcess *m_ptr;
};

void ArsonVcdWriter::begin (const ArsonProcessOpts &opts)
{
	m_filebase = ArsonTempFile::tempFileName(ARSONVCD_TEMPBASE, "bin");

	if (m_filebase != QString::null)
	{
		ArsonCueWriter::begin(opts);

		//	Build a toc/bin file using the specified mpeg files
		try {
			ArsonVcdxbuildProcess *ptr = new ArsonVcdxbuildProcess(this, m_filebase);
			vcdVisitor vv (ptr);

			//	Add each mpeg file to the new process
			m_pDoc->visitAll(vv);

			setProcess(ptr);
		}
		catch (ArsonError &err) {
			err.report();
		}
	}
#ifdef ARSONDBG
	else { Trace("ERROR: FILEBASE not set!\n"); }
#endif	//	ARSONDBG
}

/*========================================================*/

void ArsonVcdWriter::taskComplete (ArsonProcess *ptr)
{
	ArsonCueWriter::taskComplete(ptr);

	if (!m_bConverted)
	{
		//	Burn the new toc/bin to CD.
		m_bConverted = true;

		setCueFile(m_filebase + ".toc");
		writeCd();
	}
}

/*========================================================*/
