/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	A progress bar that can zoom back and forth
 *	knight-rider style to indicate a process of
 *	unknown length.
 *
 *	by Tony Sideris	(05:49PM Mar 18, 2002)
 *================================================*/
#include "arson.h"

#include <qdrawutil.h>
#include <qpalette.h>
#include <qpixmap.h>
#include <qtimer.h>
#include <qpainter.h>
#include <qstyle.h>

#include <kpixmapeffect.h>
#include <kpixmap.h>
#include <klocale.h>

#include "progressbar.h"


#define TS_TIMEOUT		75
#define TS_STEP			8

#define TS_EDGE			1
#define TS_EDGE2		2

#define IMGWIDTH(w)		((w) / 2)

/*========================================================*/

ArsonInfiniteProgressBar::ArsonInfiniteProgressBar (QWidget *parent, const char *name)
	: QProgressBar(parent, name),
	m_nMode(arsonProgressBlank),
	m_bDirLeft(false),
	m_nAniPos(0),
	m_pTimer(NULL)
{
	const QRect inner = getRect();
	m_nAniPos = -(IMGWIDTH(inner.width()));
}

/*========================================================*/

void ArsonInfiniteProgressBar::slotTimeout (void)
{
	const QRect inner = getRect();
	const int step = (int) ((double(inner.width()) * 0.75) / double(1000 / TS_TIMEOUT));

	if (!m_bDirLeft)
	{
		if (m_nAniPos <= inner.width())
			m_nAniPos += step;
		else
			m_bDirLeft = true;
	}
	else
	{
		if (m_nAniPos >= -(IMGWIDTH(inner.width())))
			m_nAniPos -= step;
		else
			m_bDirLeft = false;
	}

	repaint();
}

/*========================================================*/

void ArsonInfiniteProgressBar::createPixmaps (const QSize &size)
{
	if (infinite())
	{
		const QColor left = colorGroup().text();
		const QColor rght = colorGroup().background();

		m_pml.resize(IMGWIDTH(size.width() - TS_EDGE2),
			size.height() - TS_EDGE2);
		m_pmr.resize(IMGWIDTH(size.width() - TS_EDGE2),
			size.height() - TS_EDGE2);

		KPixmapEffect::gradient(m_pml, left, rght,
			KPixmapEffect::HorizontalGradient);
		KPixmapEffect::gradient(m_pmr, rght, left,
			KPixmapEffect::HorizontalGradient);
	}
	else
	{
		m_pml.resize(0,0);
		m_pmr.resize(0,0);
	}
}

/*========================================================*/

QRect ArsonInfiniteProgressBar::getRect (bool edge) const
{
	QRect rc = contentsRect();

	if (!edge)
		return QRect(rc.left() + TS_EDGE, rc.y() + TS_EDGE,
			rc.width() - TS_EDGE2, rc.height() - TS_EDGE2);

	return rc;
}

/*========================================================*/

void ArsonInfiniteProgressBar::setMode (int mode)
{
	if ((m_nMode = mode) == arsonProgressInfinite)
	{
		m_pTimer = new QTimer(this);

		QObject::connect(m_pTimer, SIGNAL(timeout()),
			this, SLOT(slotTimeout()));

		m_pTimer->start(TS_TIMEOUT, false);
	}
	else
	{
		delete m_pTimer;
		m_pTimer = NULL;
	}

	createPixmaps(size());
	repaint();
}

/*========================================================*/

void ArsonInfiniteProgressBar::drawContents (QPainter *ptr)
{
	if (m_nMode != arsonProgressNormal)
	{
		const QRect rc = getRect(true);
		const QRect inner = getRect();

#ifndef ARSON_KDE3
		style().drawPanel(ptr, rc.x(), rc.y(),
			rc.width(), rc.height(), colorGroup(), true, 1,
			&(colorGroup().brush(QColorGroup::Background)));
#else
		style().drawPrimitive(QStyle::PE_Panel,
			ptr, rc, colorGroup(),
			QStyle::Style_Default | QStyle::Style_Sunken,
			QStyleOption(1,0));
#endif	//	ARSON_KDE3

		if (m_nMode == arsonProgressInfinite)
		{
			int srcw = IMGWIDTH(inner.width());

			if (m_nAniPos < 0)	srcw += m_nAniPos;
			else if (m_nAniPos > srcw)
				srcw = inner.width() - m_nAniPos;

			ptr->drawPixmap(
				inner.x() + QMAX(m_nAniPos, 0),
				inner.y(),
				m_bDirLeft ? m_pml : m_pmr,
				(m_nAniPos < 0) ? (m_nAniPos * -1) : 0,
				0,
				srcw,
				inner.height());
		}
	}
	else	//	arsonProgressNormal: Default handling
		QProgressBar::drawContents(ptr);
}

/*========================================================*/

void ArsonInfiniteProgressBar::resizeEvent (QResizeEvent *ptr)
{
	createPixmaps(ptr->size());
	QProgressBar::resizeEvent(ptr);
}

/*========================================================*/
/*	Show disk usage
 *========================================================*/

#define BAR_EDGE		1
#define BAR_EDGES		2
#define BAR_MARK_HEIGHT	8

ArsonDiskUsageBar::ArsonDiskUsageBar (QWidget *parent, const char *name)
	: QFrame(parent, name),
	m_max(100), m_pos(0)
{
	init();
}

ArsonDiskUsageBar::ArsonDiskUsageBar (int max, int pos, QWidget *parent, const char *name)
	: QFrame(parent, name),
	m_max(max), m_pos(pos)
{
	init();
}

void ArsonDiskUsageBar::init (void)
{
	setSizePolicy(
		QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed));

//	setBackgroundMode(NoBackground);
	
	m_suffix = i18n("MB");

	m_color = colorGroup().mid();
	m_pTimer = NULL;
}

inline int barHeight (const QWidget *ptr)
{
	const QFontMetrics fm (ptr->fontMetrics());

	return (fm.height() - (fm.height() / 4)) - 2;
}

/*========================================================*/

QSize ArsonDiskUsageBar::minimumSizeHint (void) const
{
	return sizeHint();
}

QSize ArsonDiskUsageBar::sizeHint (void) const
{
	const QFontMetrics fm (fontMetrics());

	return QSize(1,
		fm.height() + fm.leading() + BAR_EDGES		//	Marks
		+ barHeight(this) + BAR_EDGES);				//	Progress
}

/*========================================================*/

ArsonDiskUsageBar::~ArsonDiskUsageBar (void)
{
	endTimer();
}

/*========================================================*/

void ArsonDiskUsageBar::setMax (int max)
{
	if (m_max != max)
	{
		m_max = max;
		checkTimer();
		repaint(subRect(subrectProgress));
	}
}

void ArsonDiskUsageBar::setPos (int p)
{
	if (m_pos != p)
	{
		m_pos = p;
		checkTimer();
		repaint(subRect(subrectProgress));
	}
}

/*========================================================*/

void ArsonDiskUsageBar::startTimer (void)
{
	if (!m_pTimer)
	{
		m_pTimer = new QTimer(this);
		QObject::connect(m_pTimer, SIGNAL(timeout()),
			this, SLOT(slotTimer()));

		m_pTimer->start(500);
	}
}

void ArsonDiskUsageBar::endTimer (void)
{
	if (m_pTimer)
	{
		delete m_pTimer;
		m_pTimer = NULL;

		m_color = colorGroup().mid();
		repaint(subRect(subrectProgress));
	}
}

/*========================================================*/

void ArsonDiskUsageBar::slotTimer (void)
{
	const QColorGroup cg (colorGroup());

	if (m_color == cg.mid())
		m_color = cg.dark();
	else
		m_color = cg.mid();

	repaint(subRect(subrectProgress));
}

/*========================================================*/

void ArsonDiskUsageBar::checkTimer (void)
{
	if (pos() > max())
		startTimer();
	else
		endTimer();
}

/*========================================================*/

QRect ArsonDiskUsageBar::subRect (int which) const
{
	const int bh = barHeight(this);
	
	switch (which)
	{
	case subrectProgress:
		return QRect(QPoint(BAR_EDGE, BAR_EDGE),
			QSize(width() - BAR_EDGES, bh));

	case subrectMarks:
		return QRect(QPoint(BAR_EDGE, bh + BAR_EDGES),
			QSize(width() - BAR_EDGES, height() - (bh + BAR_EDGES)));
	}

	Assert(false);
	return QRect();
}

/*========================================================*/

void ArsonDiskUsageBar::drawContents (QPainter *ptr)
{
	const QColorGroup cg (colorGroup());
	QBrush br (cg.button());

	qDrawShadePanel(ptr, 0, 0, width(),
		barHeight(this) + BAR_EDGES,
		cg, true, 1, &br);

	drawProgress(ptr, cg, subRect(subrectProgress));

	drawMarks(ptr, cg, subRect(subrectMarks));
}

/*========================================================*/

void ArsonDiskUsageBar::drawProgress (QPainter *ptr, const QColorGroup &cg, const QRect &rc)
{
	QBrush br (m_color);

	qDrawShadePanel(ptr, rc.left(), rc.top(),
		percentageOf(rc, pos()), rc.height(),
		cg, false, 1, &br);
}

/*========================================================*/
/*	Recursively draw the marks. The QPoint is actually
 *	the two bounding X coords, each call draws a mark
 *	in the center of these two points, then recurses
 *	each side.
 *========================================================*/

/*
void drawMark (QPainter *ptr, const QPoint &pt, int ypos, int lineh, int min)
{
	const int width = pt.y() - pt.x() + 1;

	if (width > min)
	{
		const int xpos = pt.x() + (width / 2);

		ptr->drawLine(xpos, ypos, xpos, ypos + lineh);

		if ((lineh /= 2) < 2)
			lineh = BAR_MARK_HEIGHT;

		drawMark(ptr, QPoint(pt.x(), xpos), ypos, lineh, min);
		drawMark(ptr, QPoint(xpos, pt.y()), ypos, lineh, min);
	}
}
*/

int ArsonDiskUsageBar::percentageOf (const QRect &rc, int pos) const
{
	const double perc = ((double(pos) / double(full())) * 100.0);
	return (int) (double(rc.width()) * (perc * 0.01));
}

void ArsonDiskUsageBar::drawMark (QPainter *ptr, const QRect &rc, int pos, bool right)
{
	const int xpos = percentageOf(rc, pos);
	const QString label (QString::number(m_mod ? (pos / m_mod) : pos) + " " + m_suffix);
	const QRect bounds (ptr->fontMetrics().boundingRect(label));
	int xt = xpos + 2, yt = rc.bottom() - bounds.height() + 3;

	ptr->drawLine(xpos, rc.top(), xpos, rc.bottom() - 4);

	if (!right)
		xt = xpos - bounds.width() - 2;
	
	ptr->drawText(xt, yt, label);
}

void ArsonDiskUsageBar::drawMarks (QPainter *ptr, const QColorGroup &cg, const QRect &rc)
{
	ptr->save();
	ptr->setPen(cg.text());

	ptr->drawLine(rc.left(), rc.top(), rc.right(), rc.top());

	drawMark(ptr, rc, 0, true);
	drawMark(ptr, rc, max());
	drawMark(ptr, rc, full());

/*
	drawMark(ptr, QPoint(rc.left(), rc.right()),
		rc.top(), BAR_MARK_HEIGHT / 2, rc.width() / 8);

	ptr->drawLine(rc.left(), rc.top(), rc.left(), rc.top() + BAR_MARK_HEIGHT);
	ptr->drawLine(rc.right(), rc.top(), rc.right(), rc.top() + BAR_MARK_HEIGHT);
*/
	ptr->restore();
}

/*========================================================*/
