#ifndef __LAYOUT_H__
#define __LAYOUT_H__

#include <qlayout.h>

/*========================================================*/
/**
 *	Simple wrapper for Qt's Bax Layouts so we
 *	get an operator<<.
 *
 *	Example:
 *	<pre>
 *	ArsonLayout lo (new QVBoxLayout(this, 11, 6));
 *	lo << new QLabel("Foo", this) << new QLabel("bar", this);
 *	lo << new QHBoxLayout(this, 0, 6);
 *	</pre>
 */

class ArsonLayout
{
public:
	ArsonLayout (QBoxLayout *pl) : m_pl(pl) { }

	ArsonLayout &operator<< (QWidget *pw)
	{ m_pl->addWidget(pw); return *this; }

	ArsonLayout &operator<< (QLayout *pl)
	{ m_pl->addLayout(pl); return *this; }

	ArsonLayout &operator<< (QLayoutItem *pi)
	{ m_pl->addItem(pi); return *this; }

	QBoxLayout *layout (void) const { return m_pl; }
	operator QBoxLayout* (void) const { return layout(); }

private:
	QBoxLayout *m_pl;
};

/*========================================================*/
/**
 *	A QGridLayout wrapper adding operator<<.
 *
 *	Example:
 *	<pre>
 *	ArsonGridLayout gl (new QGridLayout(this, 2, 2));
 *	gl.row(0) << new QLabel("foo", this) << new QLabel("bar", this);
 *	gl.row(1) << new QVBoxLayout(this, 11, 6) << new QYouGetTheIdea(this);
 *	</pre>
 */

class ArsonGridLayout
{
	class arow {
	public:
		arow (QGridLayout *pl, int r) : m_pl(pl), m_row(r), m_col(0) { }

		arow &operator<< (QWidget *pw)
		{ m_pl->addWidget(pw, m_row, m_col++); return *this; }

		arow &operator<< (QLayout *pl)
		{ m_pl->addLayout(pl, m_row, m_col++); return *this; }

		arow &operator<< (QLayoutItem *pi)
		{ m_pl->addItem(pi, m_row, m_col++); return *this; }

	private:
		QGridLayout *m_pl;
		const int m_row;
		int m_col;
	};

public:
	ArsonGridLayout (QGridLayout *pl) : m_pl(pl) { }

	arow row (int r) { return arow(m_pl, r); }

	QGridLayout *layout (void) const { return m_pl; }
	operator QGridLayout* (void) const { return layout(); }
	
private:
	QGridLayout *m_pl;
};

/*========================================================*/
#endif	/*	__LAYOUT_H__	*/
