/*
 * Copyright (c) 2001 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __DOCWIDGET_H__
#define __DOCWIDGET_H__

#include <qvbox.h>
#include <qmap.h>

#include <kstdaction.h>
#include <kurl.h>

#include "utils.h"

/*========================================================*/

class KActionCollection;

/**
 *	Wraps a KActionCollection to allow for docwidgets
 *	to easily enable/disable actions. This is also
 *	the base for ArsonActionConnector which is passed
 *	to ArsonDocWidget::connectTo which allows the
 *	current docwidget to connect itself to the signals
 *	it requires.
 */
class ArsonActionEnabler
{
public:
	ArsonActionEnabler (KActionCollection &pa) : m_actions(pa) { }

	void enable (KStdAction::StdAction act, bool enable);
	void enable (const char *name, bool enable);
	void enable (KAction *pa, bool enable);

protected:
	KActionCollection &m_actions;
};

/*========================================================*/

class ArsonDocWidget;

/**
 *	Passed to ArsonDocWidget::connectTo, this allows only
 *	one KAction to exist for each action as opposed to
 *	one KAction per docwidget per action. Each dicwidget
 *	subclass should override connectTo to connect itself
 *	to the signals it requires.
 */
class ArsonActionConnector : public ArsonActionEnabler
{
public:
	ArsonActionConnector (KActionCollection &pa, ArsonDocWidget *pw, bool connect);

	void connect (KStdAction::StdAction act, const char *slot, QObject *recv = NULL, bool enable = true);
	void connect (const char *name, const char *slot, QObject *recv = NULL, bool enable = true);
	void connect (KAction *pa, const char *slot, QObject *recv = NULL, bool enable = true);

	void disconnect (KStdAction::StdAction act);
	void disconnect (const char *name);
	void disconnect (KAction *pa);

private:
	ArsonDocWidget *m_pWidget;
	bool m_connect;
};

/*========================================================*/

class QListViewItem;

class ArsonListInserter
{
public:
	ArsonListInserter (QListViewItem *p = NULL, QListViewItem *a = NULL)
		: parent(p), after(a) { }
	
	void setParent (QListViewItem *pi);
	void clearParent (void);
	void setAfter (QListViewItem *pi);

	QListViewItem *getParent (void) const { return parent; }
	QListViewItem *getAfter (void) const { return after; }

private:
	QListViewItem *last (QListViewItem *pi);

	QListViewItem *parent;
	QListViewItem *after;
};

/*========================================================*/
/*	Exception Class - Abort current action
 *========================================================*/

class ArsonAbortAction { };

/*========================================================*/

class QXmlAttributes;
class QStatusBar;
class ArsonLvPos;
class ArsonXmlWriter;
class ArsonBaseWriter;

/**
 *	The root of all document widgets in arson.
 *
 *	All document widgets (the main widgets in the arson
 *	frame window) are derived from this class.
 */
class ArsonDocWidget : public QVBox
{
	Q_OBJECT

protected:
	/**
	 *	The statusbar information for a doc-widget.
	 *
	 *	Create a new instance of this class, and return
	 *	it from ArsonDocWidget::initStatus().
	 */
	class Status
	{
		typedef QMap<QString,QWidget*> STATUSPANES;
		
	public:
		Status (QStatusBar *psb) : m_pStatus(psb) { }

		///	Adds a QLabel widget to the statusbar
		void addPane (const char *name, const QString &text);

		///	Adds any widget to the statusbar
		void addPane (const char *name, QWidget *pw);

		///	Sets the main message of the status bar (left side)
		void message (const QString &text);

		QWidget *pane (const char *name);
		void show (bool sh);

	private:
		QStatusBar *m_pStatus;		///	The statusbar
		STATUSPANES m_panes;		///	A name->widget mapping for status panes
	};

	enum {
		sizeCanChange = 0x01,
		modified = 0x02,
		stdDocActions = 0x04,
	};

private:
	KActionCollection *m_pActions;
	Status *m_pStatus;
	KURL m_url;
	uint m_flags;

public:
	ArsonDocWidget (QWidget *parent, const char *name = NULL);
	virtual ~ArsonDocWidget (void);

	virtual void newDocument (void);
	virtual bool saveDocument (const KURL &url);
	virtual void setDocumentName (const KURL &url);
	virtual void deleteContents (void);
	virtual void setActive (bool active);

	bool canSizeChange (void) const { return is(sizeCanChange); }
	void setSizeCanChange (bool ch);

	bool isModified (void) const { return is(modified); }
	void setModified (bool mod = true);

	void setActionCollection (KActionCollection *pa) { m_pActions = pa; }
	KActionCollection *actionCollection (void) const { return m_pActions; }

	void setStatusBar (QStatusBar *psb);

	bool is (uint mask) const { return (m_flags & mask) != 0; }
	void clearf (uint mask) { m_flags &= ~mask; }
	void setf (uint mask) { m_flags |= mask; }
	void boolf (uint mask, bool b) { if (b) setf(mask); else clearf(mask); }
	void togglef (uint mask) { m_flags ^= mask; }

	virtual QString propDocType (void) const { return QString::null; }
	virtual QString propFolder (void) const { return QString::null; }
	virtual QString propItem (void) const { return QString::null; }

	virtual bool onStartElement (const QString &name,
		const QXmlAttributes &attr, ArsonListInserter &pos) { return true; }
	virtual bool onEndElement (const QString &name,
		ArsonListInserter &pos) { return true; }
	virtual void defaultLvPos (ArsonLvPos &pos) { }
	virtual void editRootElement (ArsonXmlWriter &xml) { }

	virtual void create (void) { }

	static const QString ARSON_FILTER;

protected:
	void emitSizeChanged (void);

	virtual void connectTo (ArsonActionConnector &ac);
	virtual void uiUpdated (ArsonActionEnabler &en);

	virtual ArsonBaseWriter *createFileWriter (const KURL &url) { return NULL; }
	virtual bool writeDocument (ArsonBaseWriter &writer) = 0;
	bool writeDocumentFile (ArsonBaseWriter &writer);

	virtual Status *initStatus (QStatusBar *psb);

	void setPaneText (const char *name, const QString &text);
	QWidget *pane (const char *name);
	void status (const QString &str);

	ArsonActionConnector connector (bool active);
	ArsonActionEnabler enabler (void);

protected slots:
	void slotUpdateUI (void);
	void slotModified (void);

public slots:
	virtual void refresh (void);
	void slotNew (void);
	void slotSave (void);
	void slotSaveAs (void);
	void slotExportText (void);

signals:
	void sizeChanged (void);
	void updateUI (void);
};

/*========================================================*/
/**
 *	Locks the document size while in scope.
 *
 *	This class automatically locks the specified document
 *	when instantiated, and unlocks the document when it
 *	goes outta scope.
 */
class ArsonDocSizeLock
{
public:
	ArsonDocSizeLock (ArsonDocWidget *pd)
		: m_pd(pd)
	{
		m_pd->setSizeCanChange(false);
	}

	~ArsonDocSizeLock (void)
	{
		m_pd->setSizeCanChange(true);
	}

private:
	ArsonDocWidget *m_pd;
};

/*========================================================*/

class ArsonListWnd;
class ArsonListDoc;

struct ArsonLvPos {
	ArsonLvPos (QListViewItem *p = NULL, QListViewItem *a = NULL)
		: parent(p), after(a), result(NULL) { }
	ArsonLvPos (ArsonListDoc *pd);

	void swapCurrent (void) { after = result; result = NULL; }

	QListViewItem *parent;
	QListViewItem *after;
	QListViewItem *result;
};

namespace arson {
	template<class T> QListViewItem *newListViewItem (ArsonListWnd *pw,
		QListViewItem *pItem, QListViewItem *pAfter)
	{
		if (pItem)
			return new T(pItem, pAfter);

		return new T(pw, pAfter);
	}
};

/**
 *	Base class for all items in a ArsonListDoc. ArsonListDoc
 *	stores a QListViewItem*->ArsonListItem* mapping so that
 *	app-specific data does not need to be stored in the
 *	QListViewItem.
 */
class ArsonListItem
{
public:
	ArsonListItem (void);
	virtual ~ArsonListItem (void);

	virtual QString display (void) const = 0;	//	Human
	virtual QString persist (void) const = 0;	//	Machine

	virtual QListViewItem *createItem (ArsonListWnd *parentWnd,
		QListViewItem *parentItem, QListViewItem *pAfter);

	virtual void refresh (QListViewItem *pi, ArsonDocWidget *pd);
};

/*========================================================*/
/**
 *	Visit each item in a list document.
 *
 *	Does so without regard for branches, treats all
 *	items as equal.
 *
 *	Derived a class from this that does something
 *	useful and pass it to visitAll().
 */
class ArsonListVisiter
{
public:
	virtual bool visit (QListViewItem *pi, ArsonListItem *pl) = 0;
};

/**
 *	Visit each item in a tree document.
 *
 *	Calls visit() for each normal item,
 *	beginBranch() at the start of each
 *	branch (any item with children), and
 *	endBranch() at the end of each branch.
 *
 *	Derived a class from this that does something
 *	useful and pass it to visitAll().
 */
class ArsonTreeVisiter : public ArsonListVisiter
{
public:
	virtual bool beginBranch (QListViewItem *pi, ArsonListItem *pl) { return true; }
	virtual void endBranch (void) { }
};

/*========================================================*/

class QListView;
/**
 *	Describes the (optional) root item in a doc-widget.
 *
 *	If a document is to have a root item (like data CD,
 *	and VCD) create a new instance of this class and pass
 *	it to setRootItemInfo() in the ctor.
 */
class ArsonRootItemInfo
{
public:
	ArsonRootItemInfo (const QString &dflt, const QString &xml, const QString &icon)
		: m_default(dflt), m_xmlAttr(xml), m_icon(icon), m_pItem(NULL) { }

	const QString &defaultText (void) const { return m_default; }
	const QString &xmlAttr (void) const { return m_xmlAttr; }
	const QString &icon (void) const { return m_icon; }

	void resetItem (const QString &text = QString::null);
	void createItem (QListView *pl);
	QString itemText (void) const;
	QListViewItem *item (void) { return m_pItem; }

private:
	QListViewItem *m_pItem;

	QString m_default;
	QString m_xmlAttr;
	QString m_icon;
};

/*========================================================*/

class QDropEvent;
class KListView;
class QPopupMenu;

/**
 *	The root of all doc-widgets containing a listview.
 */
class ArsonListDoc : public ArsonDocWidget
{
	Q_OBJECT

	typedef QMap<QListViewItem*,ArsonListItem*> ITEMMAP;

public:
	ArsonListDoc (QWidget *parent, const char *name = NULL);
	virtual ~ArsonListDoc (void);

	virtual bool writeDocument (ArsonBaseWriter &writer);

	virtual void addItem (ArsonListItem *ptr, ArsonLvPos *pos = NULL);
	virtual void delItem (QListViewItem *ptr);
	virtual void deleteContents (void);

	void setRootItemInfo (ArsonRootItemInfo *ptr);

	virtual void defaultLvPos (ArsonLvPos &pos);
	virtual QListViewItem *rootItem (void) const;
	ArsonListItem *item (QListViewItem *pi) const;

	int count (void) const { return m_items.count(); }
	bool empty (void) const { return !m_pList || count() == 0; }
	ArsonListWnd *listWnd (void) const { return m_pList; }
	int renameColumn (void) const { return m_renameCol; }
	void setRenameColumn (int n) { m_renameCol = n; }
	QString rootText (void) const
	{ return m_pRoot ? m_pRoot->itemText() : QString::null; }

	bool visitAll (ArsonTreeVisiter &obj) const;
	bool visitAll (ArsonListVisiter &obj) const;
	bool visitAll (ArsonTreeVisiter &obj);
	bool visitAll (ArsonListVisiter &obj);
	
	class ItemIterator {
	public:
		ItemIterator (const ArsonListDoc *pd, QListViewItem *pi = NULL);

		ItemIterator childIterator (void) const;
		ItemIterator &operator++ (void);
		ItemIterator &operator++ (int);

		operator void * (void) const { return (void *) (m_pListItem != NULL); }
		ArsonListItem *item (void) { return m_pd->item(m_pListItem); }
		QListViewItem *lvi (void) const { return m_pListItem; }
		ArsonListItem *operator-> (void) { return item(); }

	private:
		QListViewItem *m_pListItem;
		const ArsonListDoc *m_pd;
	};

	virtual void refresh (void);
	virtual void create (void);

protected:
	virtual void connectTo (ArsonActionConnector &ac);
	virtual bool buildContextMenu (QPopupMenu *pm);
	virtual ArsonListWnd *createListWnd (void);
	virtual bool onStartElement (const QString &name,
		const QXmlAttributes &attr, ArsonListInserter &pos);
	virtual void editRootElement (ArsonXmlWriter &xml);
	virtual void uiUpdated (ArsonActionEnabler &en);

	void delTreeListItem (QListViewItem *pi);

	bool editableList (void) const { return m_editableList ? true : false; }
	void setEditableList (bool b) { m_editableList = int(b); }
	
	friend class ItemIterator;

private:
	bool visit (ArsonListVisiter &obj, QListViewItem *pi);
	bool visit (ArsonTreeVisiter &obj, QListViewItem *pi);
	
	ArsonListWnd *m_pList;			//	The list window
	ITEMMAP m_items;				//	A QListViewItem->ArsonListItem map
	int m_renameCol : 8;			//	The renameable column index
	int m_editableList : 1;			//	Can the list order/content be edited?
	ArsonRootItemInfo *m_pRoot;		//	The info for the root item

protected slots:
	void slotContextMenu (KListView *ptr, QListViewItem *item, const QPoint &pnt);
	
public slots:
	void slotListDel (void);
	void slotListUp (void);
	void slotListDn (void);
	void slotListSelAll (void);
	void slotListInvertSel (void);
	void slotRename (void);
	void slotShuffle (void);
};

/*========================================================*/

class QPixmap;

class ArsonFileListItem : public ArsonListItem
{
public:
	ArsonFileListItem (void) : ArsonListItem() { }

	virtual QString display (void) const = 0;
	virtual QString persist (void) const = 0;

	virtual uint length (void) const = 0;
	virtual bool isDir (void) const = 0;

	static QPixmap loadIcon (const char *name);
};

class ArsonFileListFileItem : public ArsonFileListItem
{
public:
	ArsonFileListFileItem (void) : ArsonFileListItem() { }
	ArsonFileListFileItem (const KURL &url)
		: ArsonFileListItem(), m_url(url), m_file(url) { }

	virtual QString display (void) const { return m_url.filename(); }
	virtual QString persist (void) const { return m_url.url(); }

	virtual void refresh (QListViewItem *pi, ArsonDocWidget *pd);
	virtual uint length (void) const;

	virtual bool isDir (void) const { return false; }

	const char *local (void) const { return m_file.path(); }
	const KURL &getURL (void) const { return m_url; }
	void setURL (const KURL &url) { m_url = url; }

	virtual QListViewItem *createItem (ArsonListWnd *parentWnd,
		QListViewItem *parentItem, QListViewItem *pAfter);

private:
	ArsonNetFile m_file;
	KURL m_url;
};

class ArsonFileListDirItem : public ArsonFileListItem
{
public:
	ArsonFileListDirItem (void) : ArsonFileListItem() { }
	ArsonFileListDirItem (const QString &dn);

	virtual QString display (void) const { return m_name; }
	virtual QString persist (void) const { return m_name; }

	virtual uint length (void) const;

	virtual bool isDir (void) const { return true; }

	void setName (const QString &dn) { m_name = dn; }

	virtual QListViewItem *createItem (ArsonListWnd *parentWnd,
		QListViewItem *parentItem, QListViewItem *pAfter);

private:
	QString m_name;
};

/*========================================================*/

class ArsonFileListVisiter : public ArsonListVisiter
{
	virtual bool visit (QListViewItem *pi, ArsonListItem *pl);
	
	virtual bool visitFile (QListViewItem *pi, ArsonFileListFileItem *pf) { return true; }
	virtual bool visitDir (QListViewItem *pi, ArsonFileListDirItem *pf) { return true; }
};

class ArsonFileTreeVisiter : public ArsonTreeVisiter
{
	virtual bool visitFile (QListViewItem *pi, ArsonFileListFileItem *pf) { return true; }
	virtual bool visitDir (QListViewItem *pi, ArsonFileListDirItem *pf) { return true; }

protected:
	virtual bool beginBranch (QListViewItem *pi, ArsonListItem *pl);
	virtual bool visit (QListViewItem *pi, ArsonListItem *pl);
};

/*========================================================*/

class ArsonProgressDisplay
{
public:
	ArsonProgressDisplay (const QString &suffix, int mod = 0)
		: m_suffix(suffix), m_mod(mod) { }

	virtual QString fullText (int length) const;
	virtual int maxProgress (void) const;

	QString suffix (void) const { return m_suffix; }
	int mod (void) const { return m_mod; }

private:
	QString m_suffix;
	int m_mod;
};

/*========================================================*/

class ArsonProgress;
class ArsonProcessMgr;
class ArsonProcessUI;
class ArsonDiskUsageBar;

class ArsonFileListDoc : public ArsonListDoc
{
	Q_OBJECT

public:
	ArsonFileListDoc (QWidget *parent, const char *name = NULL);
	virtual ~ArsonFileListDoc (void);

	virtual void create (void);

	virtual void addItem (ArsonListItem *ptr, ArsonLvPos *pos = NULL);
	virtual void delItem (QListViewItem *ptr);
	virtual void deleteContents (void);

	virtual QString propItem (void) const { return "track"; }

	const ArsonProgressDisplay *progressInfo (void) const { return m_pd; }
	void setProgressInfo (ArsonProgressDisplay *pd);

	virtual void buildFileFilter (ArsonFileFilter &filter) { }
	
	bool addFileItem (const KURL &url, ArsonLvPos *pos = NULL);
	bool addDirItem (const QString &dirname, ArsonLvPos *pp = NULL);

	void openMd5Dir (const QString &dir);
	void openMd5File (const KURL &url);
	void openM3uFile (const KURL &url);

	void setMaxProgress (int maxl);
	void setProgress (int length);

	ArsonFileListFileItem *currentFileItem (void) const;

protected:
	virtual ArsonFileListItem *createFileItem (const KURL &url) const;
	virtual ArsonFileListItem *createDirItem (const QString &dn) const;
	
	virtual ArsonProcessMgr *createCdWriter (ArsonProcessUI *pUI) = 0;
	virtual ArsonProgress *createProgress (QWidget *parent) = 0;
	virtual void connectTo (ArsonActionConnector &ac);
	virtual ArsonBaseWriter *createFileWriter (const KURL &url);
	virtual void addDirectory (const KURL &url, ArsonListInserter &pos);
	virtual Status *initStatus (QStatusBar *psb);
	virtual bool buildContextMenu (QPopupMenu *pm);
	virtual ArsonListWnd *createListWnd (void);
	virtual void uiUpdated (ArsonActionEnabler &en);

	virtual bool onStartElement (const QString &name,
		const QXmlAttributes &attr, ArsonListInserter &pos);
	virtual bool onEndElement (const QString &name,
		ArsonListInserter &pos);

private:
	ArsonDiskUsageBar *m_pProgress;	///	The progress bar (minutes)
	ArsonProgressDisplay *m_pd;		///	Progress display info
	int m_maxProgress;				///	The REAL maximum progress value
	uint m_totalLength;				///	Total length of all items

protected slots:
	void slotDropped (QDropEvent *pd, QListViewItem *parent, QListViewItem *after);
	void slotSizeChanged (void);

public slots:
	void slotListAddDir (void);
	void slotListAdd (void);
	void slotWriteCD (void);
	void slotExportM3u (void);
	void slotFileOpen (void);		///	Bad name, should be slotFileRun or something
	void slotFileOpenWith (void);
	void slotFileView (void);
};

/*========================================================*/
#endif	/*	__DOCWIDGET_H__	*/
