/* Copyright (C) 2002, 2003, 2004 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#ifndef __XSU_HPP
#define __XSU_HPP

#include <boost/shared_ptr.hpp>
#include "compiler.hpp"
#include "database.hpp"
#include "error.hpp"

/** @addtogroup compilers
    @{ */
/** Implementation of a tool similar to Oracle's XML SQL Utility.
    Using a connection to a database, the XML SQL Utility takes SQL scripts
    as input and produces XML as output.
    For example the following input
    \code
    # OPEN test;
    SELECT 1;
    # CLOSE test;
    \endcode
    would result in the following output:
    \code
    <?xml version='1.0' encoding='UTF-8'?>
    <data>
      <test>
        <table>
          <meta>
            <column><![CDATA[1]]></column>
          </meta>
          <row>
            <column><![CDATA[1]]></column>
          </row>
        </table>
      </test>
    </data>
    \endcode
    
    Note, that the text after the '# OPEN'-comment is being used as a tag-name
    for the data.

    @date Thu Mar 28 2005
    @author Jan Wedekind (wedesoft@users.sourceforge.net) */
class XSU: public Compiler
{
 public:
  /** Constructor.
      The constructor takes a reference to a database as argument.
      @param _database Database to pass SQL-queries to. */
  XSU( DatabasePtr _database ): database( _database ) {}
  /** Do SQL-queries.
      @param inputStream Input containing sequence of SQL-queries.
      @param outputStream Stream to write XML-output to. */
  virtual void translate( std::istream &inputStream,
                          std::ostream &outputStream ) const
    throw (Error);
protected:
  /// Pointer to database.
  DatabasePtr database;
};

/** Pointer to XSU-object.
    @see XSU */
typedef boost::shared_ptr< XSU > XSUPtr;

///@}

#endif
