// Copyright (c) 2006, Rodrigo Braz Monteiro
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   * Redistributions of source code must retain the above copyright notice,
//     this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above copyright notice,
//     this list of conditions and the following disclaimer in the documentation
//     and/or other materials provided with the distribution.
//   * Neither the name of the Aegisub Group nor the names of its contributors
//     may be used to endorse or promote products derived from this software
//     without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// Aegisub Project http://www.aegisub.org/

/// @file audio_provider.h
/// @brief Declaration of base-class for audio providers
/// @ingroup main_headers audio_input
///

#pragma once

#include "factory_manager.h"

#include <libaegisub/exception.h>
#include <libaegisub/fs_fwd.h>

#include <boost/filesystem/path.hpp>

class AudioProvider {
protected:
	int channels;

	/// for one channel, ie. number of PCM frames
	int64_t num_samples;
	int sample_rate;
	int bytes_per_sample;
	bool float_samples;
	agi::fs::path filename;

	virtual void FillBuffer(void *buf, int64_t start, int64_t count) const = 0;

	void ZeroFill(void *buf, int64_t count) const;

public:
	virtual ~AudioProvider() { }

	void GetAudio(void *buf, int64_t start, int64_t count) const;
	void GetAudioWithVolume(void *buf, int64_t start, int64_t count, double volume) const;

	agi::fs::path GetFilename()       const { return filename; }
	int64_t       GetNumSamples()     const { return num_samples; }
	int           GetSampleRate()     const { return sample_rate; }
	int           GetBytesPerSample() const { return bytes_per_sample; }
	int           GetChannels()       const { return channels; }
	bool          AreSamplesFloat()   const { return float_samples; }

	/// @brief Does this provider benefit from external caching?
	virtual bool NeedsCache() const { return false; }
};

/// Helper base class for an audio provider which wraps another provider
class AudioProviderWrapper : public AudioProvider {
protected:
	std::unique_ptr<AudioProvider> source;
public:
	AudioProviderWrapper(std::unique_ptr<AudioProvider> src)
	: source(std::move(src))
	{
		channels = source->GetChannels();
		num_samples = source->GetNumSamples();
		sample_rate = source->GetSampleRate();
		bytes_per_sample = source->GetBytesPerSample();
		float_samples = source->AreSamplesFloat();
		filename = source->GetFilename();
	}
};

class AudioProviderFactory : public Factory<AudioProvider, agi::fs::path> {
public:
	static void RegisterProviders();

	/// Get a provider for the file
	/// @param filename URI to open
	static std::unique_ptr<AudioProvider> GetProvider(agi::fs::path const& filename);
};

DEFINE_BASE_EXCEPTION_NOINNER(AudioProviderError, agi::Exception)
/// Error of some sort occurred while decoding a frame
DEFINE_SIMPLE_EXCEPTION_NOINNER(AudioDecodeError, AudioProviderError, "audio/error")
