/**
 * Interface module to the World Magnetic Model library to retrieve the magnetic
 * field components at any given location on the Earth.
 * 
 * Performances: the wmm_getMagneticField() can evaluate more than 8000 point
 * data per second on a Intel Atom processor at 1.6 GHz.
 * 
 * @file
 * @author Umberto Salsi <salsi@icosaedro.it>
 * @version $Date: 2017/10/22 15:02:25 $
 */

#ifndef wmm_H
#define wmm_H

#ifdef wmm_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

/**
 * Result of the query for the components of the magnetic field.
 */
typedef struct {
	/** Components of the magnetic field in the NED reference (nano Tesla). */
	double X, Y, Z;
	
	/** Geomagnetic declination (magnetic variation), positive east (RAD). */
	double Decl;
	
	/** Geomagnetic inclination, positive down (RAD). */
	double Incl;
} wmm_MagneticField;

/**
 * Initializes this module.
 * @param coefficients_file File of the coefficients, normally named "WMM.COF".
 * If file not found or fails to read, it is a fatal error.
 */
EXTERN void wmm_init(char * coefficients_file);

/**
 * Allows to change the default "current date" value used when client asks for
 * magnetic field at year zero.
 * @param date Date as year and fraction.
 */
EXTERN void wmm_setCurrentTime(double date);

/**
 * Returns the components of the Earth's magnetic field at the given location.
 * @param date Date as year and fraction, example 2017.5 = 2017-07-02 12:00 UTC.
 * If set to zero, uses the current date as determined at initialization time,
 * which is accurate enough for the duration of a simulation and saves some lines
 * in the client code.
 * @param latitude Latitude, positive north (RAD).
 * @param longitude Longitude, positive east (RAD).
 * @param altitude Altitude over the WGS84 ellipsoid (m). The result is accurate
 * for a range of altitudes from 1 Km below the surface of the Earth, up to
 * 850 Km above.
 * @param mf Calculated magnetic field.
 */
EXTERN void wmm_getMagneticField(double date, double latitude, double longitude, double altitude, wmm_MagneticField *mf);

#undef EXTERN
#endif
