/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */


package org.netbeans.modules.iep.editor.tcg.model;

import org.netbeans.modules.iep.editor.share.SharedConstants;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.text.ParseException;
import java.util.Date;

/**
 * Enumerated type defining supported types such as Integer, Icon, Double,
 * etc...
 *
 * @author Bing Lu
 *
 * @since May 1, 2002
 */
public class TcgType implements Serializable {
    private static final long serialVersionUID = -4679746282355252393L;    
    

    /**
     * Constant INTEGER type
     */
    public static final TcgType INTEGER = new TcgType("integer") {

        public Object parse(String value) {

            if ((value == null) || value.equals("")) {
                return null;
            }

            return new Integer(value);
        }
    };

    /**
     * Constant INTEGER_LIST type
     */
    public static final TcgType INTEGER_LIST = new TcgType("integerList") {

        public String format(TcgProperty prop, Object value) {
            return formatList(prop, INTEGER, value);
        }

        public Object parse(String value) {
            return getList(INTEGER, value);
        }
        
        public boolean isListType() {
            return true;
        }
    
        public TcgType getElementType() {
            return INTEGER;
        }
    };

    /**
     * Constant LONG type
     */
    public static final TcgType LONG = new TcgType("long") {

        public Object parse(String value) {

            if ((value == null) || value.equals("")) {
                return null;
            }

            return new Long(value);
        }
    };

    /**
     * Constant LONG_LIST type
     */
    public static final TcgType LONG_LIST = new TcgType("longList") {

        public String format(TcgProperty prop, Object value) {
            return formatList(prop, LONG, value);
        }

        public Object parse(String value) {
            return getList(LONG, value);
        }
        
        public boolean isListType() {
            return true;
        }
    
        public TcgType getElementType() {
            return LONG;
        }
    };

    /**
     * Constant DOUBLE type
     */
    public static final TcgType DOUBLE = new TcgType("double") {

        public Object parse(String value) {

            if ((value == null) || value.equals("")) {
                return null;
            }

            return new Double(value);
        }
    };

    /**
     * Constant DOUBLE_LIST type
     */
    public static final TcgType DOUBLE_LIST = new TcgType("doubleList") {

        public String format(TcgProperty prop, Object value) {
            return formatList(prop, DOUBLE, value);
        }

        public Object parse(String value) {
            return getList(DOUBLE, value);
        }

        public boolean isListType() {
            return true;
        }

        public TcgType getElementType() {
            return DOUBLE;
        }
    };

    /**
     * Constant BOOLEAN type
     */
    public static final TcgType BOOLEAN = new TcgType("boolean") {

        public Object parse(String value) {

            if ((value == null) || value.equals("")) {
                return null;
            }

            return Boolean.valueOf(value);
        }
    };

    /**
     * Constant BOOLEAN_LIST type
     */
    public static final TcgType BOOLEAN_LIST = new TcgType("booleanList") {

        public String format(TcgProperty prop, Object value) {
            return formatList(prop, BOOLEAN, value);
        }

        public Object parse(String value) {
            return getList(BOOLEAN, value);
        }
    
        public boolean isListType() {
            return true;
        }

        public TcgType getElementType() {
            return BOOLEAN;
        }
    };

    /**
     * Constant STRING type
     */
    public static final TcgType STRING = new TcgType("string") {
        public Object parse(String value) {
            return value;
        }
    };

    /**
     * Constant STRING_LIST type
     */
    public static final TcgType STRING_LIST = new TcgType("stringList") {

        public String format(TcgProperty prop, Object value) {
            return formatList(prop, STRING, value);
        }

        public Object parse(String value) {
            return getList(STRING, value);
        }

        public boolean isListType() {
            return true;
        }
        
        public TcgType getElementType() {
            return STRING;
        }
    };

    /**
     * Constant DATE type
     */
    public static final TcgType DATE = new TcgType("date") {
        public String format(TcgProperty prop, Object value) {
            return (value == null)? "" : SharedConstants.DATE_FORMAT.format(value);
        }

        public Object parse(String value) {
            try {
                return SharedConstants.DATE_FORMAT.parse(value);
            } catch (ParseException e) {
                return null;
            }
        }
    };

    /**
     * Constant STRING_LIST type
     */
    public static final TcgType DATE_LIST = new TcgType("dateList") {

        public String format(TcgProperty prop, Object value) {
            return formatList(prop, DATE, value);
        }

        public Object parse(String value) {
            return getList(DATE, value);
        }

        public boolean isListType() {
            return true;
        }
        
        public TcgType getElementType() {
            return DATE;
        }
    };

    /**
     * Constant OBJECT type
     */
    public static final TcgType OBJECT = new TcgType("object") {
        // This method should never be called
        public String format(TcgProperty prop, Object value) {
            return (value == null) ? "null" : "" + value;
        }

        // This method should never be called
        // expect when parsing the default value during initialization
        public Object parse(String value) {
            return value;
        }
    };    
    
    private String mCode;

    private TcgType(String code) {
        mCode = code;
    }

    /**
     * Gets the tcgType given the code
     *
     * @param code String value to match
     *
     * @return The tcgType matching input code value
     */
    public static TcgType getType(String code) {
        if (code.equals("integer")) {
            return INTEGER;
        }

        if (code.equals("integerList")) {
            return INTEGER_LIST;
        }

        if (code.equals("double")) {
            return DOUBLE;
        }

        if (code.equals("doubleList")) {
            return DOUBLE_LIST;
        }

        if (code.equals("boolean")) {
            return BOOLEAN;
        }

        if (code.equals("booleanList")) {
            return BOOLEAN_LIST;
        }

        if (code.equals("string")) {
            return STRING;
        }

        if (code.equals("stringList")) {
            return STRING_LIST;
        }

        if (code.equals("object")) {
            return OBJECT;
        }        
        
        if (code.equals("long")) {
            return LONG;
        }
        
        if (code.equals("longList")) {
            return LONG_LIST;
        }
        if (code.equals("date")) {
            return DATE;
        }
        
        if (code.equals("dateList")) {
            return DATE_LIST;
        }
        return null;
    }

    /**
     * Gets the string representationreal for the value
     *
     * @param value the real value for which a string representation is desired
     *
     * @return the string representationreal for the value
     */
    public String format(TcgProperty prop, Object value) {

        return (value == null)
               ? ""
               : value.toString();
    }
    
    /**
     * Gets the real value for its string representation
     *
     * @param value the string representation for the real value
     *
     * @return the value behind the string representation
     */
    public Object parse(String value) {
        return value;
    }
    
    public boolean isListType() {
        return false;
    }
    
    public TcgType getElementType() {
        return null;
    }
    
    /**
     * Gets the code name for this object
     *
     * @return the code name for this object
     */
    public String toString() {
        return mCode;
    }

    public String getCode() {
        return mCode;
    }    
    
    private static List getList(TcgType singleType, String value) {

        List l = new ArrayList();

        if ((value == null) || value.equals("")) {
            return l;
        }

        StringTokenizer st = new StringTokenizer(value, "\n\\");

        while (st.hasMoreTokens()) {
            String t = st.nextToken().trim();

            if (!t.equals("")) {
                l.add(singleType.parse(t));
            }
        }

        return l;
    }

    private static String formatList(TcgProperty prop, TcgType singleType, Object value) {

        if (value == null) {
            return "";
        }

        StringBuffer sb = new StringBuffer();
        List l = (List) value;

        for (int i = 0, size = l.size(); i < size; i++) {
            if (i > 0) {
                sb.append("\\");
            }

            sb.append(singleType.format(prop, l.get(i)));
        }

        return sb.toString();
    }
}


/*--- Formatted in SeeBeyond Java Convention Style on Thu, Dec 5, '02 ---*/


/*------ Formatted by Jindent 3.24 Gold 1.02 --- http://www.jindent.de ------*/
