/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.sql.framework.highlighter;

import java.awt.Event;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JTextPane;
import javax.swing.KeyStroke;
import javax.swing.event.UndoableEditEvent;
import javax.swing.event.UndoableEditListener;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.undo.CannotRedoException;
import javax.swing.undo.CannotUndoException;
import javax.swing.undo.UndoManager;

import org.netbeans.modules.sql.framework.model.SQLDBTable;


/**
 * SQLEditorPane extends from JTextPane that uses syntax highlighting.
 * 
 * @author Ahimanikya Satapathy
 * @version $Revision: 1.2 $
 */
public class SQLEditorPane extends JTextPane {

    class RedoAction extends AbstractAction {
        public RedoAction() {
            super("Redo");
            setEnabled(false);
        }

        public void actionPerformed(ActionEvent e) {
            try {
                undoManager.redo();
            } catch (CannotRedoException ex) {
                System.out.println("Unable to redo: " + ex);
                ex.printStackTrace();
            }
            updateRedoState();
            undoAction.updateUndoState();
        }

        protected void updateRedoState() {
            if (undoManager.canRedo()) {
                setEnabled(true);
                putValue(Action.NAME, undoManager.getRedoPresentationName());
            } else {
                setEnabled(false);
                putValue(Action.NAME, "Redo");
            }
        }
    }

    class UndoAction extends AbstractAction {
        public UndoAction() {
            super("Undo");
            setEnabled(false);
        }

        public void actionPerformed(ActionEvent e) {
            try {
                undoManager.undo();
            } catch (CannotUndoException ex) {
                System.out.println("Unable to undo: " + ex);
                ex.printStackTrace();
            }
            updateUndoState();
            redoAction.updateRedoState();
        }

        protected void updateUndoState() {
            if (undoManager.canUndo()) {
                setEnabled(true);
                putValue(Action.NAME, undoManager.getUndoPresentationName());
            } else {
                setEnabled(false);
                putValue(Action.NAME, "Undo");
            }
        }
    }

    //This one listens for edits that can be undone.
    protected class SQLUndoableEditListener implements UndoableEditListener {
        public void undoableEditHappened(UndoableEditEvent e) {
            // TODO This flag is used to ignore edits for setting of document attributes
            // for highlighting purpose, but this flag has synchronization
            // problem, specifically when user highlight a text and
            // press some key to insert new text at the same time then
            // this check passes and returns from here even for this
            // new insert. Which in turn causes undo to throw some exception.
            // (This is because TextColorer thread happens to set this flag at the time
            // new insert occured)
            //   if(sqlDocument.isIgnoreUndoEdit()) {
            //       return;
            //   }

            //Remember the edit and update the menus.
            undoManager.addEdit(e.getEdit());
            undoAction.updateUndoState();
            redoAction.updateRedoState();
        }
    }

    /**
     * A thread that handles the actual coloring.
     */
    private SQLTextColorer colorer;

    /**
     * A lock for modifying the sqlDocument, or for actions that depend on the sqlDocument
     * not being modified.
     */
    private Object doclock = new Object();

    /**
     * A reader wrapped around the sqlDocument so that the sqlDocument can be fed into the
     * lexer.
     */
    private DocumentReader documentReader;

    /** redo helper */
    private RedoAction redoAction;

    /**
     * the styled sqlDocument that is the model for the textPane
     */
    private HighLightedDocument sqlDocument;

    // Set up the hash table that contains the styles.
    private StyleHelper styleHelper = new StyleHelper();

    /**
     * The lexer that tells us what colors different words should be.
     */
    private Lexer syntaxLexer;

    private List tables = new ArrayList();

    /** undo helpers */
    private UndoAction undoAction;

    /**
     * Undo/Redo Manager
     */
    private UndoManager undoManager = new UndoManager();

    /**
     * Constructor
     */
    public SQLEditorPane() {

        // Create the sqlDocument model.
        sqlDocument = new HighLightedDocument(this);

        // Create the text pane and configure it.
        this.setDocument(sqlDocument);
        this.setCaretPosition(0);
        this.setMargin(new Insets(5, 5, 5, 5));

        // create the new sqlDocument.
        documentReader = new DocumentReader(sqlDocument);

        // Undo and redo are actions of our own creation.
        undoAction = new UndoAction();
        redoAction = new RedoAction();

        // Add some key bindings.
        addBindings();

        // Put the initial text into the text pane and
        // set it's initial coloring style.
        initDocument();

        sqlDocument.addUndoableEditListener(new SQLUndoableEditListener());

    }

    /**
     * Color a section of the sqlDocument. The actual coloring will start somewhere before
     * the requested position and continue as long as needed.
     * 
     * @param position the starting point for the coloring.
     * @param adjustment amount of text inserted or removed at the starting point.
     */
    public void color(int position, int adjustment) {
        getTextColorer().color(position, adjustment);
    }

    /**
     * Color or recolor the entire sqlDocument
     */
    public void colorAll() {
        color(0, sqlDocument.getLength());
    }

    /**
     * retrieve the style for the given type of text.
     * 
     * @param styleName the label for the type of text ("tag" for example) or null if the
     *        styleName is not known.
     * @return the style
     */
    public SimpleAttributeSet getCustomStyle(String styleName) {
        return styleHelper.getStyle(styleName);
    }

    /**
     * @return Returns the doclock.
     */
    public Object getDoclock() {
        return doclock;
    }

    /**
     * @return Returns the documentReader.
     */
    public DocumentReader getDocumentReader() {
        return documentReader;
    }

    public String getIdentifierType(String name) {
        Iterator it = getTables().iterator();
        while (it.hasNext()) {
            SQLDBTable table = (SQLDBTable) it.next();
            if (name.equalsIgnoreCase(table.getCatalog())) {
                return StyleHelper.CATALOG;
            } else if (name.equalsIgnoreCase(table.getSchema())) {
                return StyleHelper.SCHEMA;
            } else if (name.equalsIgnoreCase(table.getAliasName())) {
                return StyleHelper.TABLE_ALIAS;
            } else if (name.equalsIgnoreCase(table.getName())) {
                return StyleHelper.TABLE;
            }
        }
        return null;
    }

    /**
     * @return Returns the sqlDocument.
     */
    public HighLightedDocument getSqlDocument() {
        return sqlDocument;
    }

    /**
     * @return Returns the syntaxLexer.
     */
    public Lexer getSyntaxLexer() {
        return syntaxLexer;
    }

    /**
     * @return Returns the tables.
     */
    public List getTables() {
        return tables;
    }

    public SQLTextColorer getTextColorer() {
        if (this.colorer == null) {
            // Start the thread that does the coloring
            this.colorer = new SQLTextColorer(this);
            this.colorer.start();
        }

        return this.colorer;
    }

    public void removeNotify() {
        if (colorer != null) {
            // Mark to stop in next run
            colorer.stopThread();
            colorer = null;
        }
        super.removeNotify();
    }

    /**
     * @param tables The tables to set.
     */
    public void setTables(List tables) {
        if (tables != null) {
            this.tables = tables;
        }
    }

    //Add a couple of emacs key bindings for navigation.
    private void addBindings() {
        KeyStroke keyUndo = KeyStroke.getKeyStroke(KeyEvent.VK_Z, Event.CTRL_MASK);
        getInputMap().put(keyUndo, undoAction);

        KeyStroke keyRedo = KeyStroke.getKeyStroke(KeyEvent.VK_Y, Event.CTRL_MASK);
        getInputMap().put(keyRedo, redoAction);
    }

    /**
     * Initialize the sqlDocument with some default text and set they initial type of
     * syntax highlighting.
     */
    private void initDocument() {
        syntaxLexer = new SQLLexer(documentReader);
    }

}

