/*
 *                 Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License
 * Version 1.0 (the "License"). You may not use this file except in
 * compliance with the License. A copy of the License is available at
 * http://www.sun.com/
 *
 * The Original Code is NetBeans. The Initial Developer of the Original
 * Code is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.

If you wish your version of this file to be governed by only the CDDL
or only the GPL Version 2, indicate your decision by adding
"[Contributor] elects to include this software in this distribution
under the [CDDL or GPL Version 2] license." If you do not indicate a
single choice of license, a recipient has the option to distribute
your version of this file under either the CDDL, the GPL Version 2 or
to extend the choice of license to its licensees as provided above.
However, if you add GPL Version 2 code and therefore, elected the GPL
Version 2 license, then the option applies only if the new code is
made subject to such option by the copyright holder.

If you wish your version of this file to be governed by only the CDDL
or only the GPL Version 2, indicate your decision by adding
"[Contributor] elects to include this software in this distribution
under the [CDDL or GPL Version 2] license." If you do not indicate a
single choice of license, a recipient has the option to distribute
your version of this file under either the CDDL, the GPL Version 2 or
to extend the choice of license to its licensees as provided above.
However, if you add GPL Version 2 code and therefore, elected the GPL
Version 2 license, then the option applies only if the new code is
made subject to such option by the copyright holder.
 */

package org.netbeans.modules.sql.framework.common.utils;

import java.net.URL;
import java.net.URLClassLoader;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.SQLException;
import java.util.Properties;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;

import org.netbeans.api.db.explorer.ConnectionManager;
import org.netbeans.api.db.explorer.DatabaseConnection;
import org.netbeans.api.db.explorer.JDBCDriver;
import org.netbeans.api.db.explorer.JDBCDriverManager;
import org.netbeans.modules.mashup.db.common.FlatfileDBConnectionFactory;
import org.netbeans.modules.mashup.db.ui.AxionDBConfiguration;

import com.sun.sql.framework.exception.DBSQLException;
import com.sun.sql.framework.utils.ScEncrypt;
import com.sun.sql.framework.jdbc.DBConnectionFactory;
import com.sun.sql.framework.utils.Logger;

/**
 *
 * @author radval
 */
public class DBExplorerConnectionUtil {
    
    private static final String LOG_CATEGORY = DBExplorerConnectionUtil.class.getName();
    
    /** Creates a new instance of DBExplorerConnectionUtil */
    public DBExplorerConnectionUtil() {
    }
    
    public static Connection createConnection(Properties connProps) throws DBSQLException {
        String driver = connProps.getProperty(DBConnectionFactory.PROP_DRIVERCLASS);
        String username = connProps.getProperty(DBConnectionFactory.PROP_USERNAME);
        String password = connProps.getProperty(DBConnectionFactory.PROP_PASSWORD);
        String url = connProps.getProperty(DBConnectionFactory.PROP_URL);
        return createConnection(driver, url, username, password);
    }
    
    public static Connection createConnection(String driverName, String url, String username, String password) throws DBSQLException {
        Connection conn = null;
        JDBCDriver drv = null;
        String schema = null;
        try {
            if(driverName.equals("org.axiondb.jdbc.AxionDriver")) {
                drv = registerAxionDriverInstance();
            } else {
                drv = registerDriverInstance(driverName);
            }
            
            // Get the connection directly. Dont go through DB Explorer.
            // It may pop up a window asking for password.
            if(driverName.equals("org.axiondb.jdbc.AxionDriver")) {
                conn = FlatfileDBConnectionFactory.getInstance().
                        getConnection(url, "sa", "sa", Thread.currentThread().getContextClassLoader());
            } else {
                conn = getConnection(drv, driverName, url, username, password);
            }
            
            // check if connection exists in DB Explorer. Else add the connection to DB Explorer.
            DatabaseConnection dbconn = null;
            DatabaseConnection[] dbconns = ConnectionManager.getDefault().getConnections();
            for (int i=0; i < dbconns.length; i++) {
                if (dbconns[i].getDriverClass().equals(driverName) &&
                        dbconns[i].getDatabaseURL().equals(url) && dbconns[i].getUser().equals(username)) {
                    dbconn = dbconns[i];
                    break;
                }
            }
            
            // dont add instance db and monitor db connections to db explorer.
            if (dbconn == null) {
                if(url.indexOf("InstanceDB") == -1 && url.indexOf("MonitorDB") == -1) {
                    if(url.startsWith("jdbc:axiondb")) {
                        schema = "";
                    } else {
                        schema = (String)username.toUpperCase();
                    }
                    dbconn = DatabaseConnection.create(drv, url, username, schema, password, true);
                    ConnectionManager.getDefault().addConnection(dbconn);
                }
            }
            try {
                conn.setAutoCommit(false);
            } catch (NullPointerException ex) {
                NotifyDescriptor d = new NotifyDescriptor.Message(
                        "Connection could not be established. Please check the Database Server.",
                        NotifyDescriptor.WARNING_MESSAGE);
                DialogDisplayer.getDefault().notify(d);
            }
        } catch (Exception e) {
            throw new DBSQLException("Connection could not be established.", e);
        }
        return conn;
    }
    
    
    /**
     * Registers an instance of Driver associated with the given driver class name. Does
     * nothing if an instance has already been registered with the JDBC DriverManager.
     *
     * @param driverName class name of driver to be created
     * @return Driver instance associated with <code>driverName</code>
     * @throws Exception if error occurs while creating or looking up the desired driver
     *         instance
     */
    
    public static JDBCDriver registerDriverInstance(final String driverName) throws Exception {
        JDBCDriver driver = null;
        JDBCDriver[] drivers;
        try {
            drivers = JDBCDriverManager.getDefault().getDrivers(driverName);
        } catch(Exception ex) {
            throw new Exception("Invalid driver name specified.");
        }
        if(driverName.equals("org.axiondb.jdbc.AxionDriver")) {
            driver = registerAxionDriverInstance();
        } else {
            if (drivers.length == 0) {
                throw new Exception("Specified JDBC Driver not found in DB Explorer.");
            } else {
                driver = drivers[0];
            }
        }
        return driver;
    }
    
    private static JDBCDriver registerAxionDriverInstance() throws Exception {
        JDBCDriver driver = null;
        String driverName = "org.axiondb.jdbc.AxionDriver";
        JDBCDriver[] drivers = JDBCDriverManager.getDefault().getDrivers(driverName);
        if (drivers.length == 0) {
            // if axion db driver not available in db explorer, add it.
            URL[] url = new URL[1];
            if(getDefaultAxionLoc() == null || getDefaultAxionLoc().equals("")) {
                throw new Exception("Axion Driver could not be located");
            }
            url[0] = new URL("file:/" + getDefaultAxionLoc());
            driver = JDBCDriver.create(driverName, "Mashup DB", driverName, url);
            JDBCDriverManager.getDefault().addDriver(driver);
        }
        if (driver == null) {
            for (int i=0; i < drivers.length; i++) {
                if (drivers[i].getClassName().equals(driverName)) {
                    driver = drivers[i];
                    break;
                }
            }
        }
        return driver;
    }
    
    /**
     * Manually load the driver class and get the connection for the specified properties.
     *
     * @return connection for the corresponding db url and properties.
     */
    private static Connection getConnection(JDBCDriver drv, String driverName, String url, String username, String password) {
        Connection conn = null;
        Driver newDriverClass = null;
        try {
            // get the driver jar files and load them manually
            URL[] urls = drv.getURLs();
            ClassLoader cl = new URLClassLoader(urls, Thread.currentThread().getContextClassLoader());
            newDriverClass = (Driver)cl.loadClass(driverName).newInstance();
            Properties prop = new Properties();
            prop.setProperty("user", username);
            prop.setProperty("password", password);
            conn = newDriverClass.connect(url, prop);
        } catch(SQLException ex) {
            try {
                // may be some class forgot to decrypt the password. Check if this one works
                Properties prop = new Properties();
                prop.setProperty("user", username);
                // if its not an encrypted password, decrypt operation will fail. Caught as general Exception
                password = ScEncrypt.decrypt(username, password);
                prop.setProperty("password", password);
                conn = newDriverClass.connect(url, prop);
            } catch(SQLException e) {
                Logger.print(Logger.ERROR, LOG_CATEGORY,
                        "Unable to get the specified connection directly.");
            } catch(Exception numex) {
                Logger.print(Logger.ERROR, LOG_CATEGORY,
                        "Unable to get the specified connection directly.");
            }
        } catch (Exception ex) {
            Logger.print(Logger.ERROR, LOG_CATEGORY, "Unable to find the driver class in the specified jar file");
        }
        return conn;
    }
    
    /*
     * Returns the default location of axiondb.jar along with the dependencies.
     *
     */
    private static String getDefaultAxionLoc() {
        File conf = AxionDBConfiguration.getConfigFile();
        Properties prop = new Properties();
        try {
            FileInputStream in = new FileInputStream(conf);
            prop.load(in);
        } catch (FileNotFoundException ex) {
            //ignore
        } catch (IOException ex) {
            //ignore
        }
        String loc = prop.getProperty(AxionDBConfiguration.PROP_DRIVER_LOC);
        return loc;
    }
}