/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.bpel.debugger.ui.psm;

import java.util.IdentityHashMap;
import java.util.Map;
import java.util.StringTokenizer;
import javax.xml.namespace.QName;
import org.netbeans.modules.bpel.debugger.api.EditorContextBridge;
import org.netbeans.modules.bpel.debugger.api.psm.ProcessStaticModel;
import org.netbeans.modules.bpel.debugger.ui.util.ModelUtil;
import org.netbeans.modules.bpel.model.api.Activity;
import org.netbeans.modules.bpel.model.api.Assign;
import org.netbeans.modules.bpel.model.api.BooleanExpr;
import org.netbeans.modules.bpel.model.api.BpelEntity;
import org.netbeans.modules.bpel.model.api.BpelModel;
import org.netbeans.modules.bpel.model.api.Branches;
import org.netbeans.modules.bpel.model.api.Catch;
import org.netbeans.modules.bpel.model.api.CatchAll;
import org.netbeans.modules.bpel.model.api.Compensate;
import org.netbeans.modules.bpel.model.api.CompensateScope;
import org.netbeans.modules.bpel.model.api.CompensationHandler;
import org.netbeans.modules.bpel.model.api.CompletionCondition;
import org.netbeans.modules.bpel.model.api.Condition;
import org.netbeans.modules.bpel.model.api.Copy;
import org.netbeans.modules.bpel.model.api.Correlation;
import org.netbeans.modules.bpel.model.api.CorrelationContainer;
import org.netbeans.modules.bpel.model.api.CorrelationSet;
import org.netbeans.modules.bpel.model.api.CorrelationSetContainer;
import org.netbeans.modules.bpel.model.api.DeadlineExpression;
import org.netbeans.modules.bpel.model.api.Documentation;
import org.netbeans.modules.bpel.model.api.Else;
import org.netbeans.modules.bpel.model.api.ElseIf;
import org.netbeans.modules.bpel.model.api.Empty;
import org.netbeans.modules.bpel.model.api.EventHandlers;
import org.netbeans.modules.bpel.model.api.Exit;
import org.netbeans.modules.bpel.model.api.ExtensibleAssign;
import org.netbeans.modules.bpel.model.api.Extension;
import org.netbeans.modules.bpel.model.api.ExtensionActivity;
import org.netbeans.modules.bpel.model.api.ExtensionContainer;
import org.netbeans.modules.bpel.model.api.ExtensionEntity;
import org.netbeans.modules.bpel.model.api.FaultHandlers;
import org.netbeans.modules.bpel.model.api.FinalCounterValue;
import org.netbeans.modules.bpel.model.api.Flow;
import org.netbeans.modules.bpel.model.api.For;
import org.netbeans.modules.bpel.model.api.ForEach;
import org.netbeans.modules.bpel.model.api.From;
import org.netbeans.modules.bpel.model.api.FromPart;
import org.netbeans.modules.bpel.model.api.FromPartContainer;
import org.netbeans.modules.bpel.model.api.If;
import org.netbeans.modules.bpel.model.api.Import;
import org.netbeans.modules.bpel.model.api.Invoke;
import org.netbeans.modules.bpel.model.api.Link;
import org.netbeans.modules.bpel.model.api.LinkContainer;
import org.netbeans.modules.bpel.model.api.Literal;
import org.netbeans.modules.bpel.model.api.MessageExchange;
import org.netbeans.modules.bpel.model.api.MessageExchangeContainer;
import org.netbeans.modules.bpel.model.api.OnAlarmEvent;
import org.netbeans.modules.bpel.model.api.OnAlarmPick;
import org.netbeans.modules.bpel.model.api.OnEvent;
import org.netbeans.modules.bpel.model.api.OnMessage;
import org.netbeans.modules.bpel.model.api.PartnerLink;
import org.netbeans.modules.bpel.model.api.PartnerLinkContainer;
import org.netbeans.modules.bpel.model.api.PatternedCorrelation;
import org.netbeans.modules.bpel.model.api.PatternedCorrelationContainer;
import org.netbeans.modules.bpel.model.api.Pick;
import org.netbeans.modules.bpel.model.api.Process;
import org.netbeans.modules.bpel.model.api.Query;
import org.netbeans.modules.bpel.model.api.ReThrow;
import org.netbeans.modules.bpel.model.api.Receive;
import org.netbeans.modules.bpel.model.api.RepeatEvery;
import org.netbeans.modules.bpel.model.api.RepeatUntil;
import org.netbeans.modules.bpel.model.api.Reply;
import org.netbeans.modules.bpel.model.api.Scope;
import org.netbeans.modules.bpel.model.api.Sequence;
import org.netbeans.modules.bpel.model.api.ServiceRef;
import org.netbeans.modules.bpel.model.api.Source;
import org.netbeans.modules.bpel.model.api.SourceContainer;
import org.netbeans.modules.bpel.model.api.StartCounterValue;
import org.netbeans.modules.bpel.model.api.Target;
import org.netbeans.modules.bpel.model.api.TargetContainer;
import org.netbeans.modules.bpel.model.api.TerminationHandler;
import org.netbeans.modules.bpel.model.api.Throw;
import org.netbeans.modules.bpel.model.api.To;
import org.netbeans.modules.bpel.model.api.ToPart;
import org.netbeans.modules.bpel.model.api.ToPartContainer;
import org.netbeans.modules.bpel.model.api.Validate;
import org.netbeans.modules.bpel.model.api.Variable;
import org.netbeans.modules.bpel.model.api.VariableContainer;
import org.netbeans.modules.bpel.model.api.Wait;
import org.netbeans.modules.bpel.model.api.While;
import org.netbeans.modules.bpel.model.api.support.SimpleBpelModelVisitor;

/**
 *
 * @author Alexander Zgursky
 */
public class ProcessStaticModelImpl implements ProcessStaticModel {
    private final QName myProcessQName;
    private PsmEntityImpl myRoot;
    private transient Map<BpelEntity, PsmEntityImpl> myBuildMap;
    private transient Map<PsmEntityImpl, PsmEntityImpl> myEventHandlersMap;
    private transient Map<PsmEntityImpl, PsmEntityImpl> myFaultHandlersMap;

    public static ProcessStaticModelImpl build(
            QName processQName, BpelModel bpelModel)
    {
        ProcessStaticModelImpl psm = new ProcessStaticModelImpl(processQName);
        if (psm.tryToBuild(bpelModel)) {
            return psm;
        } else {
            return null;
        }
    }
    
    /** Creates a new instance of ProcessStaticModelImpl */
    private ProcessStaticModelImpl(QName processQName) {
        myProcessQName = processQName;
    }
    
    public QName getProcessQName() {
        return myProcessQName;
    }
    
    public PsmEntityImpl getRoot() {
        return myRoot;
    }
    
    public PsmEntityImpl find(String xpath) {
        if (myRoot == null) {
            return null;
        } else if (!xpath.startsWith("/")) {
            return null;
        }
        
        StringTokenizer tokenizer = new StringTokenizer(xpath, "/", false);
        PsmEntityImpl currentEntity = null;
        while (tokenizer.hasMoreTokens()) {
            String token = tokenizer.nextToken();
            String noPrefix = token.substring(token.indexOf(':') + 1);
            int index;
            String localPart;
            if (noPrefix.endsWith("]")) {
                int ind = noPrefix.indexOf('[');
                localPart = noPrefix.substring(0, ind);
                index = Integer.parseInt(noPrefix.substring(ind + 1, noPrefix.length() - 1));
            } else {
                localPart = noPrefix;
                index = 1;
            }
            
            if (currentEntity == null && localPart.equals("process") && index == 1) {
                currentEntity = myRoot;
                continue;
            }
            
            if (currentEntity == null) {
                return null;
            }

            int foundIndex = 0;
            for (PsmEntityImpl child : currentEntity.getChildren()) {
                if (child.getQName().getLocalPart().equals(localPart)) {
                    foundIndex++;
                    if (foundIndex == index) {
                        currentEntity = child;
                        break;
                    }
                }
            }
            if (foundIndex != index) {
                return null;
            }
        }
        return currentEntity;
    }
    
    private boolean tryToBuild(BpelModel bpelModel) {
        BuildTask buildTask = new BuildTask(bpelModel);
        myBuildMap = new IdentityHashMap<BpelEntity, PsmEntityImpl>();
        myEventHandlersMap = new IdentityHashMap<PsmEntityImpl, PsmEntityImpl>();
        myFaultHandlersMap = new IdentityHashMap<PsmEntityImpl, PsmEntityImpl>();
        
        bpelModel.invoke(buildTask);

        //finally adding event handlers and fault handlers
        for (Map.Entry<PsmEntityImpl, PsmEntityImpl> entry : myFaultHandlersMap.entrySet()) {
            entry.getKey().addChild(entry.getValue());
        }
        
        for (Map.Entry<PsmEntityImpl, PsmEntityImpl> entry : myEventHandlersMap.entrySet()) {
            entry.getKey().addChild(entry.getValue());
        }
        
        myBuildMap = null;
        myEventHandlersMap = null;
        myFaultHandlersMap = null;
        return buildTask.getResult();
    }
    
    private void addProcess(Process bpelProcess) {
        String xpath = EditorContextBridge.normalizeXpath(
                ModelUtil.getFindHelper().getXPath(bpelProcess));
        myRoot = new PsmEntityImpl(
                xpath,
                makeQName(bpelProcess),
                bpelProcess.getName(),
                false,
                false);
        
        myBuildMap.put(bpelProcess, myRoot);
    }
    
    private void addActivity(Activity activity) {
        String xpath = EditorContextBridge.normalizeXpath(
                ModelUtil.getFindHelper().getXPath(activity));
        PsmEntityImpl psmEntity = new PsmEntityImpl(
                xpath,
                makeQName(activity),
                activity.getName(),
                true,
                false);
        
        BpelEntity bpelParent = activity.getParent();
        PsmEntityImpl psmParent = myBuildMap.get(bpelParent);
        psmParent.addChild(psmEntity);
        
        myBuildMap.put(activity, psmEntity);
    }
    
    private void addLoop(Activity activity) {
        String xpath = EditorContextBridge.normalizeXpath(
                ModelUtil.getFindHelper().getXPath(activity));
        PsmEntityImpl psmEntity = new PsmEntityImpl(
                xpath,
                makeQName(activity),
                activity.getName(),
                true,
                true);
        
        BpelEntity bpelParent = activity.getParent();
        PsmEntityImpl psmParent = myBuildMap.get(bpelParent);
        psmParent.addChild(psmEntity);
        
        myBuildMap.put(activity, psmEntity);
    }
    
    private void addEntity(BpelEntity bpelEntity) {
        String xpath = EditorContextBridge.normalizeXpath(
                ModelUtil.getFindHelper().getXPath(bpelEntity));
        PsmEntityImpl psmEntity = new PsmEntityImpl(
                xpath,
                makeQName(bpelEntity),
                null,
                false,
                false);
        
        BpelEntity bpelParent = bpelEntity.getParent();
        PsmEntityImpl psmParent = myBuildMap.get(bpelParent);
        psmParent.addChild(psmEntity);
        
        myBuildMap.put(bpelEntity, psmEntity);
    }
    
    private void addEventHandler(BpelEntity bpelEntity) {
        String xpath = EditorContextBridge.normalizeXpath(
                ModelUtil.getFindHelper().getXPath(bpelEntity));
        PsmEntityImpl psmEntity = new PsmEntityImpl(
                xpath,
                makeQName(bpelEntity),
                null,
                false,
                true);
        
        BpelEntity bpelParent = bpelEntity.getParent();
        PsmEntityImpl psmParent = myBuildMap.get(bpelParent);
        psmParent.addChild(psmEntity);
        
        myBuildMap.put(bpelEntity, psmEntity);
    }
    
    private void addEventHandlers(BpelEntity bpelEntity) {
        String xpath = EditorContextBridge.normalizeXpath(
                ModelUtil.getFindHelper().getXPath(bpelEntity));
        PsmEntityImpl psmEntity = new PsmEntityImpl(
                xpath,
                makeQName(bpelEntity),
                null,
                false,
                false);
        
        BpelEntity bpelParent = bpelEntity.getParent();
        PsmEntityImpl psmParent = myBuildMap.get(bpelParent);
        //Delayed to add event handlers and fault handlers to add them
        //later in the end of the children list
        
        //psmParent.addChild(psmEntity);
        myEventHandlersMap.put(psmParent, psmEntity);
        
        myBuildMap.put(bpelEntity, psmEntity);
    }
    
    private void addFaultHandlers(BpelEntity bpelEntity) {
        String xpath = EditorContextBridge.normalizeXpath(
                ModelUtil.getFindHelper().getXPath(bpelEntity));
        PsmEntityImpl psmEntity = new PsmEntityImpl(
                xpath,
                makeQName(bpelEntity),
                null,
                false,
                false);
        
        BpelEntity bpelParent = bpelEntity.getParent();
        PsmEntityImpl psmParent = myBuildMap.get(bpelParent);
        //Delayed to add event handlers and fault handlers to add them
        //later in the end of the children list

        //psmParent.addChild(psmEntity);
        myFaultHandlersMap.put(psmParent, psmEntity);
        
        myBuildMap.put(bpelEntity, psmEntity);
    }
    
    private QName makeQName(BpelEntity bpelEntity) {
        String prefix = bpelEntity.getPeer().getPrefix();
        if (prefix != null) {
            return new QName(
                    BpelEntity.BUSINESS_PROCESS_NS_URI,
                    bpelEntity.getPeer().getLocalName(),
                    prefix);
        } else {
            return new QName(
                    BpelEntity.BUSINESS_PROCESS_NS_URI,
                    bpelEntity.getPeer().getLocalName());
        }
    }
    
    private class BuildTask implements Runnable {
        private final BpelModel myBpelModel;
        private boolean myResult;

        public BuildTask(BpelModel bpelModel) {
            myBpelModel = bpelModel;
        }
        
        public boolean getResult() {
            return myResult;
        }

        public void run() {
            if (!myBpelModel.getState().equals(BpelModel.State.VALID)) {
                return;
            }
            
            if (myBpelModel.getProcess() == null) {
                return;
            }
            
            MyVisitor visitor = new MyVisitor();
            myBpelModel.getProcess().accept(visitor);
            myResult = true;
        }
    }
    
    
    private class MyVisitor implements SimpleBpelModelVisitor {
//        private BuildTask myBuildTask;
//        
//        public MyVisitor(BuildTask buildTask) {
//            myBuildTask = buildTask;
//        }
        
        public void visit(Process process) {
            addProcess(process);
        }

        public void visit(Empty empty) {
            addActivity(empty);
        }

        public void visit(Invoke invoke) {
            addActivity(invoke);
        }

        public void visit(Receive receive) {
            addActivity(receive);
        }

        public void visit(Reply reply) {
            addActivity(reply);
        }

        public void visit(Assign assign) {
            addActivity(assign);
        }

        public void visit(Wait wait) {
            addActivity(wait);
        }

        public void visit(Throw throv) {
            addActivity(throv);
        }

        public void visit(Exit terminate) {
            addActivity(terminate);
        }

        public void visit(Flow flow) {
            addActivity(flow);
        }

        public void visit(While whil) {
            addLoop(whil);
        }

        public void visit(Sequence sequence) {
            addActivity(sequence);
        }

        public void visit(Pick pick) {
            addActivity(pick);
        }

        public void visit(Scope scope) {
            addActivity(scope);
        }

        public void visit(PartnerLinkContainer container) {
            //noop
        }

        public void visit(PartnerLink link) {
            //noop
        }

        public void visit(FaultHandlers handlers) {
            addFaultHandlers(handlers);
        }

        public void visit(Catch catc) {
            addEntity(catc);
        }

        public void visit(EventHandlers handlers) {
            addEventHandlers(handlers);
        }

        public void visit(OnMessage message) {
            addEntity(message);
        }

        public void visit(CompensationHandler handler) {
            addEntity(handler);
        }

        public void visit(VariableContainer container) {
            //noop
        }

        public void visit(Variable variable) {
            //noop
        }

        public void visit(CorrelationSetContainer container) {
            //noop
        }

        public void visit(CorrelationSet set) {
            //noop
        }

        public void visit(Source source) {
            //noop
        }

        public void visit(Target target) {
            //noop
        }

        public void visit(CorrelationContainer container) {
            //noop
        }

        public void visit(Correlation correlation) {
            //noop
        }

        public void visit(PatternedCorrelation correlation) {
            //noop
        }

        public void visit(PatternedCorrelationContainer container) {
            //noop
        }

        public void visit(To to) {
            //noop
        }

        public void visit(From from) {
            //noop
        }

        public void visit(Compensate compensate) {
            addActivity(compensate);
        }

        public void visit(LinkContainer container) {
            //noop
        }

        public void visit(Link link) {
            //noop
        }

        public void visit(Copy copy) {
            //noop
        }

        public void visit(CatchAll holder) {
            addEntity(holder);
        }

        public void visit(BooleanExpr expr) {
            //noop
        }

        public void visit(Branches branches) {
            //noop
        }

        public void visit(CompletionCondition condition) {
            //noop
        }

        public void visit(Condition condition) {
            //noop
        }

        public void visit(DeadlineExpression expression) {
            //noop
        }

        public void visit(Documentation documentation) {
            //noop
        }

        public void visit(Else els) {
            addEntity(els);
        }

        public void visit(ElseIf elseIf) {
            addEntity(elseIf);
        }

        public void visit(ExtensibleAssign assign) {
            //noop
        }

        public void visit(ExtensionActivity activity) {
            //noop
        }

        public void visit(Validate validate) {
            addActivity(validate);
        }

        public void visit(ToPart toPart) {
            //noop
        }

        public void visit(ToPartContainer toPartContainer) {
            //noop
        }

        public void visit(TerminationHandler handler) {
            addEntity(handler);
        }

        public void visit(TargetContainer container) {
            //noop
        }

        public void visit(StartCounterValue value) {
            //noop
        }

        public void visit(SourceContainer container) {
            //noop
        }

        public void visit(ReThrow rethrow) {
            addActivity(rethrow);
        }

        public void visit(RepeatUntil repeatUntil) {
            addLoop(repeatUntil);
        }

        public void visit(RepeatEvery repeatEvery) {
            //noop
        }

        public void visit(OnEvent event) {
            addEventHandler(event);
        }

        public void visit(OnAlarmPick alarmPick) {
            addEntity(alarmPick);
        }

        public void visit(OnAlarmEvent alarmEvent) {
            addEventHandler(alarmEvent);
        }

        public void visit(ExtensionContainer container) {
            //noop
        }

        public void visit(Extension extension) {
            //noop
        }

        public void visit(FinalCounterValue value) {
            //noop
        }

        public void visit(ForEach forEach) {
            addLoop(forEach);
        }

        public void visit(Literal literal) {
            //noop
        }

        public void visit(Import imp) {
            //noop
        }

        public void visit(If iff) {
            addActivity(iff);
        }

        public void visit(FromPart fromPart) {
            //noop
        }

        public void visit(FromPartContainer fromPartContainer) {
            //noop
        }

        public void visit(For fo) {
            //noop
        }

        public void visit(MessageExchangeContainer container) {
            //noop
        }

        public void visit(MessageExchange exchange) {
            //noop
        }

        public void visit(ServiceRef ref) {
            //noop
        }

        public void visit(ExtensionEntity entity) {
            //noop
        }

        public void visit(CompensateScope compensateScope) {
            addActivity(compensateScope);
        }

        public void visit(Query query) {
            //noop
        }
        
    }
    
}
