/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.cnd.apt.utils;

import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;
import org.netbeans.modules.cnd.apt.debug.APTTraceFlags;


/**
 * APT string table manager
 * Responsibility:
 *  - only one instance per String object
 *  - based on weak references to allow GC of unused strings
 * 
 * @author Vladimir Voskresensky
 */
public abstract class APTStringManager  {
    
    public abstract String getString(String text);
    public abstract void dispose();

    private static final Map<String, APTSingleStringManager> instances = Collections.synchronizedMap(new HashMap<String, APTSingleStringManager>());

    private static final int STRING_MANAGER_DEFAULT_CAPACITY=1024;

    /*package*/ static final String TEXT_MANAGER="Manager of sharable texts"; // NOI18N
    /*package*/ static final int    TEXT_MANAGER_INITIAL_CAPACITY=STRING_MANAGER_DEFAULT_CAPACITY;
    /*package*/ static final String FILE_PATH_MANAGER="Manager of sharable file paths"; // NOI18N
    /*package*/ static final int    FILE_PATH_MANAGER_INITIAL_CAPACITY=STRING_MANAGER_DEFAULT_CAPACITY;
    
    public static APTStringManager instance(String kind) {
        return instance(kind, STRING_MANAGER_DEFAULT_CAPACITY);
    }
    
    /*package*/ static APTSingleStringManager instance(String kind, int initialCapacity) {
        APTSingleStringManager instance = instances.get(kind);
        if (instance == null) {
            instance = new APTSingleStringManager(initialCapacity);
            instances.put(kind, instance);
        }
        return instance;
    }  
        
    /*package*/ static final class APTSingleStringManager extends APTStringManager {
        private final Map<String, Reference<String>> strings;
        private final WeakSharedSet<String> storage;

        /** Creates a new instance of APTStringManager */
        private APTSingleStringManager(int initialCapacity) {
            storage = new WeakSharedSet<String>(initialCapacity);
            strings = new WeakHashMap<String, Reference<String>>();
        }

        // we need exclusive copy of string => use "new String(String)" constructor
        private final String lock = new String("lock in APTStringManager"); // NOI18N

        /**
         * returns shared string instance equal to input text.
         * 
         * @param test - interested shared string 
         * @return the shared instance of text
         * @exception NullPointerException If the <code>text</code> parameter
         *                                 is <code>null</code>.
         */
        public final String getString(String text) {
            if (text == null) {
                throw new NullPointerException("null string is illegal to share"); // NOI18N
            }
            String outText = null;
            synchronized (lock) {
                if (APTTraceFlags.APT_USE_STORAGE_SET) {
                    outText = storage.addOrGet(text);
                } else {
                    Reference<String> val = strings.get(text);
                    // hope between this two lines GC doesn't remove map-entry associated
                    // with returned value from "strings" WeakHashMap,
                    // otherwise we should always "put" outText back into map
                    outText = val != null ? val.get()  : null;
                    if (outText == null) {                
                        outText = text;
                        strings.put(outText, new WeakReference<String>(outText));                
                    }
                }
            }
            assert (outText != null);
            assert (outText.equals(text));
            return outText;
        }

        public final void dispose() {
            if (APTTraceFlags.APT_USE_STORAGE_SET) {
                storage.clear();
            } else {
                strings.clear();
            }
        }
    }
    
    /*package*/ static final class APTCompoundStringManager extends APTStringManager {
        private final APTSingleStringManager[] instances;
        private final int capacity; // primary number for better distribution
        /*package*/APTCompoundStringManager(int capacity) {
            this.capacity = capacity;
            instances = new APTSingleStringManager[capacity];
            for (int i = 0; i < instances.length; i++) {
                instances[i] = APTStringManager.instance(APTStringManager.TEXT_MANAGER + i, APTStringManager.TEXT_MANAGER_INITIAL_CAPACITY);
            }
        }
        
        private APTStringManager getDelegate(String text) {
            if (text == null) {
                throw new NullPointerException("null string is illegal to share"); // NOI18N
            }            
            int index = text.hashCode() % capacity;
            if (index < 0) {
                index += capacity;
            }
            return instances[index];
        }
        
        public final String getString(String text) {
            return getDelegate(text).getString(text);
        }

        public final void dispose() {
            for (int i = 0; i < instances.length; i++) {
                instances[i].dispose();
            }            
        }        
    }    
}
