/*
 * Copyright 2006 Ricoh Corporation.
 * 
 * 
 * APACHE LICENSE VERSION 2.0
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * 
 * RICOH DEVELOPER PROGRAM SUPPORT:
 * 
 * Support for this software is available only to "Premier Plus" members
 * of the Ricoh Developer Program (RiDP).  You may find out more 
 * information about the Program at
 * 
 *      http://americas.ricoh-developer.com
 * 
 * Premier plus members may find answers and ask questions through the
 * RiDP customer help website at
 * 
 *      https://ridp.custhelp.com
 * 
 * Developers who are not RiDP members may still use this software as
 * stipulated in the license terms given above.
 *
 */

// import ESA (Embedded Software Architecture)
import jp.co.ricoh.dsdk.panel.*;
import jp.co.ricoh.dsdk.panel.event.*;
import jp.co.ricoh.dsdk.util.*;
import jp.co.ricoh.dsdk.xlet.*;

/**
 * A children's puzzle involving numbered tiles.
 */
public class Puzzle15Xlet implements Xlet
{
    private static final boolean TRACE = false;

    XletContext xletContext;

    Frame frameWindow = null;

    Button titleButton = null;
    
    Label numberOfMovesLabel;
    
    Button[] tileButtonArray = new Button[4 * 4];
    
    Button aboutButton = null;
    
    Button scrambleButton = null;
    
    Button solveButton = null;
    
    Button exitButton = null;
    
    Puzzle15Model puzzle15Model = new Puzzle15Model();

    boolean solving = false;
    
    Thread solvingThread;

    /**
     * Does nothing.
     */
    public Puzzle15Xlet( )
    {
    }

    /**
     * Called when the xlet is first initialized.
     *
     * Note that the lifecycle methods of an xlet,
     * initXlet, startXlet, pauseXlet and destroyXlet must
     * be declared "synchronized".  See item 2 on p. 18
     * of the xlet develop manual for more information.
     */
    public synchronized void initXlet( XletContext context ) throws XletStateChangeException
    {        
        this.xletContext = context;
        makeUI();
    }

    /**
     * Called when the xlet is started.
     */
    public synchronized void startXlet( )
    {
    }

    /**
     * Called when the xlet is paused.
     */
    public synchronized void pauseXlet( )
    {
    }

    /**
     * Called when the xlet is no longer needed.
     */
    public synchronized void destroyXlet( boolean unconditional )
    {
        //
        // An xlet must insure that its threads are all halted
        // when destroyXlet is called; otherwise there would
        // be a memory leak in the system.
        //
        stopSolving();
    }

    /**
     * Returns a reference to the tile (a button).
     */
    private Button getTile( int x, int y )
    {
        return tileButtonArray[x + (y * 4)];
    }

    /**
     * Returns a reference to the main window.
     */
    private Frame getFrame( ) throws XletStateChangeException
    {
        // find the frame window
        Container parent = null;
        
        try
        {
            parent = xletContext.getContainer();
        }
        catch (UnavailableContainerException ex)
        {
            throw new XletStateChangeException(ex.toString());
        }
        
        while(!(parent instanceof Frame))
        {
            parent = parent.getParent();
            
            if (parent == null)
            {
                return null;
            }
        }
        
        return (Frame)parent;
    }

    /**
     * Creates the labels, tiles and buttons.
     */
    private void makeUI( ) throws XletStateChangeException
    {
        // find the frame window
        frameWindow = getFrame();

        // create the title (button)
        titleButton = new Button("15 Puzzle");
        
        titleButton.setLocation(25, 60);
        titleButton.setSize(160, 40);
        titleButton.setFont(Font.ALNUM16);
        titleButton.setShape(Button.Shape.BGW);
        titleButton.setWink(true);
        
        frameWindow.add(titleButton);
        
        // create the "Move:" label
        numberOfMovesLabel = new Label("Moves: 0");
        
        numberOfMovesLabel.setSize(160, 40);
        numberOfMovesLabel.setLocation(35, 130);
        numberOfMovesLabel.setFont(Font.ALNUM16);
        
        frameWindow.add(numberOfMovesLabel);
    
        // create the tiles
        final int tileWidth = 50;
        final int tileHeight = 50;
        int top = 15;
        int tileNumber = 0;

        for (int y=0; y < 4; ++y)
        {
            int left = 220;
            
            for (int x=0; x < 4; ++x)
            {
                // create the tile
                Button tile = new Button();
                
                tile.setLabel("");
                tile.setLocation(left, top);
                tile.setSize(tileWidth, tileHeight);
                tile.setFont(Font.NUM20);
                tile.setShape(Button.Shape.ZAB);
        
                tile.addActionListener(new PuzzleChangeListener(tile, x, y));
                
                frameWindow.add(tile);

                // add the tile to the array
                tileButtonArray[tileNumber++] = tile;
                
                left += tileWidth;
            }
            
            top += tileHeight;
        }

        // initialize with the model
        syncAllTilesToModel();
        
        // create the "About" button
        aboutButton = new Button("About...");

        aboutButton.setLocation(470, 20);
        aboutButton.setSize(80, 30);
        aboutButton.setShape(Button.Shape.NOCNR2);
        
        aboutButton.addActionListener(
            new ActionListener()
            {
                public void actionPerformed(ActionEvent e)
                {
                    tr("About button pressed.");
                    Puzzle15AboutDialog aboutDialog = new Puzzle15AboutDialog(frameWindow);
                    aboutDialog.show();
                    
                    return;
                }
            }
        );
    
        frameWindow.add(aboutButton);
        
        // create the "Scramble" button
        scrambleButton = new Button("Scramble");

        scrambleButton.setLocation(470, 70);
        scrambleButton.setSize(80, 30);
        scrambleButton.setShape(Button.Shape.NOCNR2);
        
        scrambleButton.addActionListener(
            new ActionListener()
            {
                public void actionPerformed(ActionEvent e)
                {
                    scrambleTiles();
                }
            }
        );
    
        frameWindow.add(scrambleButton);

        // create the "Solve" button
        solveButton = new Button("Solve");

        solveButton.setLocation(470, 120);
        solveButton.setSize(80, 30);
        solveButton.setShape(Button.Shape.NOCNR2);
        
        solveButton.addActionListener(
            new ActionListener()
            {
                public void actionPerformed(ActionEvent e)
                {
                    tr("Solve button pressed.");
                    if (solving)
                    {
                        stopSolving();
                    }
                    else
                    {
                        startSolving();
                    }
                }
            }
        );
    
        frameWindow.add(solveButton);

        // create the "Exit" button
        exitButton = new Button("Exit");
        
        exitButton.setSize(80, 30);
        exitButton.setLocation(470, 170);
        exitButton.setShape(Button.Shape.NOCNR2);
        
        exitButton.addActionListener(
            new ActionListener()
            {
                public void actionPerformed(ActionEvent e)
                {
                    //
                    // Terminate the application by calling
                    // destroyXlet directly.  We then call
                    // notifyDestroyed on the xlet context to
                    // let it know that we were destroyed.
                    //
                    // See item 9 of p. 23 of the xlet develop
                    // manual.
                    //
                    destroyXlet(true);
                    xletContext.notifyDestroyed();
                    
                    return;
                }
            }
        );
    
        frameWindow.add(exitButton);
        
        //
        // Add a "move listener" to the puzzle model. It notifies us
        // whenever a tile is moved so we can update the view. For speed,
        // only the affected tiles are refreshed.
        //
        puzzle15Model.addMoveListener(
            new Puzzle15Model.MoveListener()
            {
                //
                // Note that the "moved" method may be called from a background
                // thread; however, the ESA DSDK panel APIs are thread-safe, so
                // this is not a problem.
                //
                public void moved(final int fromH, final int fromV, final int toH, final int toV, final int moveNbr)
                {
                    handleTileMoveAnimation(fromH, fromV, toH, toV, moveNbr);
                }
            }
        );
    }
    
    /**
     * This method is called whenever anyone tells the puzzle15 model
     * to move a tile.  This can be done either in response to a user
     * event (clicking on a tile) or automatically from a background
     * thread when the puzzle is being solved.
     *
     * Note that the ESA DSDK Panel APIs are thread-safe, so it is
     * okay to call them from a background thread.
     */
    public void handleTileMoveAnimation(final int fromH, final int fromV, final int toH, final int toV, final int moveNbr)
    {
        tr("Moved called: {" + fromH + ", " + fromV + "} -> {" + toH + ", " + toV + "}");

        if (moveNbr > 0)
        {
            syncTileToModel(fromH, fromV);
            syncTileToModel(toH, toV);
            setNumberOfMoves(moveNbr);
        }
        else
        {
            syncAllTilesToModel();
        }
    }

    /**
     * Move all of the tiles to random locations.
     */
    public void scrambleTiles()
    {
	tr("Scramble button pressed.");

        if (!solving)
        {
            puzzle15Model.scramble();
            syncAllTilesToModel();
            setNumberOfMoves(0);
        }
    }

    /**
     * Start solving:  create a new solving thread that will
     * call the 'solve' method of the puzzle15 model.
     */
    public void startSolving()
    {
        solving = true;


        solveButton.setLabel("Stop Solve");
        solveButton.repaint();
        //setDisplayMode(DisplayMode.GRAY);
        scrambleButton.setEnabled(false);
        scrambleButton.repaint();
        
        tr("Ask for threadgroup...");
        ThreadGroup xletThreadGroup = (ThreadGroup) xletContext.getXletProperty(XletContext.THREADGROUP);
        
        tr("Start a new thread...");
        solvingThread = new Thread(xletThreadGroup, new Solver() /* target */);
        solvingThread.setDaemon(true);
        solvingThread.start();
    }

    /**
     * Stops the solving action by interrupting its thread and waiting
     * for the interrupted thread to terminate.
     */
    public void stopSolving( )
    {
        if (solving)
        {
            solvingThread.interrupt();
            
            try
            {
                solvingThread.join();
            }
            catch (InterruptedException ex)
            {
            }
            
            solvingThread = null;
            solving = false;
        }
    }

    /**
     * Make sure that the tiles in the UI match the
     * tiles in the model.
     */
    private void syncAllTilesToModel()
    {
        for (int h=0; h < 4; ++h)
        {
            for (int v=0; v < 4; ++v)
            {
                syncTileToModel(h, v);
            }
        }
    }
    
    /**
     * Refreshes the specified tile to its current value according to
     * the model.
     */
    private void syncTileToModel( int x, int y )
    {
        // find out how to display this tile from the model
        int tileNumber = puzzle15Model.get(x, y);
        String label = (tileNumber == 0)? "": Integer.toString(tileNumber);
        Button.Shape shape = (tileNumber == 0)? Button.Shape.NONE: Button.Shape.ZAB;
        
        // update the tile
        Button tile = getTile(x, y);

        tile.setLabel(label);
        tile.setShape(shape);
        tile.repaint();
    }

    /**
     * Refreshes the "Move:" static text.
     */
    private void setNumberOfMoves( int n )
    {
        numberOfMovesLabel.setText("Moves: " + n);
        numberOfMovesLabel.repaint();
    }
    
    /**
     * Background thread that calls the puzzle15 model's "solve"
     * method and handles displaying the solution to the user
     * via animating the numeric buttons displayed on the LCD.
     */
    private final class Solver implements Runnable 
    {
        long start = 0;

        /**
	 * Here is where we fire off the solution algorithm.
	 * We provide it with a callback function that will
	 * sleep between each move so that the user can see
	 * the tile animation happening.  When everything
	 * is done, we set the display of the "Solve" button
	 * back to "Solve".
	 */
	public void run()
        {
            start = System.currentTimeMillis();
        
            puzzle15Model.solve(
                new Puzzle15Model.MoveHandler()
                {
                    public void handleMove(int h, int v, int moveNum)
                    {
                        sleepDuringSolution(h, v, moveNum);
                    }
                }
            );

            solving = false;

            solveButton.setLabel("Solve");
            solveButton.repaint();
            
            scrambleButton.setEnabled(true);
            scrambleButton.repaint();
        }

        /**
	 * This routine is called by the handleMove callback method
	 * given to the puzzle15 model.  All we're doing here is
	 * sleeping for a short period of time so that the user can
	 * see the tile animation happening.  The actual tile
	 * animation is done by a change listener added to the model
	 * when it is created in "makeUI".
	 */
	public void sleepDuringSolution(int h, int v, int moveNum)
        {
            long sleepTime = 300 - (System.currentTimeMillis() - start);

            if (sleepTime > 0)
            {
                try
                {
                    Thread.sleep(sleepTime);
                }
                catch (InterruptedException ex)
                {
                    solvingThread.interrupt();
                }
            }

            start = System.currentTimeMillis();
        }
	
    }

    /**
     * A listener that responds to a tap of any of the "tile" buttons.
     */    
    private class PuzzleChangeListener implements ActionListener
    {
        private Component item;
        
        private int h;
        
        private int v;
    
        public PuzzleChangeListener(Component item, int h, int v)
        {
            this.item = item;
            this.h = h;
            this.v = v;
            
            return;
        }
    
        public void actionPerformed(ActionEvent evt)
        {
            if (solving)
            {
                tr("Change event: automatic solving is already running");
                return;
            }
            
            tr("Change event: move to (" + h + ", " + v + ")");
            
            puzzle15Model.move(h, v);
            
            return;
        }
    }
    
    /**
     * Print a trace message if the TRACE flag is set
     */
    private void tr( String s )
    {
        if (TRACE)
        {
            System.err.println("@@@ " + s);
        }
    }
}

