/*
 * Copyright 2006 Ricoh Corporation.
 * 
 * 
 * APACHE LICENSE VERSION 2.0
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * 
 * RICOH DEVELOPER PROGRAM SUPPORT:
 * 
 * Support for this software is available only to "Premier Plus" members
 * of the Ricoh Developer Program (RiDP).  You may find out more 
 * information about the Program at
 * 
 *      http://americas.ricoh-developer.com
 * 
 * Premier plus members may find answers and ask questions through the
 * RiDP customer help website at
 * 
 *      https://ridp.custhelp.com
 * 
 * Developers who are not RiDP members may still use this software as
 * stipulated in the license terms given above.
 *
 */ 

import jp.co.ricoh.dsdk.panel.AttributeType;
import jp.co.ricoh.dsdk.panel.Button;
import jp.co.ricoh.dsdk.panel.Container;
import jp.co.ricoh.dsdk.panel.Label;
//import jp.co.ricoh.dsdk.panel.Window;
import jp.co.ricoh.dsdk.panel.event.ActionEvent;
import jp.co.ricoh.dsdk.panel.event.ActionListener;
import jp.co.ricoh.dsdk.system.LockMachineState;
import jp.co.ricoh.dsdk.xlet.UnavailableContainerException;
import jp.co.ricoh.dsdk.xlet.Xlet;
import jp.co.ricoh.dsdk.xlet.XletContext;
import jp.co.ricoh.dsdk.xlet.XletStateChangeException;

/**
 * This sample demonstrates how to correctly lock down the machine state
 * and then free it again on exit.  This is a simplified version of 
 * LockControlTest, which allows you to individually lock and release
 * the different lock modes.
 */
public class SampleLockControl implements Xlet, ActionListener 
{
	private XletContext context = null;
	private Container rootContainer;

	/*
	 * Labels for the different lock states
	 */
	private Label lockPwOffLbl          = new Label("Power Off");
	private Label lockPnlOffLbl         = new Label("Panel Off");
	private Label lockPnlReleaseLbl     = new Label("Panel Release");
	private Label lockOfflineLbl        = new Label("Offline");
	private Label lockIntShiftLbl       = new Label("IntShift");

	/*
	 * Show the status of the different lock modes.
	 */
	private Label lockPwOffStatLbl          = new Label("State :");
	private Label lockPnlOffStatLbl         = new Label("State :");
	private Label lockPnlReleaseStatLbl     = new Label("State :");
	private Label lockOfflineStatLbl        = new Label("State :");
	private Label lockIntShiftStatLbl       = new Label("State :");
	
	/*
	 * The lock mode can be either "locked" or "unlocked"	
	 */
	private final String STAT_LOCKED   = "State : Locked";
	private final String STAT_UNLOCKED = "State : Unlocked";

	/*
	 * The lock button toggles the state between locked and unlocked;
	 * the end button terminates the program.
	 */
	private Button lockBtn          = new Button("Lock");
	private Button endBtn           = new Button("END");

	/*
	 * Cache the lock machine state instance.
	 */
	private LockMachineState lockMcnStat = null;
	private String strTitle = null;
	private boolean lockState = false;

	/*
	 * @see jp.co.ricoh.dsdk.xlet.Xlet#initXlet(jp.co.ricoh.dsdk.xlet.XletContext)
	 */
	public synchronized void initXlet(XletContext xletContext) throws XletStateChangeException 
	{
		context = xletContext;

		String[] args = (String[]) context.getXletProperty(XletContext.ARGS);
		if (args != null && args.length > 0) 
		{
			strTitle = args[0];
		}

		try 
		{
			lockMcnStat = LockMachineState.getInstance();

			rootContainer = context.getContainer();
			createGUI();

		} 
		catch (UnavailableContainerException ex) 
		{
			throw new XletStateChangeException();
		}

	}

	/*
	 * @see jp.co.ricoh.dsdk.xlet.Xlet#startXlet()
	 */
	public synchronized void startXlet() throws XletStateChangeException 
	{
	}

	/*
	 * @see jp.co.ricoh.dsdk.xlet.Xlet#pauseXlet()
	 */
	public synchronized void pauseXlet() 
	{
	}
	
	/*
	 * @see jp.co.ricoh.dsdk.xlet.Xlet#destroyXlet(boolean)
	 */
	public synchronized void destroyXlet(boolean force) throws XletStateChangeException 
	{

		if (force) 
		{
			//
			// ATTENTION: be sure to unlock everything
			// you may have locked prior to stopping this xlet
			// This implementation of destroyXlet takes no 
			// chances, and unlocks everything regardless of state
			//
			lockMcnStat.unlockPowerOff(this);
			lockMcnStat.unlockPanelOff(this);
			lockMcnStat.unlockPanelRelease(this);
			lockMcnStat.unlockOffline(this);
			lockMcnStat.unlockIntShift(this);

			rootContainer.removeAll();
		} 
		else 
		{
			throw new XletStateChangeException();
		}
	}

	/**
	 * Create user interface elements
	 */
	private void createGUI() 
	{
		final int base_x      = 30;
		final int base_y      = 40;
		final int base_width  = 95;
		final int base_height = 25;
		final int dist_x      = base_width + 15;
		final int dist_y      = base_height + 5;

		Label titleMsg = new Label(strTitle);
		titleMsg.setBounds(240, 15, 150, 20);
		rootContainer.add(titleMsg);

		// Power Off
		lockPwOffLbl.setBounds(base_x, base_y, base_width, base_height);
		rootContainer.add(lockPwOffLbl);

		lockPwOffStatLbl.setBounds(base_x, base_y + dist_y, base_width, base_height);
		rootContainer.add(lockPwOffStatLbl);
		
		lockBtn.setBounds(base_x, base_y + dist_y * 2, base_width, base_height);
		lockBtn.addActionListener(this);
		rootContainer.add(lockBtn);

		// Panel Off
		lockPnlOffLbl.setBounds(base_x + dist_x, base_y, base_width, base_height);
		rootContainer.add(lockPnlOffLbl);

		lockPnlOffStatLbl.setBounds(base_x + dist_x, base_y + dist_y, base_width, base_height);
		rootContainer.add(lockPnlOffStatLbl);
		
		// Panel Release
		lockPnlReleaseLbl.setBounds(base_x + dist_x * 2, base_y, base_width, base_height);
		rootContainer.add(lockPnlReleaseLbl);

		lockPnlReleaseStatLbl.setBounds(base_x + dist_x * 2, base_y + dist_y, base_width, base_height);
		rootContainer.add(lockPnlReleaseStatLbl);

		// Offline
		lockOfflineLbl.setBounds(base_x + dist_x * 3, base_y, base_width, base_height);
		rootContainer.add(lockOfflineLbl);

		lockOfflineStatLbl.setBounds(base_x + dist_x * 3, base_y + dist_y, base_width, base_height);
		rootContainer.add(lockOfflineStatLbl);

		// IntShift
		lockIntShiftLbl.setBounds(base_x + dist_x * 4, base_y, base_width, base_height);
		rootContainer.add(lockIntShiftLbl);

		lockIntShiftStatLbl.setBounds(base_x + dist_x * 4, base_y + dist_y, base_width, base_height);
		rootContainer.add(lockIntShiftStatLbl);
		
		// End
		endBtn.setBounds(base_x + dist_x * 4, base_y + dist_y * 5, base_width, base_height);
		endBtn.addActionListener(this);
		rootContainer.add(endBtn);

		this.showCurrentLockState();
	}

	/* 
	 * @see jp.co.ricoh.dsdk.panel.event.ActionListener#actionPerformed(jp.co.ricoh.dsdk.panel.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent ae) 
	{

		Button btnEvt = (Button) ae.getSource();

		/*
		 * Process a button press
		 */
		if (btnEvt.equals(endBtn)) 
		{
			try
			{
				destroyXlet(true);
				context.notifyDestroyed();
			}
			catch(XletStateChangeException e)
			{
				System.out.println(e.toString());
			}
		}
		else
		{
			toggleLock();
		}
	}

	/**
	 * Repaint one status label when its state changes.
	 * @param blnLocked
	 * @param statLbl
	 */
	private void repaintDisplay(boolean blnLocked, Label statLbl)
	{
		if(blnLocked)
		{
			statLbl.setText(STAT_LOCKED);
			statLbl.repaint();
		}
		else
		{
			statLbl.setText(STAT_UNLOCKED);
			statLbl.repaint();
		}
	}
	
	/**
	 * Switches between locked and unlocked state
	 * if you wish to test a subset of locks, simple
	 * comment out the appropriate code inside the following
	 * method
	 */
	private void toggleLock()
	{
		//
		// Respond to changes in locked state
		//
		if (!lockState) 
		{
			this.lockMachine();
		}
		//
		// Switch to unlocked state
		//
		else
		{
			this.unlockMachine();
		}

		System.out.println(" State: " + lockState);
	}

	private void lockMachine()
	{
		lockState = true;
		
		//
		// The power-off lock prevents the user from turning off the MFP
		//
		lockMcnStat.lockPowerOff(this);

		//
		// The panel-off lock prevents the energy-saving mode from
		// turning off the display.  In general, you should not
		// use this lock mode to keep the panel on for long periods
		// of time.
		//
		lockMcnStat.lockPanelOff(this);

		//
		// The panel-release lock prevents the user from exiting
		// the Java function via other function keys (e.g. "Copy").
		// It is still possible to switch between any Java
		// applications that may be running.
		//
		lockMcnStat.lockPanelRelease(this);

		//
		// The online lock prevents the machine from going into
		// energy-saving mode.  In general, you should not use
		// this lock mode to keep the machine online for long
		// periods of time.
		// 
		lockMcnStat.lockOffline(this);

		//
		// The int-shift lock prevents the user from switching
		// to the "copy" function with the interrupt key.
		// In general, if you lock the panel you should also
		// lock the int shift mode.
		//
		lockMcnStat.lockIntShift(this);

		lockBtn.setAttributeType(AttributeType.INVERSE);
		lockBtn.repaint();
		
		this.showCurrentLockState();
	}
	
	
	private void unlockMachine()
	{
		lockState = false;
		
		//
		// The various 'unlock' calls will return a boolean indicating
		// whether the machine actually unlocked or not.  You may also
		// query the locked state with the various 'isLocked' calls.
		//
		lockMcnStat.unlockPowerOff(this);
		lockMcnStat.unlockPanelOff(this);
		lockMcnStat.unlockPanelRelease(this);
		lockMcnStat.unlockOffline(this);
		lockMcnStat.unlockIntShift(this);

		lockBtn.setAttributeType(AttributeType.NONE);
		lockBtn.repaint();
		
		this.showCurrentLockState();
	}
	
	/**
	 * Set the various UI labels to reflect the current lock state of the machine
	 */
	private void showCurrentLockState()
	{
		repaintDisplay(lockMcnStat.isLockedPowerOff(), lockPwOffStatLbl);
		repaintDisplay(lockMcnStat.isLockedPanelOff(), lockPnlOffStatLbl);
		repaintDisplay(lockMcnStat.isLockedPanelRelease(), lockPnlReleaseStatLbl);
		repaintDisplay(lockMcnStat.isLockedOffline(), lockOfflineStatLbl);
		repaintDisplay(lockMcnStat.isLockedIntShift(), lockIntShiftStatLbl);
	}
}
