/*
* Copyright 2005 Sun Microsystems, Inc. All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
*
* - Redistributions of source code must retain the above copyright
*   notice, this list of conditions and the following disclaimer.
*
* - Redistribution in binary form must reproduce the above copyright
*   notice, this list of conditions and the following disclaimer in
*   the documentation and/or other materials provided with the
*   distribution.
*
* Neither the name of Sun Microsystems, Inc. or the names of
* contributors may be used to endorse or promote products derived
* from this software without specific prior written permission.
*
* This software is provided "AS IS," without a warranty of any
* kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
* WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
* EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES
* SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
* DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN
* OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR
* FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
* PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
* LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE SOFTWARE,
* EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*
* You acknowledge that Software is not designed, licensed or intended
* for use in the design, construction, operation or maintenance of
* any nuclear facility.
*/

package com.sun.j2ee.blueprints.opc.workflowmanager.handlers;

import java.rmi.*;

import javax.jms.*;
import javax.ejb.*;

import com.sun.j2ee.blueprints.opc.purchaseorder.*;
import com.sun.j2ee.blueprints.opc.orderreceiver.*;
import com.sun.j2ee.blueprints.opc.JNDINames;
import com.sun.j2ee.blueprints.processmanager.ejb.*;
import com.sun.j2ee.blueprints.servicelocator.*;
import com.sun.j2ee.blueprints.servicelocator.ejb.*;
import com.sun.j2ee.blueprints.opc.financial.*;
import com.sun.j2ee.blueprints.opc.utils.*;
import com.sun.j2ee.blueprints.opc.mailer.*;

/**
 * This is the PO handler that gets called by the
 * OPC Work Flow Manager. The handler first calls
 * the PO receiver that persists the PO. Then
 * it makes a call to the Credit Card Verifier
 * and then the CRM component. Finally it calls the 
 * order filler component. 
 */
public class POHandler {
    
    private ProcessManagerLocal processManager;
    private ServiceLocator sl;
    private POReceiver poReceiver;
    private CreditCardVerifier cardVerifier;
    
    public POHandler()throws HandlerException {
        try{
            sl = new ServiceLocator();
            cardVerifier = new CreditCardVerifier();
            poReceiver = new POReceiver();
            ProcessManagerLocalHome pmHome = (ProcessManagerLocalHome)sl.getLocalHome(JNDINames.PM_EJB);
            processManager = pmHome.create();
        } catch (Exception exe) {
            System.err.println(exe);
            throw new HandlerException("OPC Exception creating POHandler");  
        }
    }
    
    public void handle(Message message) throws HandlerException {

        PurchaseOrder po = null;
        String poID = null;
        String emailID = null;
        boolean sendMail = sl.getBoolean(JNDINames.SEND_MAIL);
        
        //extract the PO from the message
        try {
            if(message instanceof ObjectMessage){
                ObjectMessage objMsg = (ObjectMessage) message;
                po = (PurchaseOrder)objMsg.getObject();
            }    
            if(po != null){ 
                poID = po.getPoId();
                emailID = po.getEmailId();             
                //persist the PO
                poReceiver.persistPO(po);
                String lodgOrderStatus = po.getLodging() == null 
                                         ? OrderStatusNames.COMPLETED 
                                         : OrderStatusNames.PENDING;
                String actyOrderStatus = po.getActivities().length == 0
                                         ? OrderStatusNames.COMPLETED 
                                         : OrderStatusNames.PENDING;          
                String airlineOrderStatus = (po.getDepartureFlightInfo() == null)
                                            && (po.getReturnFlightInfo() == null) 
                                            ? OrderStatusNames.COMPLETED 
                                            : OrderStatusNames.PENDING;
                processManager.createManager(poID, OrderStatusNames.PENDING,
                                            actyOrderStatus, airlineOrderStatus, 
                                            lodgOrderStatus);
           
                //next call Credit Card Verifier                
                String creditCardXML = po.getCreditCard().toXML();
                boolean ccStatus = cardVerifier.verifyCreditCard(creditCardXML);       

                //change status and submit the order to order filler if credit
                //card is approved
                if (ccStatus){
                    processManager.updateStatus(poID, OrderStatusNames.APPROVED);                   
                    //call CRM 
                    if(sendMail){
                        String subject = "Your Adventure Builder order has been  approved";
                        String msg = "Your order (# " + poID + " ) has been approved.";
                        msg += " Thank you for shopping with us and we hope to see you again soon";
                        sendMail(emailID, subject, msg);                    
                    }   
                    //next call order filler
                    if (JMSUtils.sendMessage(JNDINames.ORDER_FILLER_MDB_QUEUE, 
                                         JNDINames.DOC_TYPE, JNDINames.PO_DOCUMENT,
                                         (Object)po) == false){
                        processManager.updateStatus(poID, OrderStatusNames.ORDER_FILLER_ERROR);
                        processManager.updateOrderErrorStatus(poID, true);
                    } else {
                        processManager.updateStatus(poID, OrderStatusNames.SUBMITTED);
                    }   
                } else { 
                    processManager.updateStatus(poID, OrderStatusNames.DENIED);
                    //call CRM
                    if(sendMail){ 
                        String subject = "Your Adventure Builder order has been  denied";
                        String msg = "Your order (# " + poID + " ) has been denied. ";
                        msg += " Thank you for shopping with us and we hope to see you again soon";
                        sendMail(emailID, subject, msg);                      
                    }
                }
            }    
        } catch (CreateException ce) {
            //call CRM to notify the customer
            if(sendMail){
                String subject = "Problems processing your Adventure Builder order";
                String msg = "We had problems processing your Adventure Builder order.";
                msg += " Please resubmit the order";
                sendMail(emailID, subject, msg);             
            }
        } catch (RemoteException re) {
            //call process manager and set error status
            try{  
                processManager.updateStatus(poID,OrderStatusNames.PAYMENT_PROCESSING_ERROR);
                processManager.updateOrderErrorStatus(poID, true);
            } catch(FinderException fe){
                System.err.println(fe);
            }    
        } catch (Exception exe) {
            System.err.println(exe);
            throw new HandlerException("OPC Exception handling PO");
        }  
    }
    
    private void sendMail(String emailID, String subject, String msg){
        Mail mail = new Mail(emailID, subject, msg);                
        String xmlMail = mail.toXML();
        JMSUtils.sendMessage(JNDINames.CRM_MDB_QUEUE, 
                            JNDINames.DOC_TYPE, JNDINames.MAIL_DOCUMENT,
                            xmlMail);
    }    
}
