/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.system.handler;

import org.apache.avalon.excalibur.system.Container;
import org.apache.avalon.excalibur.system.RoleManager;
import org.apache.avalon.framework.activity.Startable;
import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.excalibur.logger.LoggerManager;
import org.apache.avalon.framework.logger.Logger;

/**
 * The ThreadSafeComponentHandler to make sure components are initialized
 * and destroyed correctly.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version CVS $Revision: 1.3 $ $Date: 2002/01/29 16:18:53 $
 * @since 4.0
 */
public class PerThreadComponentHandler implements ComponentHandler {
    private ThreadLocalComponent m_instance;
    private final ComponentFactory m_factory;
    private boolean m_initialized = false;
    private boolean m_disposed = false;
    private final Logger m_logger;

    /**
     * Create a ComponentHandler that takes care of hiding the details of
     * whether a Component is ThreadSafe, Poolable, or SingleThreaded.
     * It falls back to SingleThreaded if not specified.
     */
    protected PerThreadComponentHandler ( final Class componentClass,
                                          final Configuration config,
                                          final ComponentManager manager,
                                          final Context context )
        throws Exception
    {
        RoleManager roles = (RoleManager) context.get( Container.ROLE_MANAGER );
        LoggerManager logkit = (LoggerManager) context.get( Container.LOGGER_MANAGER );

        m_factory = new ComponentFactory( componentClass, config, manager, context, roles, logkit );
        m_instance = new ThreadLocalComponent( m_factory );
        m_logger = logkit.getLoggerForCategory("system.handler.perthread");
    }

    public boolean isInitialized()
    {
        return m_initialized;
    }

    /**
     * Initialize the ComponentHandler.
     */
    public void initialize()
    throws Exception
    {
        if( m_initialized )
        {
            return;
        }

        if (m_logger.isDebugEnabled())
        {
            if (this.m_factory != null)
            {
                m_logger.debug("ComponentHandler initialized for: " + this.m_factory.getCreatedClass().getName());
            }
            else
            {
               m_logger.debug("ComponentHandler initialized for: " + this.m_instance.getClass().getName());
            }
        }

        m_initialized = true;
    }

    /**
     * Get a reference of the desired Component
     */
    public final Component get()
        throws Exception
    {
        if( ! m_initialized )
        {
            throw new IllegalStateException( "You cannot get a component from an uninitialized holder." );
        }

        if( m_disposed )
        {
            throw new IllegalStateException( "You cannot get a component from a disposed holder" );
        }

        return m_instance.getComponent();
    }

    /**
     * Return a reference of the desired Component
     */
    public void put( final Component component )
    {
        if( !m_initialized )
        {
            throw new IllegalStateException( "You cannot put a component in an uninitialized holder." );
        }
    }

    /**
     * Dispose of the ComponentHandler and any associated Pools and Factories.
     */
    public void dispose()
    {
        try {
            m_factory.decommission( m_instance.getComponent() );

            m_instance = null;
        }
        catch( final Exception e )
        {
            if (m_logger.isWarnEnabled())
            {
                m_logger.warn( "Error decommissioning component: " +
                                m_factory.getCreatedClass().getName(), e );
            }
        }

        m_disposed = true;
    }

    private final static class ThreadLocalComponent extends ThreadLocal
    {
        private final ComponentFactory m_factory;

        protected ThreadLocalComponent(ComponentFactory factory)
        {
            m_factory = factory;
        }

        protected Object initialValue()
        {
            try
            {
                return m_factory.newInstance();
            }
            catch (Exception e)
            {
                return null;
            }
        }

        public Component getComponent()
        {
            return (Component) this.get();
        }
    }
}
