/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.profile;


/**
 * The ProfileReport interface is an event based reporting mechanism.  It is
 * modeled after the Simple API for XML (SAX), and is effectively simple itself.
 * The ProfileReport may or may not serialize information to a persistent source,
 * but this is how the Profiler publishes it's information.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 */
public interface ProfileReport {

    /**
     * The report is started with this method.  This is called once at the
     * beginning of a profile run, and signifies to the ProfileReport object
     * that it should prepare to handle more requests.
     */
    void startReport();

    /**
     * The addGroup method is called when using cascading Profilables.  The
     * algorithm is that each group is fully evaluated until there are no more
     * subgroups (i.e. the column names).  In order to grasp how it should be
     * called, please reference the following group heirarchy:
     *
     * <p><center><img src="doc-files/call-order.png"/></center></p>
     *
     * <p>
     *   In this hierarchy, the call order would be like this:
     * </p>
     * <pre>
     *   report.addGroup("G1", {"C1", "C2"});
     *   report.addGroup("G2", {"C3", "C4"});
     * </pre>
     * <p>
     *   In the event that there are subgroups before we get to the column name,
     *   the tree is fully evaluated to the end.  For instance, let us assume
     *   that "C2" has two column names "C2a" and "C2b".  The call order would
     *   be like this:
     * </p>
     * <pre>
     *   report.addGroup("G1", {"C1", "C2"});
     *   // no sub groups for "C1"
     *   report.addGroup("C2", {"C2a", "C2b"});
     *   report.addGroup("G2", {"C3", "C4"});
     * </pre>
     * <p>
     *   When you align the last line of names, this is the order of samples
     *   passed in the sample() method.  In this last example, the order would
     *   be "C1", "C2a", "C2b", "C3", "C4".  It is imperitive that the samples
     *   gathered are in this precise order.
     * </p>
     */
    void addGroup( String groupName, String[] subgroupNames );

    /**
     * This method is called each time the Profiler takes a sample.  It has the
     * timestamp that the samples were called, and an array of the actual
     * values.
     */
    void sample( long timestamp, int[] values );

    /**
     * This method is called once at the end of a profile run, and signifies
     * that the ProfileReport object may close it's resources, and no other
     * events will be called.
     */
    void endReport();
}