/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.pool;

import java.security.SecureRandom;
import org.apache.avalon.excalibur.command.RepeatedCommand;
import org.apache.avalon.excalibur.event.Queue;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * This interface is for a PoolManager that creates pools that are managed
 * asynchronously.  The contract is that the controller type is specified in
 * the constructor.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version CVS $Revision: 1.5 $ $Date: 2002/01/28 19:29:47 $
 * @since 4.1
 */
public class DefaultPoolManager implements PoolManager
{
    private final long          m_managerKey;
    private final SecureRandom  m_keyGenerator;
    private final HashMap       m_keyMap = new HashMap();
    private final HashMap       m_factoryMap = new HashMap();

    public DefaultPoolManager()
    {
        this( null );
    }

    public DefaultPoolManager( final Queue commandQueue )
    {
        m_keyGenerator = new SecureRandom();
        m_managerKey = m_keyGenerator.nextLong();

        if ( null != commandQueue )
        {
            try
            {
                commandQueue.enqueue( new PoolManagerCommand( m_keyMap ) );
            }
            catch (Exception e)
            {
                // ignore silently for now
            }
        }
    }

    /**
     * Return a managed pool that has a controller.
     */
    public Pool getManagedPool( ObjectFactory factory, int initialEntries )
        throws Exception
    {
        ManagablePool pool = (ManagablePool) m_factoryMap.get( factory );

        if ( null == pool )
        {
            final long poolKey = getKey();
            pool =  new VariableSizePool( factory, initialEntries, poolKey );
            m_keyMap.put( pool, new Long( poolKey ) );
            m_factoryMap.put( factory, pool );
        }

        return pool;
    }

    /**
     * Return a new key for the pool and controller.
     */
    private final long getKey()
    {
        return m_keyGenerator.nextLong();
    }

    private final static class PoolManagerCommand implements RepeatedCommand
    {
        private final Map m_map;
        private final int m_min = 4;
        private final int m_max = 256;
        private final int m_grow = 4;

        protected PoolManagerCommand( Map map )
        {
            m_map = map;
        }

        public long getDelayInterval()
        {
            return 10 * 1000L;
        }

        public long getRepeatInterval()
        {
            return 10 * 1000L;
        }

        public int getNumberOfRepeats()
        {
            return 0;
        }

        public void execute()
            throws Exception
        {
            Iterator i = m_map.keySet().iterator();

            while (i.hasNext())
            {
                ManagablePool pool = (ManagablePool) i.next();
                long key = ((Long) m_map.get( pool )).longValue();
                int size = pool.size( key );

                if ( size < m_min )
                {
                    pool.grow( m_grow, key );
                }

                if ( size > m_max )
                {
                    pool.shrink( m_grow, key );
                }
            }
        }
    }
}
