/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for the raw MIDI interface
 */

#include "driver.h"
#include "midi.h"

#ifdef GUSCFG_MIDI

static unsigned int midi_file_flags( unsigned short f_flags )
{
  switch ( f_flags & O_ACCMODE ) {
    case O_WRONLY:	return GUS_MIDIF_USED_OUT;
    case O_RDONLY:	return GUS_MIDIF_USED_IN;
    default:		return GUS_MIDIF_USED;
  }
}

static void gus_midi_raw_rx_command( unsigned int device, unsigned char *buffer, unsigned int count )
{
  gus_card_t *card;
  struct GUS_STRU_MIDI *midi;
  
  card = gus_cards[ device >> 16 ];
  midi = &card -> midi[ device & 0xffff ];
  if ( midi -> rx_size - midi -> rx_used < count ) return;	/* shit, but we don't have free space */
  while ( count-- > 0 ) {
    midi -> rx_buf[ midi -> rx_head++ ] = *buffer++;
    midi -> rx_head %= midi -> rx_size;
    midi -> rx_used++;
  }
  if ( GETLOCK( card, midi_raw_in ) & WK_SLEEP ) {
    GETLOCK( card, midi_raw_in ) &= ~WK_SLEEP;
    WAKEUP( card, midi_raw_in );
  }
}

int gus_open_midi_raw( unsigned short minor, struct file *file )
{
  gus_card_t *card;
  struct GUS_STRU_MIDI *midi;
  unsigned int flags;
  int tmp, direct;

  direct = 0;
  if ( minor < GUS_MINOR_GBEGIN ) {
    switch ( minor & GUS_MINOR_OSS_MASK ) {
      case GUS_MINOR_MIDI: break;
      case GUS_MINOR_MIDI1: return -ENODEV;
      case GUS_MINOR_DMMIDI: direct = 1; break;
      case GUS_MINOR_DMMIDI1: return -ENODEV;
    }
  } else {
    if ( (minor & GUS_MINOR_GDEVMASK) == GUS_MINOR_GMIDI_RAW1 ) return -ENODEV;
  }
  minor >>= 4;
  minor &= 7;
  if ( minor > gus_cards_count ) return -ENODEV;
  card = gus_cards[ minor ];
  file -> private_data = card;
  midi = &card -> midi[ GUS_MIDID_UART ];
  flags = midi_file_flags( file -> f_flags );
  if ( ( midi -> flags & flags ) != 0 ) return -EBUSY;
  if ( flags & GUS_MIDIF_USED_OUT ) {
    if ( !(midi -> flags & GUS_MIDIF_USED_OUT ) ) {
      midi -> tx_size = GUS_MIDI_QUEUE_SIZE;
      midi -> tx_used = midi -> tx_head = midi -> tx_tail = 0;
      if ( ( tmp = midi -> init_write( card ) ) < 0 ) return tmp;
      midi -> flags |= GUS_MIDIF_USED_OUT | GUS_MIDIF_RAW_OUT;
      if ( direct ) midi -> flags |= GUS_MIDIF_DIRECT_OUT;
    }
  }
  if ( flags & GUS_MIDIF_USED_IN ) {
    if ( !(midi -> flags & GUS_MIDIF_USED_IN) ) {
      midi -> rx.size = GUS_MIDI_BUF_SIZE;
      midi -> rx.dev = ( minor /* aka card # */ << 16 ) | GUS_MIDID_UART;
      midi -> rx.ack = gus_midi_raw_rx_command;
      midi -> rx_size = GUS_MIDI_QUEUE_SIZE;
      midi -> rx_used = midi -> rx_head = midi -> rx_tail = 0;
      gus_midi_cmd_init( &midi -> rx, 1 );
      if ( ( tmp = midi -> init_read( card ) ) < 0 ) {
        if ( flags & GUS_MIDIF_USED_OUT ) {
          midi -> done_write( card );
          midi -> flags &= ~(GUS_MIDIF_USED_OUT | GUS_MIDIF_RAW_OUT | GUS_MIDIF_DIRECT_OUT);
        }
        return tmp;
      }
      midi -> flags |= GUS_MIDIF_USED_IN | GUS_MIDIF_RAW_IN;
      if ( direct ) midi -> flags |= GUS_MIDIF_DIRECT_IN;
      midi -> prech_timeout = 0;
      midi -> thru_mask = 0;
    }
  }
  MOD_INC_USE_COUNT;
  return 0;
}

void gus_release_midi_raw( struct file *file )
{
  gus_card_t *card;
  struct GUS_STRU_MIDI *midi;
  unsigned int flags;
  char sense[ 1 ];

  card = (gus_card_t *)file -> private_data;
  midi = &card -> midi[ GUS_MIDID_UART ];
  flags = midi_file_flags( file -> f_flags );
  if ( flags & GUS_MIDIF_USED_OUT ) {
    sense[ 0 ] = GUS_MCMD_COMMON_SENSING;
    midi -> putcmd( card, sense, 1 );
    midi -> done_write( card );
    midi -> flags &= ~(GUS_MIDIF_USED_OUT | GUS_MIDIF_RAW_OUT | GUS_MIDIF_DIRECT_OUT);
  }
  if ( flags & GUS_MIDIF_USED_IN ) {
    midi -> done_read( card );
    midi -> flags &= ~(GUS_MIDIF_USED_IN | GUS_MIDIF_RAW_IN | GUS_MIDIF_DIRECT_IN);
  }
  MOD_DEC_USE_COUNT;
}

int gus_ioctl_midi_raw( struct file *file, unsigned int cmd, unsigned long arg )
{
  gus_card_t *card;
  struct GUS_STRU_MIDI *midi;
  unsigned int flags;
  int val;

#if 0
  printk( "ioctl = 0x%x\n", cmd );
#endif
  card = (gus_card_t *)file -> private_data;
  midi = &card -> midi[ GUS_MIDID_UART ];
  flags = midi_file_flags( file -> f_flags );
  if ( ((cmd >> 8) & 0xff) == 'C' ) return -ENXIO;
  switch ( cmd ) {
    case SNDCTL_MIDI_PRETIME:
      if ( !(flags & GUS_MIDIF_USED_IN) ) return -EINVAL;
      val = IOCTL_IN( arg );
      if ( val < 0 ) val = 0;
      val = ( HZ * val ) / 10;
      midi -> prech_timeout = val;
      break;
  }
  return -EINVAL;
}

int gus_read_midi_raw( struct file *file, char *buf, int count )
{
  gus_card_t *card;
  struct GUS_STRU_MIDI *midi;
  unsigned int flags;
  int size;

  if ( VERIFY_AREA( VERIFY_WRITE, buf, count ) ) return -EIO;
  card = (gus_card_t *)file -> private_data;
  midi = &card -> midi[ GUS_MIDID_UART ];
  flags = midi_file_flags( file -> f_flags );
  if ( !(flags & GUS_MIDIF_USED_IN) ) return -ENODEV;
  size = 0;
  while ( !midi -> rx_used ) {
    GETLOCK( card, midi_raw_in ) |= WK_SLEEP;
    SLEEP( card, midi_raw_in, midi -> prech_timeout > 0 ? midi -> prech_timeout : HZ * 60 * 60 );
    GETLOCK( card, midi_raw_in ) &= ~WK_SLEEP;
    if ( TABORT( card, midi_raw_in ) ) return -EINTR;
    if ( TIMEOUT( card, midi_raw_in ) ) return 0; 
  }
  while ( midi -> rx_used > 0 && count > 0 ) {
    put_fs_byte( midi -> rx_buf[ midi -> rx_tail++ ], buf++ );
    midi -> rx_tail %= midi -> rx_size;
    midi -> rx_used--;
    size++;
    count--;
  }
  return size;
}

int gus_write_midi_raw( struct file *file, char *buf, int count )
{
  gus_card_t *card;
  struct GUS_STRU_MIDI *midi;
  unsigned int flags;
  int i, size;
  char buffer[ 32 ];

  if ( VERIFY_AREA( VERIFY_READ, buf, count ) ) return -EIO;
  card = (gus_card_t *)file -> private_data;
  midi = &card -> midi[ GUS_MIDID_UART ];
  flags = midi_file_flags( file -> f_flags );
  if ( !(flags & GUS_MIDIF_USED_OUT) ) return -ENODEV;
  size = 0;
  while ( count > 0 ) {
    i = count > sizeof( buffer ) ? sizeof( buffer ) : count;
    MEMCPY_FROMFS( buffer, buf, i );
    while ( midi -> tx_size - midi -> tx_used < i ) {
      GETLOCK( card, midi_raw_out ) |= WK_SLEEP;
      SLEEP( card, midi_raw_out, HZ * 60 * 60 );
      GETLOCK( card, midi_raw_out ) &= ~WK_SLEEP;
      if ( TABORT( card, midi_raw_out ) ) return -EINTR;
      if ( TIMEOUT( card, midi_raw_out ) ) return 0; 
    }
    midi -> putcmd( card, buffer, i );
    count -= i;
    size += i;
  }
  return size;
}

#ifdef GUS_POLL

unsigned int gus_poll_midi_raw( struct file *file, poll_table *wait )
{
  gus_card_t *card;
  struct GUS_STRU_MIDI *midi;
  unsigned int xflags;
  unsigned long flags;
  unsigned int mask;
  
  card = (gus_card_t *)file -> private_data;
  midi = &card -> midi[ GUS_MIDID_UART ];
  xflags = midi_file_flags( file -> f_flags );
  CLI( &flags );
  if ( xflags & GUS_MIDIF_USED_IN ) {
    GETLOCK( card, midi_raw_in ) |= WK_SLEEP;
    SLEEP_POLL( card, midi_raw_in, wait );
  }
  if ( xflags & GUS_MIDIF_USED_OUT ) {
    GETLOCK( card, midi_raw_out ) |= WK_SLEEP;
    SLEEP_POLL( card, midi_raw_out, wait );
  }
  STI( &flags );
  
  mask = 0;
  if ( xflags & GUS_MIDIF_USED_IN ) {
    if ( midi -> rx_used )
      mask |= POLLIN | POLLRDNORM;
  }
  if ( xflags & GUS_MIDIF_USED_OUT ) {
    if ( midi -> tx_used < midi -> tx_size )
      mask |= POLLOUT | POLLWRNORM;
  }

  return mask;
}

#else

int gus_select_midi_raw( struct file *file, int sel_type, select_table *wait )
{
  gus_card_t *card;
  struct GUS_STRU_MIDI *midi;
  unsigned int xflags;
  unsigned long flags;
  
  card = (gus_card_t *)file -> private_data;
  midi = &card -> midi[ GUS_MIDID_UART ];
  xflags = midi_file_flags( file -> f_flags );
  switch ( sel_type ) {
    case SEL_IN:
      if ( !(xflags & GUS_MIDIF_USED_IN) ) return -ENODEV;
      CLI( &flags );
      if ( !midi -> rx_used ) {
        GETLOCK( card, midi_raw_in ) |= WK_SLEEP;
        SLEEPS( card, midi_raw_in, wait );
        STI( &flags );
        return 0;
      }
      GETLOCK( card, midi_raw_in ) &= ~WK_SLEEP;
      STI( &flags );
      return 1;
    case SEL_OUT:
      if ( !(xflags & GUS_MIDIF_USED_OUT) ) return -ENODEV;
      CLI( &flags );
      if ( midi -> tx_used >= midi -> tx_size ) {
        GETLOCK( card, midi_raw_out ) |= WK_SLEEP;
        SLEEPS( card, midi_raw_out, wait );
        STI( &flags );
        return 0;
      }
      GETLOCK( card, midi_raw_out ) &= ~WK_SLEEP;
      STI( &flags );
      return 1;
    case SEL_EX:
      break;
  }
  return 0;
}

#endif

/*
 *  /dev/dmfmX
 */

#ifdef GUSCFG_OSS

int gus_open_fm_raw( unsigned short minor, struct file *file )
{
  PRINTK( "gus: /dev/dmfd0 device isn't supported...\n" );
  MOD_INC_USE_COUNT;
  return 0;
}

void gus_release_fm_raw( struct file *file )
{
  MOD_DEC_USE_COUNT;
}

int gus_ioctl_fm_raw( struct file *file, unsigned int cmd, unsigned long arg )
{
  return 0;
}

#endif /* GUSCFG_OSS */

#endif /* GUSCFG_MIDI */
