/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  GF1 PATCH INTERFACE
 */

#include "driver.h"
#include "tables.h"

static void voice_interrupt_wave( gus_card_t *card, gus_voice_t *voice );
static void voice_interrupt_volume( gus_card_t *card, gus_voice_t *voice );
static void voice_interrupt_effect( gus_card_t *card, gus_voice_t *voice );
static void voice_start( gus_card_t *card, gus_voice_t *voice );
static void voice_go( gus_card_t *card, gus_voice_t *voice );
static void voice_stop( gus_card_t *card, gus_voice_t *voice, unsigned char mode );
static void voice_control( gus_card_t *card, gus_voice_t *voice, unsigned char control );
static void voice_freq( gus_card_t *card, gus_voice_t *voice, unsigned int freq );
static void voice_volume( gus_card_t *card, gus_voice_t *voice, unsigned short volume );
static void voice_loop( gus_card_t *card, gus_voice_t *voice, unsigned int start, unsigned int end );
static void voice_ramp( gus_card_t *card, gus_voice_t *voice, unsigned char start, unsigned char end, unsigned char rate, unsigned char control );
static void voice_pos( gus_card_t *card, gus_voice_t *voice, unsigned int position );
static void voice_pan( gus_card_t *card, gus_voice_t *voice, unsigned short pan );
static void voice_lfo( gus_card_t *card, gus_voice_t *voice, unsigned char *data );
static void voice_private1( gus_card_t *card, gus_voice_t *voice, unsigned char *data );

static struct GUS_STRU_INSTRUMENT_VOICE_COMMANDS voice_commands = {
  voice_interrupt_wave,
  voice_interrupt_volume,
  voice_interrupt_effect,
  voice_start,
  voice_go,
  voice_stop,
  voice_control,
  voice_freq,
  voice_volume,
  voice_loop,
  voice_ramp,
  voice_pos,
  voice_pan,
  voice_lfo,
  voice_private1
};

static void note_stop( gus_card_t *card, gus_voice_t *voice, int wait );
static void note_wait( gus_card_t *card, gus_voice_t *voice );
static void note_off( gus_card_t *card, gus_voice_t *voice );
static void note_volume( gus_card_t *card, gus_voice_t *voice );
static void note_pitchbend( gus_card_t *card, gus_voice_t *voice );
static void note_vibrato( gus_card_t *card, gus_voice_t *voice );
static void note_tremolo( gus_card_t *card, gus_voice_t *voice );

static struct GUS_STRU_INSTRUMENT_NOTE_COMMANDS note_commands = {
  note_stop,
  note_wait,
  note_off,
  note_volume,
  note_pitchbend,
  note_vibrato,
  note_tremolo
};

static void chn_trigger_down( gus_card_t *card, gus_channel_t *channel, gus_instrument_t *instrument, unsigned char note, unsigned char velocity, unsigned char priority );
static void chn_trigger_up( gus_card_t *card, gus_note_t *note );
static void chn_control( gus_card_t *card, gus_channel_t *channel, unsigned short p1, unsigned short p2 );

static struct GUS_STRU_INSTRUMENT_CHANNEL_COMMANDS channel_commands = {
  chn_trigger_down,
  chn_trigger_up,
  chn_control
};

static void do_volume_envelope( gus_card_t *card, gus_voice_t *voice );
static void do_pan_envelope( gus_card_t *card, gus_voice_t *voice );

/*
 *
 */
 
static void voice_interrupt_wave( gus_card_t *card, gus_voice_t *voice )
{
  unsigned int old_flags;

#if 0
  printk( "wave irq %i!!!, pos = 0x%x, vol = 0x%x, flags = 0x%x\n", voice -> number, gus_i_read_addr( card, GF1_VA_CURRENT, 1 ), gus_i_read16( card, GF1_VW_VOLUME ), voice -> flags );
#endif
  gus_gf1_stop_voice( card, voice -> number );
  old_flags = voice -> flags;
  if ( old_flags & VFLG_USED )
    {
      if ( old_flags & VFLG_DYNAMIC )
        gus_voice_and_note_free( card, voice );
       else
        voice -> flags &= ~VFLG_RUNNING;
    }
  if ( old_flags & VFLG_SHUTDOWN )
    {
      voice -> flags &= ~VFLG_SHUTDOWN;
      voice_go( card, voice );
    }
}

static void voice_interrupt_volume( gus_card_t *card, gus_voice_t *voice )
{
#if 0
  printk( "volume irq %i!!!, pos = 0x%x, flags = 0x%x\n", voice -> number, gus_i_read_addr( card, GF1_VA_CURRENT, 1 ), voice -> flags );
#endif
  if ( voice -> flags & (VFLG_RUNNING|VFLG_SHUTDOWN) )
    do_volume_envelope( card, voice );
   else
    gus_gf1_stop_voice( card, voice -> number );
}

static void voice_interrupt_effect( gus_card_t *card, gus_voice_t *voice )
{
  if ( voice -> flags & (VFLG_RUNNING|VFLG_SHUTDOWN) )
    if ( voice -> flags & VFLG_EFFECT_TIMER1 )
      do_pan_envelope( card, voice );
}

/*
 *
 */

static void stop_note_voice( gus_card_t *card, gus_voice_t *voice )
{
  gus_engine_midi_go_note_stop( card, voice -> note, 1 );
}

static void do_volume_envelope( gus_card_t *card, gus_voice_t *voice )
{
  unsigned long flags;
  unsigned short next = 0, rate = 0, old_volume;
  unsigned char program_next_ramp;
  gus_wave_t *wave;
  unsigned int old_flags;

  wave = voice -> wave;
  while ( 1 ) {
    program_next_ramp = 0;
#if 0
    printk( "voice = %i, state = %i, index = %i, wave = 0x%x, flg = 0x%x\n", voice -> number, voice -> data.patch.venv_state, voice -> data.patch.venv_index, (int)wave, wave -> data.patch.flags );
#endif
    switch ( voice -> data.patch.venv_state ) {
      case PATCH_BEFORE:
        voice -> data.patch.venv_next = 0;
        voice -> data.patch.venv_index = 0;
        voice -> data.patch.venv_state++;
        CLI( &flags );
        gf1_select_voice( card, voice -> number );
        gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
        gus_i_write16( card, GF1_VW_VOLUME, MIN_VOLUME );
        if ( !card -> gf1.enh_mode )
          {
            gus_delay( card );
            gus_i_write16( card, GF1_VW_VOLUME, MIN_VOLUME );
          }
        STI( &flags );
        break;
      case PATCH_ATTACK:
        program_next_ramp++;
        if ( wave -> data.patch.flags & GUS_WAVE_PATCH_ENVELOPE )
          {
            next = wave -> data.patch.envelope_offset[ voice -> data.patch.venv_index ];
            rate = wave -> data.patch.envelope_rate[ voice -> data.patch.venv_index ];
            if ( ++voice -> data.patch.venv_index > 2 )
              voice -> data.patch.venv_state++;	/* ok.. sustain */
          }
         else
          {
            next = 255;
            rate = card -> gf1.volume_ramp;
            voice -> data.patch.venv_state++;
          }
        break; 
      case PATCH_SUSTAIN:
        if ( wave -> data.patch.flags & GUS_WAVE_PATCH_ENVELOPE )
          {
            if ( !(wave -> data.patch.flags & GUS_WAVE_PATCH_SUSTAIN) )
              {
                voice -> data.patch.venv_state++;
                continue;
              }
          }
        CLI( &flags );
        gf1_select_voice( card, voice -> number );
        gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
        gus_i_write16( card, GF1_VW_VOLUME, ( (int)voice -> data.patch.volume * (int)voice -> data.patch.venv_next ) / 255 );
        STI( &flags );
        return;
      case PATCH_VAR_RELEASE:
        if ( wave -> data.patch.flags & GUS_WAVE_PATCH_ENVELOPE )
          {
            program_next_ramp++;
            if ( voice -> data.patch.venv_index < 3 )
              voice -> data.patch.venv_index = 3;
            next = wave -> data.patch.envelope_offset[ voice -> data.patch.venv_index ];
            rate = wave -> data.patch.envelope_rate[ voice -> data.patch.venv_index ];
            if ( ++voice -> data.patch.venv_index > 5 )
              voice -> data.patch.venv_state++;	/* ok.. final release */
          }
         else
          voice -> data.patch.venv_state++;	/* force final release */
        break;
      case PATCH_FINAL_RELEASE:
        program_next_ramp++;
        next = 0;
        rate = card -> gf1.volume_ramp;
        voice -> data.patch.venv_state++;
        break; 
      case PATCH_DONE:
        gus_gf1_stop_voice( card, voice -> number );
        old_flags = voice -> flags;
        if ( old_flags & VFLG_DYNAMIC )
          gus_voice_and_note_free( card, voice );
         else
          voice -> flags &= ~VFLG_RUNNING;
        if ( old_flags & VFLG_SHUTDOWN )
          {
            voice -> flags &= ~VFLG_SHUTDOWN;
            voice_go( card, voice );
          }
        return;
      case PATCH_VOLUME:
        program_next_ramp++;
        next = voice -> data.patch.venv_next;
        rate = voice -> data.patch.venv_rate;
        voice -> data.patch.venv_state = voice -> data.patch.venv_state_prev;
        break;
    }
    if ( !program_next_ramp ) continue;
    voice -> data.patch.venv_next = next;
    voice -> data.patch.venv_rate = rate;
    CLI( &flags );
    gf1_select_voice( card, voice -> number );
    gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
    old_volume = gus_i_read16( card, GF1_VW_VOLUME ) >> 8;
    STI( &flags );
#if 0
    printk( "voice = %i, rate = 0x%x, next = 0x%x, old_volume = 0x%x\n", voice -> number, rate, next, old_volume );
#endif
    if ( !rate ) continue;
    next = ( ( (int)voice -> data.patch.volume * (int)next ) / 255 ) >> 8;
#if 0
    printk( "voice = %i, volume = 0x%x, nextf = 0x%x\n", voice -> number, voice -> data.patch.volume, next );
#endif
    if ( old_volume < MIN_OFFSET ) old_volume = MIN_OFFSET;
    if ( next < MIN_OFFSET ) next = MIN_OFFSET;
    if ( next > MAX_OFFSET ) next = MAX_OFFSET;
    if ( old_volume == next ) continue;
    voice -> data.patch.volume_control &= ~0xc3;
    voice -> data.patch.volume_control |= 0x20;
    CLI( &flags );
    gf1_select_voice( card, voice -> number );
    if ( old_volume > next )
      {
        gus_write8( card, GF1_VB_VOLUME_START, next );
        gus_write8( card, GF1_VB_VOLUME_END, old_volume );
        voice -> data.patch.volume_control |= 0x40;
      }
     else
      {
        gus_write8( card, GF1_VB_VOLUME_START, old_volume );
        gus_write8( card, GF1_VB_VOLUME_END, next );
      }
    gus_write8( card, GF1_VB_VOLUME_RATE, rate );
    gus_write8( card, GF1_VB_VOLUME_CONTROL, voice -> data.patch.volume_control );
    if ( !card -> gf1.enh_mode )
      {
        gus_delay( card );
        gus_write8( card, GF1_VB_VOLUME_CONTROL, voice -> data.patch.volume_control );
      }
    STI( &flags );
    return;
  }
}

static void do_pan_envelope( gus_card_t *card, gus_voice_t *voice )
{
  unsigned long flags;
  unsigned char old_pan;

  if ( card -> gf1.enh_mode )
    {
      voice -> flags &= ~(VFLG_EFFECT_TIMER1|VFLG_PAN);
      return;
    }
  if ( !(voice -> flags & VFLG_PAN) )
    voice -> flags |= VFLG_EFFECT_TIMER1|VFLG_PAN;
  CLI( &flags );
  gf1_select_voice( card, voice -> number );
  old_pan = gus_i_read8( card, GF1_VB_PAN ) & 0x0f;
  if ( old_pan > voice -> data.patch.gf1_pan ) old_pan--;
  if ( old_pan < voice -> data.patch.gf1_pan ) old_pan++;
  gus_write8( card, GF1_VB_PAN, old_pan );
  STI( &flags );
  if ( old_pan == voice -> data.patch.gf1_pan )
    voice -> flags &= ~(VFLG_EFFECT_TIMER1|VFLG_PAN);
}

static void set_enhanced_pan( gus_card_t *card, gus_voice_t *voice )
{
  unsigned long flags;
  unsigned short vlo, vro, pan;
  
  pan = voice -> data.patch.pan;
  vlo = ATTEN( (ATTEN_TABLE_SIZE-1) - pan );
  vro = ATTEN( pan );
  if ( pan != ATTEN_TABLE_SIZE - 1 && pan != 0 )
    {
      vlo >>= 1;
      vro >>= 1;
    }
  vlo <<= 4;
  vro <<= 4;
  if ( !(voice -> flags & VFLG_WAIT_FOR_START) && (voice -> flags & VFLG_RUNNING) )
    {
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      if ( voice -> data.patch.vlo != vlo )
        gus_write16( card, GF1_VW_OFFSET_LEFT_FINAL, voice -> data.patch.vlo );
      if ( voice -> data.patch.vro != vro );
        gus_write16( card, GF1_VW_OFFSET_RIGHT_FINAL, voice -> data.patch.vro );
      STI( &flags );
    }
  voice -> data.patch.vlo = vlo;
  voice -> data.patch.vro = vro;
}

static void change_effect_volume( gus_card_t *card, gus_voice_t *voice, int both )
{
  unsigned short atten;
  gus_channel_t *channel;
  unsigned char rev_depth, chorus_depth;

  if ( voice -> flags & VFLG_DYNAMIC )
    {
      if ( !voice -> note ) return;
      channel = voice -> note -> channel;
      if ( !channel ) return;
      atten = MAX_VOLUME - ( voice -> data.patch.volume >> 4 );
      rev_depth = chorus_depth = 0;
      if ( channel -> number == 9 )
        {
          gus_instrument_t *instrument;
     
          instrument = voice -> instrument;
          if ( !instrument ) return;
          switch ( instrument -> data.patch.effect1 ) {
            case GUS_INSTR_IE_REVERB:
              rev_depth = instrument -> data.patch.effect1_depth;
              break;
            case GUS_INSTR_IE_CHORUS:
              chorus_depth = instrument -> data.patch.effect1_depth;
              break;
          }
          switch ( instrument -> data.patch.effect2 ) {
            case GUS_INSTR_IE_REVERB:
              rev_depth = instrument -> data.patch.effect2_depth;
              break;
            case GUS_INSTR_IE_CHORUS:
              chorus_depth = instrument -> data.patch.effect2_depth;
              break;
          }
        }
      gus_set_effects_volume( card, voice,
  			atten + channel -> effect1_level + rev_depth,
  			atten + channel -> effect3_level + chorus_depth,
  			1 );
    }
}

/*
 *
 */

static void voice_start( gus_card_t *card, gus_voice_t *voice )
{
  if ( voice -> flags & VFLG_RUNNING )
    voice_stop( card, voice, 0x00 );
  voice -> flags &= VFLG_DYNAMIC | VFLG_SHUTDOWN | VFLG_WAIT_FOR_START1 | VFLG_OSS_PAN;
  voice -> flags |= VFLG_USED | VFLG_WAIT_FOR_START | VFLG_SELECT_WAVE;
  voice -> wave = NULL;
  voice -> instrument = NULL;
  voice -> data.patch.frequency = 0;
  voice -> data.patch.note = 0xffff;
  voice -> data.patch.pitchbend = 1024;
}

static void select_wave( gus_card_t *card, gus_voice_t *voice, unsigned int freq )
{
  unsigned char control, mode, format;
  gus_wave_t *wave;

  if ( !(voice -> flags & VFLG_SELECT_WAVE) ) return;
  voice -> wave = wave = gus_select_patch_wave( voice -> layer -> wave, freq );
  if ( !wave ) return;
  format = wave -> format;
  voice -> data.patch.loop_start = wave -> loop_start;
  if ( format & GUS_WAVE_LOOP )
    voice -> data.patch.loop_end = wave -> loop_end;
   else
    voice -> data.patch.loop_end = ( wave -> size << 4 ) - ( format & GUS_WAVE_16BIT ? 40 : 24 );
  voice -> data.patch.position =
    (format & GUS_WAVE_BACKWARD) && !wave -> start ?
      wave -> loop_end :
      wave -> start;
  control = 0x00;
  mode = 0x20;			/* enable offset registers */
  if ( format & GUS_WAVE_16BIT ) control |= 0x04;
  if ( format & GUS_WAVE_BACKWARD ) control |= 0x40;
  if ( format & GUS_WAVE_LOOP ) control |= 0x08; else control |= 0x20;
  if ( format & GUS_WAVE_BIDIR ) control |= 0x10;
  if ( format & GUS_WAVE_ULAW ) control |= 0x40;
  if ( format & GUS_WAVE_ROM ) control |= 0x80;
  voice -> data.patch.control = control;
  voice -> data.patch.mode = mode; 
  voice -> flags &= ~VFLG_SELECT_WAVE;
}

static void voice_go( gus_card_t *card, gus_voice_t *voice )
{
  gus_wave_t *wave;
  unsigned long flags;
  unsigned int begin;
  unsigned char w_16;

#if 0
  printk( "go - 1 - instrument = 0x%lx, flags = 0x%x\n", (long)voice -> instrument, voice -> flags );
#endif
  if ( !voice -> instrument ) return;
  if ( voice -> flags & VFLG_SHUTDOWN ) return;	/* wait for shutdown of old wave */
  if ( !(voice -> flags & VFLG_WAIT_FOR_START) ) return;
  voice -> flags &= ~VFLG_WAIT_FOR_START;
  if ( voice -> data.patch.note != 0xffff )
    voice_freq( card, voice, GUS_FREQ_NOTE | voice -> data.patch.note );
   else
    voice_freq( card, voice, GUS_FREQ_HZ | voice -> data.patch.frequency );
  if ( voice -> flags & VFLG_SELECT_WAVE )
    {
      if ( voice -> flags & VFLG_DYNAMIC )
        gus_voice_and_note_free( card, voice );
       else
        voice -> flags = 0;
      return;
    }
  voice_pan( card, voice, voice -> data.patch.voice_pan );
  wave = voice -> wave;
  w_16 = wave -> format & GUS_WAVE_16BIT;
  begin = wave -> begin.memory;
  CLI( &flags );
  gf1_select_voice( card, voice -> number );
  if ( ( gus_read8( card, GF1_VB_ADDRESS_CONTROL ) & 0x03 ) == 0 )
    {
      gus_ctrl_stop( card, GF1_VB_ADDRESS_CONTROL );
      gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
#if defined( GUSCFG_DEBUG_INSTRUMENTS ) && 0
      PRINTK( "patch_voice_go: -%i- running? flags = 0x%x, volume = 0x%x\n", voice -> number, voice -> flags, gus_read16( card, GF1_VW_VOLUME ) );
#endif
    }
  gus_write16( card, GF1_VW_FREQUENCY, voice -> fc_register + voice -> lfo_fc );
  if ( !card -> gf1.enh_mode )
    gus_write8( card, GF1_VB_PAN, voice -> data.patch.pan >> 3 );
   else
    {
      gus_write16( card, GF1_VW_OFFSET_LEFT, voice -> data.patch.vlo );
      gus_write16( card, GF1_VW_OFFSET_LEFT_FINAL, voice -> data.patch.vlo );
      gus_write16( card, GF1_VW_OFFSET_RIGHT, voice -> data.patch.vro );
      gus_write16( card, GF1_VW_OFFSET_RIGHT_FINAL, voice -> data.patch.vro );
      if ( !(voice -> flags & VFLG_DYNAMIC) )
        {
          gus_write8( card, GF1_VB_ACCUMULATOR, voice -> data.patch.effect_accumulator );
          gus_write16( card, GF1_VW_EFFECT_VOLUME, voice -> data.patch.effect_volume );
          gus_write16( card, GF1_VW_EFFECT_VOLUME_FINAL, voice -> data.patch.effect_volume );
        }
    }
  gus_write_addr( card, GF1_VA_START, begin + voice -> data.patch.loop_start, w_16 );
  gus_write_addr( card, GF1_VA_END, begin + voice -> data.patch.loop_end, w_16 );
  gus_write_addr( card, GF1_VA_CURRENT, begin + voice -> data.patch.position, w_16 );
  STI( &flags );
  voice -> data.patch.venv_state = PATCH_BEFORE;
  voice -> data.patch.volume_control = 0x03;
  do_volume_envelope( card, voice );
  change_effect_volume( card, voice, 1 );
  CLI( &flags );
  gf1_select_voice( card, voice -> number );
  if ( card -> gf1.enh_mode )
    gus_write8( card, GF1_VB_MODE, voice -> data.patch.mode );
  gus_write8( card, GF1_VB_ADDRESS_CONTROL, voice -> data.patch.control );
  if ( !card -> gf1.enh_mode )
    {
      gus_delay( card );
      gus_write8( card, GF1_VB_ADDRESS_CONTROL, voice -> data.patch.control );
    }
  STI( &flags );
#if 0
  gus_print_voice_registers( card );
#endif
#if 0
  printk( " -%i- go freq = %i, ofreq = %i, cfreq = %i, bend = %i\n", voice -> number, voice -> fc_register, voice -> data.patch.ofc_reg, gus_i_read16( card, GF1_VW_FREQUENCY ), voice -> data.patch.pitchbend );
#endif
  voice -> flags |= VFLG_RUNNING;
}

static void voice_stop( gus_card_t *card, gus_voice_t *voice, unsigned char mode )
{
  unsigned long flags;
  unsigned char control;

  if ( !(voice -> flags & VFLG_RUNNING) ) return;
  switch ( mode ) {
    case 0x00: 
      if ( card -> gf1.volume_ramp > 0 &&
           (voice -> flags & VFLG_RUNNING) && !(voice -> flags & VFLG_SHUTDOWN) )
        {
          if ( voice -> data.patch.venv_state < PATCH_FINAL_RELEASE )
            {
              voice -> data.patch.venv_state = PATCH_FINAL_RELEASE;
              do_volume_envelope( card, voice );
            }
          if ( voice -> flags & VFLG_RUNNING )
            voice -> flags |= VFLG_SHUTDOWN;
        }
      if ( !(voice -> flags & VFLG_SHUTDOWN) )
        {
          CLI( &flags );
          gus_gf1_stop_voice( card, voice -> number );
          gus_i_write16( card, GF1_VW_VOLUME, MIN_VOLUME );
          STI( &flags );
        }
      voice -> flags &= ~VFLG_RUNNING;
      break;
    case 0x01:
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      control = gus_read8( card, GF1_VB_ADDRESS_CONTROL );
      control &= ~(0x83 | 0x04);
      control |= 0x20;
      gus_write8( card, GF1_VB_ADDRESS_CONTROL, control );
      STI( &flags );
      break;
    case 0x02:
      voice -> flags |= VFLG_RELEASING;
      if ( voice -> data.patch.venv_state <= PATCH_SUSTAIN )
        {
          voice -> data.patch.venv_state = PATCH_VAR_RELEASE;
          do_volume_envelope( card, voice );
        }
      break;
  }
}

static void voice_control( gus_card_t *card, gus_voice_t *voice, unsigned char control )
{
}

static unsigned int note_to_freq( int note_num )
{

  /*
   * This routine converts a midi note to a frequency (multiplied by 1000)
   */

  int             note, octave, note_freq;
  int             notes[] =
  {
    261632, 277189, 293671, 311132, 329632, 349232,
    369998, 391998, 415306, 440000, 466162, 493880
  };

#define BASE_OCTAVE	5

  octave = note_num / 12;
  note = note_num % 12;

  note_freq = notes[note];

  if (octave < BASE_OCTAVE)
    note_freq >>= (BASE_OCTAVE - octave);
  else if (octave > BASE_OCTAVE)
    note_freq <<= (octave - BASE_OCTAVE);

  /*
   * note_freq >>= 1;
   */

  return note_freq;
}

static void voice_freq( gus_card_t *card, gus_voice_t *voice, unsigned int freq )
{
  unsigned long flags;
  short scale;
  gus_wave_t *wave;

#if 0
  printk ( " -%i- (%i) freq = 0x%x\n", voice -> number, card -> gf1.timer_abs_tick, freq );
#endif
  switch ( freq & GUS_FREQ_MASK ) {
    case GUS_FREQ_HZ:
      voice -> data.patch.frequency = freq & GUS_FREQ_DATA;
      select_wave( card, voice, voice -> data.patch.frequency );
      if ( voice -> flags & VFLG_WAIT_FOR_START ) return;
      voice -> data.patch.ofc_reg = gus_compute_freq( voice -> data.patch.frequency, gus_compute_patch_rate( voice -> wave ), card -> gf1.playback_freq );
      break;
    case GUS_FREQ_OSSNOTE:
    case GUS_FREQ_NOTE:
      voice -> data.patch.note = freq & GUS_FREQ_DATA;
      scale = 0;
      if ( ( freq & GUS_FREQ_MASK ) == GUS_FREQ_OSSNOTE )
        freq = note_to_freq( voice -> data.patch.note );
       else
        freq = gus_scale_table[ voice -> data.patch.note ];
      if ( freq > 2147483L )
        {
          freq >>= 1;
          scale++;
        }
      freq = ( freq * 1000 ) / 1024;
      while ( scale-- ) freq <<= 1;
      if ( !(voice -> flags & VFLG_RUNNING) )
        {
          voice -> layer = voice -> instrument -> info.layer;
          select_wave( card, voice, freq );
        }
      if ( voice -> flags & VFLG_WAIT_FOR_START ) return;
      if ( (wave = voice -> wave) == NULL ) return;
      if ( wave -> data.patch.scale_factor != 1024 )
        {
          short sn, sn_interp;
          unsigned int l1;
      
          l1 = ((unsigned int)voice -> data.patch.note - wave -> data.patch.scale_frequency) * wave -> data.patch.scale_factor;
          sn_interp = (unsigned short)(l1 & ((1 << 10) - 1));
          sn = (unsigned short)((l1 >> 10) + wave -> data.patch.scale_frequency);
          if ( ( freq & GUS_FREQ_MASK ) == GUS_FREQ_OSSNOTE )
            l1 = note_to_freq( sn );
           else
            l1 = gus_scale_table[ sn ];
          voice -> data.patch.ofc_reg =
            (unsigned short)(((((61L * sn_interp)>>10) + 1024L) *
              (unsigned int)
              gus_compute_freq( l1,
                                gus_compute_patch_rate( wave ),
                                card -> gf1.playback_freq )) >> 10);
#if 0
          printk( "instr = %i, ofc_reg = 0x%x, sn = %i\n", voice -> instrument -> number.instrument, voice -> data.patch.ofc_reg, sn );
#endif
        }
       else
        voice -> data.patch.ofc_reg = gus_compute_freq( gus_scale_table[ voice -> data.patch.note ], gus_compute_patch_rate( wave ), card -> gf1.playback_freq );
      break;
    case GUS_FREQ_PITCHBEND:
      voice -> data.patch.pitchbend = freq & GUS_FREQ_DATA;
      if ( voice -> flags & VFLG_WAIT_FOR_START ) return;
      break;
    default:
      return;
  }
  if ( voice -> data.patch.pitchbend == 1024 )
    voice -> fc_register = voice -> data.patch.ofc_reg;
   else
    {
      unsigned int newfc;
             
      newfc = (((unsigned int)voice -> data.patch.ofc_reg * (unsigned int)voice -> data.patch.pitchbend) >> 10);
      if ( newfc > 65535 ) newfc = 65535;
      voice -> fc_register = newfc;
    }
#if 0
  printk( " -%i- set freq = %i, ofreq = %i, bend = %i\n", voice -> number, voice -> fc_register, voice -> data.patch.ofc_reg, voice -> data.patch.pitchbend );
#endif
  if ( voice -> flags & VFLG_WAIT_FOR_START ) return;
  if ( voice -> flags & VFLG_RUNNING )
    {
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      gus_i_write16( card, GF1_VW_FREQUENCY, voice -> fc_register + voice -> lfo_fc );
      STI( &flags );
    }
}

static void voice_volume( gus_card_t *card, gus_voice_t *voice, unsigned short volume )
{
  unsigned short new_volume;

  new_volume = gus_translate_voice_volume( card, volume ) << 4;
  if ( new_volume == voice -> data.patch.volume ) return;	/* speedup */
  voice -> data.patch.volume = new_volume;
  if ( voice -> flags & VFLG_RUNNING )
    {
      voice -> data.patch.venv_state_prev = voice -> data.patch.venv_state;
      voice -> data.patch.venv_state = PATCH_VOLUME;
      do_volume_envelope( card, voice );
      change_effect_volume( card, voice, 0 );
    }
}

static void voice_loop( gus_card_t *card, gus_voice_t *voice, unsigned int start, unsigned int end )
{
}

static void voice_ramp( gus_card_t *card, gus_voice_t *voice, unsigned char start, unsigned char end, unsigned char ramp, unsigned char control )
{
}

static void voice_pos( gus_card_t *card, gus_voice_t *voice, unsigned int position )
{
  unsigned long flags;

  voice -> data.patch.position = position;
  if ( voice -> flags & VFLG_WAIT_FOR_START ) return;
  if ( voice -> flags & VFLG_RUNNING )
    {
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      gus_write_addr( card, GF1_VA_CURRENT, voice -> wave -> begin.memory + voice -> data.patch.position, voice -> data.patch.control & 0x04 );
      STI( &flags );
    }
}

static void voice_pan( gus_card_t *card, gus_voice_t *voice, unsigned short pan )
{
  unsigned long flags;
  short pan_offset;

  if ( !(voice -> flags & VFLG_OSS_PAN) )
    if ( pan > 16383 ) pan = 16383;
  voice -> data.patch.voice_pan = pan;
  if ( voice -> flags & VFLG_WAIT_FOR_START ) return;
  if ( !voice -> wave ) return;
  if ( !(voice -> flags & VFLG_OSS_PAN) )
    {
      pan_offset = (short)(voice -> wave -> data.patch.balance >> 1) - 64;
      pan_offset = (pan_offset + ((short)(pan >> 7) - 64)) + 64;
    }
   else
    {
      if ( voice -> flags & VFLG_OSS_BALANCE )
        pan_offset = pan >> 7;
       else
        {
          pan_offset = (short)voice -> wave -> data.patch.balance - 128;
          pan_offset += (short)(pan >> 6) - 128;
          pan_offset /= 4;
          pan_offset += 64;
        }
    }
  if ( pan_offset < 0 ) pan_offset = 0;
  if ( pan_offset > 127 ) pan_offset = 127;
#if 0
  printk( "flg = 0x%x, pan = %i, balance = %i, pan_offset = %i\n", voice -> flags & (VFLG_OSS_PAN | VFLG_OSS_BALANCE), pan, voice -> wave -> data.patch.balance, pan_offset );
#endif
  voice -> data.patch.pan = pan_offset;
  if ( voice -> flags & VFLG_RUNNING )
    {
      voice -> data.patch.gf1_pan = voice -> data.patch.pan >> 3;
      if ( !card -> gf1.full_range_pan )
        {
          if ( voice -> data.patch.gf1_pan == 0 ) voice -> data.patch.gf1_pan++;
          if ( voice -> data.patch.gf1_pan == 15 ) voice -> data.patch.gf1_pan--;          
        }
      if ( voice -> flags & VFLG_RUNNING )
        {
          if ( card -> gf1.smooth_pan )
            {
              voice -> flags &= ~VFLG_PAN;
              do_pan_envelope( card, voice );
            }
           else
            {
              CLI( &flags );
              gf1_select_voice( card, voice -> number );
              gus_write8( card, GF1_VB_PAN, voice -> data.patch.gf1_pan );
              STI( &flags );
            }
        }
    }
   else
    set_enhanced_pan( card, voice );
}

static void voice_lfo( gus_card_t *card, gus_voice_t *voice, unsigned char *data )
{
  gus_lfo_command( card, voice -> number, data );
}

static unsigned char get_effects_mask( gus_card_t *card, int value )
{
  if ( value > 7 ) return 0;
  if ( card -> gf1.effects && card -> gf1.effects -> chip_type == GUS_EFFECT_CHIP_INTERWAVE )
    return card -> gf1.effects -> chip.interwave.voice_output[ value ];
  return 0;
}

static void voice_private1( gus_card_t *card, gus_voice_t *voice, unsigned char *data )
{
  unsigned long flags;
  unsigned char uc;
  
  switch ( *data ) {
    case GUS_PRIV1_IW_EFFECT:
      uc = get_effects_mask( card, gus_get_byte( data, 4 ) );
      uc |= get_effects_mask( card, gus_get_byte( data, 4 ) >> 4 );
      uc |= get_effects_mask( card, gus_get_byte( data, 5 ) );
      uc |= get_effects_mask( card, gus_get_byte( data, 5 ) >> 4 );
      voice -> data.patch.effect_accumulator = uc;
      voice -> data.patch.effect_volume = gus_translate_voice_volume( card, gus_get_word( data, 2 ) ) << 4;
      if ( !card -> gf1.enh_mode ) return;
      if ( voice -> flags & VFLG_WAIT_FOR_START ) return;
      if ( voice -> flags & VFLG_RUNNING )
        {
          CLI( &flags );
          gf1_select_voice( card, voice -> number );
          gus_write8( card, GF1_VB_ACCUMULATOR, voice -> data.patch.effect_accumulator );
          gus_write16( card, GF1_VW_EFFECT_VOLUME_FINAL, voice -> data.patch.effect_volume );
          STI( &flags );
        }
      break;
  }
}

/*
 *
 */

static void note_stop( gus_card_t *card, gus_voice_t *voice, int wait )
{
#if 0
  printk( " -%i- note_stop!! - flags = 0x%x\n", voice -> number, voice -> flags );
#endif
  if ( (voice -> flags & (VFLG_RUNNING | VFLG_USED)) == (VFLG_RUNNING | VFLG_USED) )
    voice_stop( card, voice, 0x00 );
}

static void note_wait( gus_card_t *card, gus_voice_t *voice )
{
  unsigned int old_flags;

#if 0
  printk( " -%i- note_wait = 0x%x\n", voice -> number, voice -> flags );
#endif
  old_flags = voice -> flags;
  if ( (old_flags & VFLG_USED) &&
       (old_flags & (VFLG_RUNNING|VFLG_SHUTDOWN)) )
    {
      if ( old_flags & VFLG_DYNAMIC )
        {
          gus_voice_and_note_free( card, voice );
          voice -> flags |= old_flags & VFLG_SHUTDOWN;
        }
       else
        voice -> flags = 0;
    }
}

static void note_off( gus_card_t *card, gus_voice_t *voice )
{
  gus_wave_t *wave;
  gus_note_t *note = voice -> note;

  if ( !(voice -> flags & VFLG_RUNNING) ) return;
  if ( voice -> flags & VFLG_RELEASING ) return;
  wave = voice -> wave;
  voice -> flags |= VFLG_RELEASING;
  voice -> flags &= ~VFLG_SUSTAINING;
  voice -> priority = ~0;		/* min priority */
  note -> flags &= ~NFLG_SUSTAIN;
  note -> flags |= NFLG_RELEASING;
  if ( voice -> data.patch.venv_state <= PATCH_SUSTAIN )
    {
      voice -> data.patch.venv_state = PATCH_VAR_RELEASE;
      do_volume_envelope( card, voice );
    }
}

static void note_volume( gus_card_t *card, gus_voice_t *voice )
{
  gus_channel_t *channel;
  unsigned short volume;

#if 0
  printk( "volume = %i, expression = %i, velocity = %i, pan = %i\n", channel -> volume, channel -> expression, voice -> note -> velocity, channel -> pan );
#endif
  if ( !voice -> note ) return;
  channel = voice -> note -> channel;
  if ( !channel ) return;
  volume = VOLUME( card -> mixer.soft_volume_level ) +
           VOLUME( channel -> volume ) +
           VOLUME( channel -> expression ) +
           VOLUME( voice -> note -> velocity );
  if ( volume > MAX_VOLUME ) volume = MAX_VOLUME;
  volume = MAX_VOLUME - volume;
  voice_volume( card, voice, volume | GUS_VOLUME_EXP );
  voice_pan( card, voice, channel -> pan << 7 );
}

static void note_pitchbend( gus_card_t *card, gus_voice_t *voice )
{
  if ( voice -> note && voice -> note -> channel )
    voice_freq( card, voice, GUS_FREQ_PITCHBEND | voice -> note -> channel -> gf1_pitchbend );
}

static void note_vibrato( gus_card_t *card, gus_voice_t *voice )
{
  gus_channel_t *channel;
  short depth, bdepth, current_depth, sweep = 0;
  unsigned short freq;
  
  if ( !voice -> note || !voice -> wave ) return;
  channel = voice -> note -> channel;
  if ( !channel ) return;
  bdepth = voice -> wave -> data.patch.vibrato_depth >> 1;
  if ( channel -> vib_depth )
    {
      if ( bdepth < 0 )
        depth = gus_compute_vibrato( -(((channel -> vib_depth >> 1) * (127 + bdepth) >> 7) - bdepth), voice -> fc_register );
       else
        depth = gus_compute_vibrato( bdepth + ((channel -> vib_depth >> 1) * (127 - bdepth) >> 7), voice -> fc_register );
      if ( bdepth )
        sweep = voice -> wave -> data.patch.vibrato_sweep * (127 - channel -> vib_depth) >> 7;
    }
   else
    {
      depth = gus_compute_vibrato( bdepth, voice -> fc_register );
      if ( depth )
        sweep = voice -> wave -> data.patch.vibrato_sweep;
    }
  current_depth = !sweep ? depth << 5 : 0;
  if ( voice -> flags & VFLG_VIBRATO_RUNNING )
    {
      if ( depth )
        gus_lfo_change_depth( card, voice -> number, GUS_LFO_VIBRATO, depth );
       else
        {
          gus_lfo_shutdown( card, voice -> number, GUS_LFO_VIBRATO );
          voice -> flags &= ~VFLG_VIBRATO_RUNNING;
        }
    }
   else
  if ( depth )
    {
      freq = (short)(( ( (int)voice -> wave -> data.patch.vibrato_rate * 222 ) + 475 ) / 100);
      if ( freq == 4 ) freq = 523;
#if 0
      printk( "lfo vibrato setup: freq = %i, current_depth = %i, depth = %i, sweep = %i\n", freq, current_depth, depth, sweep );
#endif
      gus_lfo_setup( card,
      			voice -> number, GUS_LFO_VIBRATO, freq,
      			current_depth, depth, sweep,
      			GUS_STRU_IW_LFO_SHAPE_TRIANGLE );
      voice -> flags |= VFLG_VIBRATO_RUNNING;
    }
}

static void note_tremolo( gus_card_t *card, gus_voice_t *voice )
{
  gus_channel_t *channel;
  short depth, bdepth, current_depth, sweep = 0;
  unsigned short freq;

  if ( !voice -> note || !voice -> wave ) return;  
  channel = voice -> note -> channel;
  if ( !channel ) return;
  bdepth = voice -> wave -> data.patch.tremolo_depth >> 1;
  if ( channel -> trem_depth )
    {
      if ( bdepth < 0 )
        depth = -((channel -> trem_depth * (MAX_TDEPTH + bdepth) / MAX_TDEPTH) - bdepth);
       else
        depth = bdepth + (channel -> trem_depth * (MAX_TDEPTH - bdepth) / MAX_TDEPTH);
      if ( bdepth )
        sweep = voice -> wave -> data.patch.tremolo_sweep * (127 - channel -> trem_depth) >> 7;
    }
   else
    {
      depth = bdepth;
      if ( depth )
        sweep = voice -> wave -> data.patch.tremolo_sweep;
    }
  current_depth = !sweep ? depth << 5 : 0;
  if ( voice -> flags & VFLG_TREMOLO_RUNNING )
    {
      if ( depth )
        gus_lfo_change_depth( card, voice -> number, GUS_LFO_TREMOLO, depth );
       else
        {
          gus_lfo_shutdown( card, voice -> number, GUS_LFO_TREMOLO );
          voice -> flags &= ~VFLG_TREMOLO_RUNNING;
        }
    }
   else
  if ( depth )
    {
      freq = (short)(( ( (int)voice -> wave -> data.patch.tremolo_rate * 222 ) + 475 ) / 100);
      if ( freq == 4 ) freq = 523;
#if 0
      printk( "lfo tremolo setup: freq = %i, current_depth = %i, depth = %i, sweep = %i\n", freq, current_depth, depth, sweep );
#endif
      gus_lfo_setup( card,
      			voice -> number, GUS_LFO_TREMOLO, freq,
      			current_depth, depth, sweep,
      			GUS_STRU_IW_LFO_SHAPE_TRIANGLE );
      voice -> flags |= VFLG_TREMOLO_RUNNING;
    }
}

/*
 *
 */
 
static void chn_trigger_down( gus_card_t *card, gus_channel_t *channel, gus_instrument_t *instrument, unsigned char note, unsigned char velocity, unsigned char priority )
{
  gus_note_t *pnote;
  gus_voice_t *voice;
  
#if 0
  {
    static int count = 0;
    if ( count++ > 0 ) return;
  }
#endif
  if ( ( pnote = gus_note_alloc( card, channel ) ) == NULL ) return;
  pnote -> commands = &note_commands;
  pnote -> voices = NULL;
  pnote -> number = note;
  pnote -> velocity = velocity;
  pnote -> channel = channel;
  pnote -> instrument = instrument;
  pnote -> priority = priority;
  if ( ( voice = gus_voice_alloc( card, pnote, priority, stop_note_voice ) ) == NULL /* || voice -> number != 0 */ )
    {
      gus_note_free( card, pnote );
      return;
    }
  voice -> commands = &voice_commands;
  voice -> note = pnote;
#if 0
  printk( "note on - instrument = %i, note = %i, velocity = %i\n", instrument -> number.instrument, note, velocity );
#endif
  /* ok.. now start voice */
  voice_start( card, voice );
  voice -> instrument = instrument;
  voice -> layer = instrument -> info.layer;
  voice -> data.patch.pitchbend = channel -> gf1_pitchbend;
  voice_freq( card, voice, GUS_FREQ_NOTE | note );
  note_volume( card, voice );
  voice_go( card, voice );
  if ( voice -> wave )
    {
      if ( voice -> wave -> data.patch.vibrato_depth || channel -> vib_depth )
        note_vibrato( card, voice );
      if ( voice -> wave -> data.patch.tremolo_depth || channel -> trem_depth )
        note_tremolo( card, voice );
    }
}

static void chn_trigger_up( gus_card_t *card, gus_note_t *note )
{
}

static void chn_control( gus_card_t *card, gus_channel_t *channel, unsigned short p1, unsigned short p2 )
{
  switch ( p1 ) {
    case GUS_MCTL_E1_REVERB_DEPTH:
      channel -> effect1_level = VOLUME( p2 );
      gus_engine_midi_recompute_volume( card, channel );
      break;
    case GUS_MCTL_E3_CHORUS_DEPTH:
      channel -> effect3_level = VOLUME( p2 );
      gus_engine_midi_recompute_volume( card, channel );
      break;
  }
}

/*
 *
 */
 
void gus_gf1_patch_init( gus_card_t *card )
{
  gus_engine_instrument_register( GUS_INSTR_PATCH, &voice_commands, &note_commands, &channel_commands );
}
