/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for (General) MIDI support
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <errno.h>
#include "libgus.h"
#include "libgus_local.h"
#define NO_DPRINTF

#if 0
#define MIDI_DEBUG
#endif

/*
 *  defines
 */

#define MIDI_FILE		"/dev/gusmidi"
#define REALTIME_BUFFER_SIZE	512
#if 0
#define GUS_SPECIAL_SYSEX	"GUSEX"
#define GUS_SPECIAL_SYSEX_SIZE	5
#endif

/*
 *  structures
 */

typedef gus_midi_device_t DEVICE;

/*
 *  variables
 */

static DEVICE *midi_out_devices = NULL;
static DEVICE *midi_in_devices = NULL;
static DEVICE *acard;

static int midi_fd;
static unsigned char old_device;
static unsigned char old_command;

static size_t midi_out_size_normal;
static size_t midi_out_size_realtime;
static size_t midi_out_size;
static size_t midi_out_used_normal;
static size_t midi_out_used_realtime;
static size_t midi_out_used;
static size_t midi_in_size;
static size_t midi_in_used;
static unsigned char *midi_out_buffer_normal;
static unsigned char *midi_out_buffer_realtime;
static unsigned char *midi_out_buffer;
static unsigned char *midi_in_buffer;

static int timer_running;
static int realtime_flag;
static int detach_flag;

static int timer_base;
static int timer_tempo;

static char *_s_gus_midi_error = NULL;

/*
 *  local functions
 */

static void reset_variables( void )
{
  timer_running = 0;
  old_device = 0xff;
  old_command = 0;
  midi_out_used =
  midi_out_used_normal =
  midi_out_used_realtime =
  midi_in_used = 0;
}

static int search_output_channel( int chn )
{
  if ( chn < 0 || chn >= GUS_MIDI_CHANNELS )
    {
      acard = NULL;
      return GUS_MIDID_COMMON;
    }
  chn = 1 << chn;
  acard = midi_out_devices;
  while ( acard )
    {
      if ( acard -> channels & chn ) break;
      acard = acard -> next;
    }
  if ( !acard ) return GUS_MIDID_COMMON;
  return acard -> device;
}

static void free_cards1( DEVICE *card )
{
  DEVICE *ocard;
  
  while ( card )
    {
      ocard = card;
      card = card -> next;
      free( ocard );
    }
}

static void free_cards( void )
{
  free_cards1( midi_out_devices );
  free_cards1( midi_in_devices );
  free( midi_out_buffer_normal );
  free( midi_out_buffer_realtime );
  free( midi_in_buffer );
  midi_out_buffer_normal =
  midi_out_buffer_realtime =
  midi_out_buffer =
  midi_in_buffer = NULL;
  midi_out_devices = NULL;
  midi_in_devices = NULL;
}

static char *getparam( char *line, int param )
{
  char *s, *d;
  static char str[ 256 + 1 ];
 
  s = line;
  while ( *s && param > 0 )
    {
      if ( *s == ':' ) param--;
      s++;
    }
  while ( *s && *s <= ' ' ) s++;
  d = str;
  while ( *s && *s != ':' ) *d++ = *s++;
  *d = 0; --d;
  while ( str[ 0 ] && *d <= ' ' ) *d-- = 0;
  return str;
}

/*
 *
 */

static void put_byte( unsigned char b )
{
  if ( !midi_out_devices ) return;
  if ( midi_out_used < midi_out_size )
    midi_out_buffer[ midi_out_used++ ] = b;
   else
    fprintf( stderr, "midi out buffer overflow, used = %i, size = %i\n", midi_out_used, midi_out_size );
}
 
static void device_select( unsigned char device, unsigned short size )
{
  while ( 3 + size + midi_out_used > midi_out_size )
    gus_midi_write();
  put_byte( (unsigned char)(size >> 8) );
  put_byte( (unsigned char)size );
  put_byte( device );
}

static void put_command( unsigned char device, unsigned char cmd, unsigned char chn, unsigned short size )
{
  device_select( device, size + 1 );
  put_byte( cmd | chn );
}

/*
 *  functions
 */

char *gus_midi_error( void )
{
  return _s_gus_midi_error;
}

char *gus_midi_error_set( char *error )
{
  if ( _s_gus_midi_error ) {
    free( _s_gus_midi_error ); 
    _s_gus_midi_error = NULL;
  }
  if ( error ) _s_gus_midi_error = strdup( error );
  return _s_gus_midi_error;
}

int gus_midi_fill_device_structure( gus_midi_device_t *device, int mode, int device_number, unsigned short channels )
{
  if ( !device ) return -1;
  if ( mode != GUS_MIDI_OPEN_MODE_READ &&
       mode != GUS_MIDI_OPEN_MODE_WRITE ) return -1;
  if ( device_number < 0 && device_number >= 0xff ) return -1;
  memset( device, 0, sizeof( *device ) );
  device -> mode = mode;
  device -> device = device_number;
  device -> channels = channels;
  device -> midi_device = 0x10;			/* default for Roland */
  device -> emulation = GUS_MIDI_EMUL_AUTO;	/* auto */
  return 0;
}

int gus_midi_open_intelligent( int mode, char *cfg_file, size_t buffer_size, int flags )
{
  FILE *in;
  int line_number, i;
  char line[ 512 + 1 ];
  gus_midi_device_t card;
  gus_midi_device_t *first_card = NULL, *pcard = NULL, *pcard1 = NULL;
  char *s;

  gus_midi_error_set( NULL );
  mode &= O_ACCMODE;
  if ( buffer_size < 512 || buffer_size > 128 * 1024 )
    {
      sprintf( line, "Buffer size out of range." );
      gus_midi_error_set( line );
      return -1;
    }
  if ( !cfg_file ) cfg_file = GUS_FILE_MIDI_CONF;
  if ( ( in = fopen( cfg_file, "r" ) ) == NULL )
    {
      gus_dprintf( "gus_midi_open_intelligent: can't open cfg file '%s'", cfg_file );
      sprintf( line, "Can't open configuration file '%s'.", cfg_file );
      gus_midi_error_set( line );
      return -1;
    }
  line_number = 0;
  while ( !feof( in ) )
    {
      line_number++;
      gus_midi_fill_device_structure( &card, GUS_MIDI_OPEN_MODE_WRITE, 0x00, 0x00 );
      if ( fgets( line, sizeof( line ) - 1, in ) == NULL ) break;
      if ( line[ 0 ] == '#' ) continue;
      s = getparam( line, 0 );
      if ( !*s ) continue;
      if ( !strcmp( s, "out" ) ) card.mode = GUS_MIDI_OPEN_MODE_WRITE; else
      if ( !strcmp( s, "in" ) ) card.mode = GUS_MIDI_OPEN_MODE_READ; else
        {
          gus_dprintf( "gus_midi_open_intelligent: wrong direction (line %i)", line_number );
          sprintf( line, "Wrong direction (line %i).", line_number );
          gus_midi_error_set( line );
          free_cards1( pcard );
          return -1;
        }
      if ( mode != O_RDWR )
        {
          if ( mode == O_RDONLY && card.mode == GUS_MIDI_OPEN_MODE_WRITE ) continue;
          if ( mode == O_WRONLY && card.mode == GUS_MIDI_OPEN_MODE_READ ) continue;
        }
#if 0
      printf( "mode = %i, card.mode = %i\n", mode, card.mode );
#endif
      s = getparam( line, 1 );
      if ( !strcmp( s, "gf1" ) || !strcmp( s, "synth" ) ) card.device = GUS_MIDID_SYNTH; else
      if ( !strcmp( s, "ext" ) || !strcmp( s, "uart" ) ) card.device = GUS_MIDID_UART; else
        {
          gus_dprintf( "gus_midi_open_intelligent: wrong device (line %i)", line_number );
          sprintf( line, "Wrong device (line %i).", line_number );
          gus_midi_error_set( line );
          free_cards1( pcard );
          return -1;
        }
      i = atoi( getparam( line, 2 ) ) - 1;
      if ( i < 0 || i >= GUS_CARDS )
        {
          gus_dprintf( "gus_midi_open_intelligent: wrong card (line %i)", line_number );
          sprintf( line, "Wrong card (line %i).", line_number );
          gus_midi_error_set( line );
          free_cards1( pcard );
          return -1;
        }
      card.device |= i << 4;
      for ( s = getparam( line, 3 ), i = 0; *s; s++, i++ )
        if ( *s == '0' ) card.channels &= ~( 1 << i ); else
        if ( *s == '1' ) card.channels |= ( 1 << i ); else
          {
            gus_dprintf( "gus_midi_open_intelligent: wrong channel mark (line %i)", line_number );
            sprintf( line, "Wrong channel mark (line %i).", line_number );
            gus_midi_error_set( line );
            free_cards1( pcard );
            return -1;
          }
      s = getparam( line, 4 );
      if ( *s )
        card.midi_device = atoi( s );
      s = getparam( line, 5 );
      if ( *s )
        {
          if ( !strcmp( s, "gm" ) ) card.emulation = GUS_MIDI_EMUL_GM; else
          if ( !strcmp( s, "gs" ) ) card.emulation = GUS_MIDI_EMUL_GS; else
          if ( !strcmp( s, "mt-32" ) || !strcmp( s, "mt32" ) ) card.emulation = GUS_MIDI_EMUL_MT32; else
          if ( !strcmp( s, "auto" ) ) card.emulation = GUS_MIDI_EMUL_AUTO; else
            {
              gus_dprintf( "gus_midi_open_intelligent: wrong emulation (line %i)", line_number );
              sprintf( line, "Wrong emulation (line %i).", line_number );
              gus_midi_error_set( line );
              free_cards1( pcard );
              return -1;
            }
        }

      pcard1 = (gus_midi_device_t *)malloc( sizeof( gus_midi_device_t ) );
      if ( !pcard1 )
        {
          gus_dprintf( "gus_midi_open_intelligent: alloc error" );
          sprintf( line, "Allocation error." );
          gus_midi_error_set( line );
          free_cards1( pcard );
          return -1;
        }
      memcpy( pcard1, &card, sizeof( card ) );
      if ( !first_card )
        first_card = pcard1;
       else
        pcard -> next = pcard1;
      pcard = pcard1;
    }
  fclose( in );
  return gus_midi_open( mode, first_card, buffer_size, flags );
}

/*
 *
 */
 
int gus_midi_cards( void )
{
  return gus_cards();
}

int gus_midi_devices( struct GUS_STRU_MIDI_DEVICES *devices )
{
  int no_open = !midi_out_devices && !midi_in_devices;

  if ( no_open )
    {
      midi_fd = open( MIDI_FILE, O_RDONLY );
      if ( midi_fd < 0 ) return -1;
    }
  if ( ioctl( midi_fd, GUS_MIDI_DEVICES, devices ) < 0 )
    {
      gus_dprintf( "gus_midi_devices: ioctl error (%s)", strerror( errno ) );
      return -1;
    }
  if ( no_open )
    close( midi_fd );
  return 0;
}

int gus_midi_device_info( struct GUS_STRU_MIDI_DEVICE_INFO *info )
{
  int no_open = !midi_out_devices && !midi_in_devices;

  if ( no_open )
    {
      midi_fd = open( MIDI_FILE, O_RDONLY );
      if ( midi_fd < 0 ) return -1;
    }
  if ( ioctl( midi_fd, GUS_MIDI_DEVICE_INFO, info ) < 0 )
    {
      gus_dprintf( "gus_midi_device_info: ioctl error (%s)", strerror( errno ) );
      return -1;
    }
  if ( no_open )
    close( midi_fd );
  return 0;
}

/*
 *  MIDI device detection routine
 */

static void autodetection_sysex( void *privatedata, int device, unsigned char *data, int len )
{
  gus_midi_device_t *devo;

#if 0
  {
    int i;

    printf( "sysex: device = 0x%x, data = 0x%lx, len = %i\n", device, (long)data, len );
    for ( i = 0; i < len; i++ )
      printf( "%02x:", data[ i ] );
    printf( "\n" );
  }
#endif
  devo = gus_midi_output_device( device );
  if ( !devo ) return;		/* huh.. something is wrong */
  if ( data[ 0 ] == 0x7e && data[ 1 ] == devo -> midi_device &&
       data[ 2 ] == 0x06 && data[ 3 ] == 0x02 &&	/* inquiry reply */
       len == 13 )
    {
      devo -> midi_id_number = data[ 4 ];
      devo -> midi_family_code = ( data[ 5 ] << 8 ) | data[ 6 ];
      devo -> midi_family_number_code = ( data[ 7 ] << 8 ) | data[ 8 ];
      devo -> midi_software_revision = ( data[ 9 ] << 24 ) |
      				       ( data[ 10 ] << 16 ) |
      				       ( data[ 11 ] << 8 ) |
      				       data[ 12 ];
    }
}
 
int gus_midi_autodetection( int device )
{
  static gus_midi_callbacks_t callbacks = {
    17,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    autodetection_sysex,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL
  };
  static unsigned char gm_inquiry_request[ 4 ] = { 0x7e, 0x10, 0x06, 0x01 };
  gus_midi_device_t *devo, *devi;  
  
#ifdef MIDI_DEBUG
  printf( "gus_midi_autodetection\n" );
#endif  
  devo = gus_midi_output_device( device );
  if ( !devo ) return -1;
  devi = gus_midi_input_device( device );
  if ( !devi ) return -1;
  if ( !devo -> midi_id_number )
    {
      /* try get device info now */
      gus_midi_abort();
      gm_inquiry_request[ 1 ] = devo -> midi_device;
      gus_midi_sysex( device, gm_inquiry_request, 4 );
      gus_midi_flush();
      while ( !devo -> midi_id_number )
        {
          fd_set fds_read;
          struct timeval timeout;
          
 	  timeout.tv_sec = 1;
 	  timeout.tv_usec = 0;
          FD_ZERO( &fds_read );
          FD_SET( midi_fd, &fds_read );
          if ( !select( midi_fd + 1, &fds_read, NULL, NULL, &timeout ) )
            break;
          if ( FD_ISSET( midi_fd, &fds_read ) )
            gus_midi_input( &callbacks );
        }
      if ( !devo -> midi_id_number )
        devo -> midi_id_number = 0xffff;
    }
  devo -> emulation = GUS_MIDI_EMUL_GM;
  if ( devo -> midi_id_number == 0xffff ) return 0;	/* autodetection failed */

  /* ok.. Roland check */

#if 0
  printf( "midi_id_number = 0x%x, family_code = 0x%x\n", devo -> midi_id_number, devo -> midi_family_code[ 0 ] );
#endif
  if ( devo -> midi_id_number == 0x41 )			/* Roland */
    {
      if ( devo -> midi_family_code == 0x4200 )		/* GS */
        devo -> emulation = GUS_MIDI_EMUL_GS;
    }
  
  return 0;
}

gus_midi_device_t *gus_midi_output_device( int device )
{
  gus_midi_device_t *result;
  
  for ( result = midi_out_devices; result; result = result -> next )
    if ( result -> device == device ) return result;
  return NULL;
}

gus_midi_device_t *gus_midi_input_device( int device )
{
  gus_midi_device_t *result;
  
  for ( result = midi_in_devices; result; result = result -> next )
    if ( result -> device == device ) return result;
  return NULL;
}

gus_midi_device_t *gus_midi_output_devices( void )
{
  return midi_out_devices;
}

gus_midi_device_t *gus_midi_input_devices( void )
{
  return midi_in_devices;
}

int gus_midi_open( int mode, gus_midi_device_t *card, size_t buffer_size, int flags )
{
  int res, open_index;
  DEVICE *ocard, *c, *oldout, *oldin;
  struct GUS_STRU_MIDI_OPEN mopen;

#if 0
  printf( "mode = 0x%x, card = 0x%lx, buffer_size = 0x%x\n", mode, (long)card, buffer_size );  
#endif
  if ( !card ) return -1;
  if ( buffer_size < 512 || buffer_size > 128 * 1024 ) return -1;
  mode &= O_ACCMODE;
  open_index = 0;
  realtime_flag = 0;
  detach_flag = 1;
  midi_out_devices = midi_in_devices = NULL;
  midi_out_buffer = 
  midi_out_buffer_normal =
  midi_out_buffer_realtime =
  midi_in_buffer = NULL;
  oldout = oldin = NULL;
  ocard = card;
  while ( card )
    {
      if ( mode != O_RDWR )
        {
          if ( mode == O_RDONLY && card -> mode == GUS_MIDI_OPEN_MODE_WRITE )
            {
              card = card -> next;
              continue;
            }
          if ( mode == O_WRONLY && card -> mode == GUS_MIDI_OPEN_MODE_READ )
            {
              card = card -> next;
              continue;
            }
        }
      c = (DEVICE *)calloc( 1, sizeof( DEVICE ) );
      if ( !c )
        {
          free_cards1( ocard );
          free_cards();
          return -1;
        }
      c -> device = mopen.devices[ open_index ].device = card -> device;
      c -> mode = mopen.devices[ open_index ].mode = card -> mode;
      c -> channels = card -> channels;
      c -> midi_device = card -> midi_device;
      c -> emulation = card -> emulation;
      if ( ( c -> device & 0x0f ) == GUS_MIDID_SYNTH )
        {
          c -> midi_id_number = 0x100;
          c -> midi_family_code = 0x4200;
          c -> midi_family_number_code = 0x0001;
          c -> midi_software_revision = LIBGUS_VERSION; 	/* pretty :-) */
        }
      card = card -> next;
      open_index++;
      if ( c -> mode == GUS_MIDI_OPEN_MODE_WRITE )
        {
          if ( !midi_out_devices ) midi_out_devices = c;
          if ( oldout ) oldout -> next = c;
          oldout = c;
        }
       else
        {
          if ( !midi_in_devices ) midi_in_devices = c;
          if ( oldin ) oldin -> next = c;
          oldin = c;
        }
    }
  if ( !midi_out_devices && !midi_in_devices )
    {
      gus_midi_error_set( "no devices available" );
      return -1;
    }
  mopen.devices_used = open_index;
  res = 0;
  open_index = O_RDWR;
  if ( !midi_out_devices )
    open_index = O_RDONLY;
  if ( !midi_in_devices && !( flags & GUS_MIDI_OF_ECHO_ENABLE ) )
    open_index = O_WRONLY;
  midi_fd = open( MIDI_FILE, open_index | ( flags & GUS_MIDI_OF_NONBLOCK ? O_NONBLOCK : 0 ) );
  if ( midi_fd < 0 ) res = -1;
  if ( !res ) res = ioctl( midi_fd, GUS_MIDI_DEVICE_OPEN, &mopen );
  if ( res < 0 )
    {
      gus_dprintf( "gus_midi_open: open error (%s)", strerror( errno ) );
      gus_midi_error_set( "midi open error" );
      res = -1;
    }
  free_cards1( ocard );

  midi_out_size = 
  midi_out_size_normal = buffer_size;
  midi_in_size = midi_in_devices ? buffer_size : REALTIME_BUFFER_SIZE;
  midi_out_size_realtime = REALTIME_BUFFER_SIZE;
  reset_variables();

  if ( !res )
    {
      if ( midi_out_devices )
        {
          midi_out_buffer = (unsigned char *)malloc( midi_out_size );
          midi_out_buffer_normal = midi_out_buffer;
          midi_out_buffer_realtime = (unsigned char *)malloc( midi_out_size_realtime );
        }
      midi_in_buffer = (unsigned char *)malloc( midi_in_size );
      if ( ( midi_out_devices && ( !midi_out_buffer || !midi_out_buffer_realtime ) ) || !midi_in_buffer )
        {
          close( midi_fd );
          res = -1;
        }
    }

  if ( res < 0 )
    free_cards();
   else
    {
      DEVICE *device;
      struct GUS_STRU_MIDI_DEVICE_INFO info;
      
      /* we need know emulation */
      for ( device = midi_out_devices; device; device = device -> next )
        {
          info.device = device -> device;
          gus_midi_device_info( &info );
          device -> cap = info.cap;
          if ( device -> emulation == GUS_MIDI_EMUL_AUTO )
            {
              if ( device -> cap & GUS_MIDI_CAP_SYNTH )
                device -> emulation = gus_midi_emulation_get( device -> device );
               else
                if ( gus_midi_autodetection( device -> device ) < 0 )
                  device -> emulation = GUS_MIDI_EMUL_GM;	/* defaults to GM */
            }
           else
            {
              if ( device -> cap & GUS_MIDI_CAP_SYNTH )
                gus_midi_emulation_set( device -> device, device -> emulation );
            }
          if ( device -> cap & GUS_MIDI_CAP_SYNTH )
            {
              gus_info_t info;
              
              if ( gus_midi_synth_info( device -> device, &info ) >= 0 )
                if ( info.flags & GUS_STRU_INFO_F_ENHANCED ) device -> midi_family_number_code = 0x0002;
            }
        }
    }

  gus_midi_timer_base( 320 );
  gus_midi_timer_tempo( 60 );

  return res;
}

int gus_midi_close( void )
{
  gus_midi_detach();
  gus_midi_error_set( NULL );
  if ( ioctl( midi_fd, GUS_MIDI_DEVICE_CLOSE ) < 0 )
    {
      gus_dprintf( "gus_midi_open: close error (%s)", strerror( errno ) );
      return -1;
    }
  free_cards();
  close( midi_fd );
  return 0;
}

int gus_midi_get_file_descriptor( void )
{
  return midi_fd;
}

int gus_midi_realtime( int enable )
{
  if ( enable )
    {
      if ( realtime_flag ) return 0;
      midi_out_used_normal = midi_out_used;
      midi_out_buffer = midi_out_buffer_realtime;
      midi_out_size = midi_out_size_realtime;
      midi_out_used = midi_out_used_realtime;
      realtime_flag = 1;
    }
   else
    {
      if ( !realtime_flag ) return 0;
      midi_out_used_realtime = midi_out_used;
      midi_out_buffer = midi_out_buffer_normal;
      midi_out_size = midi_out_size_normal;
      midi_out_used = midi_out_used_normal;
      realtime_flag = 0;
    }
  return 0;
}

int gus_midi_gs_sysex_checksum( unsigned char *data, int size )
{
  unsigned char result = 0;
  
  data += 4;		/* skip first 4 bytes */
  size -= 5;
  if ( size <= 0 ) return -1;
  while ( size-- > 0 )
    result += *data++;
  *data = ( 0x80 - ( result & 0x7f ) ) & 0x7f;
  return 0;
}

static void kill_sounds( void )
{
  int i, base, tempo;
  DEVICE *device;

  base = timer_base;
  tempo = timer_tempo;
  gus_midi_timer_base( 320 );
  gus_midi_timer_tempo( 100 );
  gus_midi_timer_start();
  gus_midi_wait( 20 );		/* 50ms */
  for ( device = midi_out_devices; device; device = device -> next )
    {  
      for ( i = 0; i < 16; i++ )
        {
          gus_midi_control( device -> device, i, GUS_MCTL_ALL_SOUNDS_OFF, 0 );
          gus_midi_control( device -> device, i, GUS_MCTL_RESET_CONTROLLERS, 0 );
        }
    }
  gus_midi_flush();
  gus_midi_timer_stop();
  gus_midi_timer_base( base );
  gus_midi_timer_tempo( tempo );
}

int gus_midi_reset( void )
{
  DEVICE *device;
  static char gm_reset[ 4 ] = { 0x7e, 0x7f, 0x09, 0x01 };
  static char gs_reset[ 9 ] = { 0x41, 0x10, 0x42, 0x12, 0x40, 0x00, 0x7f, 0x00, 0x41 };

#ifdef MIDI_DEBUG
  printf( "gus_midi_reset\n" );
#endif
  gus_midi_timer_stop();
  gus_midi_abort();
  for ( device = midi_out_devices; device; device = device -> next )
    {
      switch ( device -> emulation ) {
        case GUS_MIDI_EMUL_GS:
          gs_reset[ 1 ] = device -> midi_device;
          gus_midi_sysex( device -> device, gs_reset, sizeof( gs_reset ) );
          break;
        default:
          gus_midi_sysex( device -> device, gm_reset, sizeof( gm_reset ) );
          break;
      }
    }
  gus_midi_flush();
  kill_sounds();
  reset_variables();
#ifdef MIDI_DEBUG
  printf( "gus_midi_reset - end\n" );
#endif
  detach_flag = 0;
  return 0;
}

int gus_midi_detach( void )
{
  DEVICE *device;
  static char gm_detach[ 4 ] = { 0x7e, 0x7f, 0x09, 0x02 };
  static char gs_detach[ 9 ] = { 0x41, 0x10, 0x42, 0x12, 0x40, 0x00, 0x7f, 0x7f, 0x42 };
  
#ifdef MIDI_DEBUG
  printf( "gus_midi_detach\n" );
#endif
  if ( detach_flag ) return 0;
  gus_midi_timer_stop();
  gus_midi_abort();
  for ( device = midi_out_devices; device; device = device -> next )
    {
      switch ( device -> emulation ) {
        case GUS_MIDI_EMUL_GS:
          gs_detach[ 1 ] = device -> midi_device;
          gus_midi_sysex( device -> device, gs_detach, sizeof( gs_detach ) );
          break;
        default:
          gus_midi_sysex( device -> device, gm_detach, sizeof( gm_detach ) );
      }
    }
  gus_midi_flush();
  kill_sounds();
  reset_variables();
  detach_flag = 1;
  return 0;
}

int gus_midi_emulation_set( int device, int emulation )
{
  DEVICE *d;

  d = gus_midi_output_device( device );
  if ( !d ) return -1;
  if ( d -> cap & GUS_MIDI_CAP_SYNTH )
    {
      struct GUS_STRU_MIDI_EMULATION emul;
  
      emul.device = device;
      emul.emulation = emulation;
      if ( ioctl( midi_fd, GUS_MIDI_SET_EMULATION, &emul ) < 0 )
        {
          gus_dprintf( "gus_midi_emulation_set: %s\n", strerror( errno ) );
          return -1;
        }
      d -> emulation = emulation;
    }
   else
    {
      if ( emulation == GUS_MIDI_EMUL_AUTO )
        {
          if ( gus_midi_autodetection( device ) < 0 ) return -1;
        }
       else
        {
          if ( emulation < 0 || emulation > GUS_MIDI_EMUL_MT32 ) return -1;
          d -> emulation = emulation;
        }
      return 0;
    }
  return 0;
}

int gus_midi_emulation_get( int device )
{
  DEVICE *d;
  
  d = gus_midi_output_device( device );
  if ( !d )
    {
      gus_dprintf( "gus_midi_emulation_get: device 0x%x not found?\n", device );
      return -1;
    }
  if ( d -> cap & GUS_MIDI_CAP_SYNTH )
    {
      struct GUS_STRU_MIDI_EMULATION emul;
      
      emul.device = device;
      emul.emulation = 0xffff;
      if ( ioctl( midi_fd, GUS_MIDI_GET_EMULATION, &emul ) < 0 )
        {
          gus_dprintf( "gus_midi_emulation_get: %s\n", strerror( errno ) );
          return -1;
        }
      d -> emulation = emul.emulation;
      return emul.emulation;
    }
   else
    return d -> emulation;
}

int gus_midi_note_on( int device, int chn, int note, int vel )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_note_on: dev = %i, chn=%i, note=%i, vel=%i\n", device, chn, note, vel );
#endif
  if ( !midi_out_devices ) return -1;
  if ( device == GUS_MIDID_COMMON )	/* special channel handling */
    if ( ( device = search_output_channel( chn ) ) == GUS_MIDID_COMMON ) return -1;
  put_command( device, GUS_MCMD_NOTE_ON, chn, 2 );
  put_byte( note > 127 ? 127 : note );
  put_byte( vel > 127 ? 127 : vel );
  return 0;
}

int gus_midi_note_off( int device, int chn, int note, int vel )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_note_off: dev = %i, chn = %i, note = %i, vel = %i\n", device, chn, note, vel );
#endif
  if ( !midi_out_devices ) return -1;
  if ( device == GUS_MIDID_COMMON )	/* special channel handling */
    if ( ( device = search_output_channel( chn ) ) == GUS_MIDID_COMMON ) return -1;
  put_command( device, GUS_MCMD_NOTE_OFF, chn, 2 );
  put_byte( note > 127 ? 127 : note );
  put_byte( vel > 127 ? 127 : vel );
  return 0;
}

int gus_midi_note_pressure( int device, int chn, int note, int vel )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_key_pressure: chn=%i, note=%i, vel=%i\n", chn, note, vel );
#endif
  if ( !midi_out_devices ) return -1;
  if ( device == GUS_MIDID_COMMON )	/* special channel handling */
    if ( ( device = search_output_channel( chn ) ) == GUS_MIDID_COMMON ) return -1;
  put_command( device, GUS_MCMD_NOTE_PRESSURE, chn, 2 );
  put_byte( note > 127 ? 127 : note );
  put_byte( vel > 127 ? 127 : vel );
  return 0;
}

int gus_midi_channel_pressure( int device, int chn, int vel )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_chn_pressure: chn=%i, vel=%i\n", chn, vel );
#endif
  if ( !midi_out_devices ) return -1;
  if ( device == GUS_MIDID_COMMON )	/* special channel handling */
    if ( ( device = search_output_channel( chn ) ) == GUS_MIDID_COMMON ) return -1;
  put_command( device, GUS_MCMD_CHANNEL_PRESSURE, chn, 1 );
  put_byte( vel > 127 ? 127 : vel );
  return 0;
}

int gus_midi_bender( int device, int chn, int value )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_bender: chn=%i, value=%i\n", chn, value );
#endif
  if ( !midi_out_devices ) return -1;
  if ( device == GUS_MIDID_COMMON )	/* special channel handling */
    if ( ( device = search_output_channel( chn ) ) == GUS_MIDID_COMMON ) return -1;
  if ( value < 0 ) value = 0;
  if ( value > 0x3fff ) value = 0x3fff;
  put_command( device, GUS_MCMD_BENDER, chn, 2 );
  put_byte( value & 0x7f );
  put_byte( ( value >> 7 ) & 0x7f );
  return 0;
}

int gus_midi_program_change( int device, int chn, int program )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_program_change: chn = %i, program = %i\n", chn, program );
#endif
  if ( !midi_out_devices ) return -1;
  if ( device == GUS_MIDID_COMMON )	/* special channel handling */
    if ( ( device = search_output_channel( chn ) ) == GUS_MIDID_COMMON ) return -1;
  put_command( device, GUS_MCMD_PGM_CHANGE, chn, 1 );
  put_byte( program );
  return 0;
}
 
int gus_midi_control( int device, int chn, int controller, int value )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_control: chn=%i, controller=%i, value=%i\n", chn, controller, value );
#endif
  if ( !midi_out_devices ) return -1;
  if ( device == GUS_MIDID_COMMON )	/* special channel handling */
    if ( ( device = search_output_channel( chn ) ) == GUS_MIDID_COMMON ) return -1;
  put_command( device, GUS_MCMD_CONTROL, chn, 2 );
  put_byte( controller > 127 ? 127 : controller );
  put_byte( value > 127 ? 127 : value );
  return 0;
}

#if 0

int gus_midi_extended_control( int device, int chn, int p1, int p2 )
{
  unsigned char sysex[ GUS_SPECIAL_SYSEX_SIZE + 6 ];

#ifdef MIDI_DEBUG
  printf( "gus_midi_extended_control: dev=%i, chn=%i, cmd=0x%x, val=0x%x\n", device, chn, p1, p2 );
#endif
  if ( !midi_out_devices ) return -1;
  if ( device == GUS_MIDID_COMMON )	/* special channel handling */
    if ( ( device = search_output_channel( chn ) ) == GUS_MIDID_COMMON ) return -1;
  memcpy( sysex, GUS_SPECIAL_SYSEX, GUS_SPECIAL_SYSEX_SIZE );
  sysex[ GUS_SPECIAL_SYSEX_SIZE + 0 ] = 'c';	/* extended control */
  sysex[ GUS_SPECIAL_SYSEX_SIZE + 1 ] = chn & 0x0f;
  sysex[ GUS_SPECIAL_SYSEX_SIZE + 2 ] = ( p1 >> 8 ) & 0x7f;
  sysex[ GUS_SPECIAL_SYSEX_SIZE + 3 ] = ( p1 >> 0 ) & 0x7f;
  sysex[ GUS_SPECIAL_SYSEX_SIZE + 4 ] = ( p2 >> 8 ) & 0x7f;
  sysex[ GUS_SPECIAL_SYSEX_SIZE + 5 ] = ( p2 >> 0 ) & 0x7f;
  return gus_midi_sysex( device, sysex, GUS_SPECIAL_SYSEX_SIZE + 6 );
}

#endif

int gus_midi_sysex( int device, unsigned char *data, int len )
{
  gus_midi_device_t *devo;

#ifdef MIDI_DEBUG
  printf( "gus_midi_sysex: dev=%i, data=0x%x, len=%i\n", device, (int)data, len );
#endif
  if ( !midi_out_devices ) return -1;
  if ( !data || len <= 0 || len > 512 ) return -1;
  if ( device == GUS_MIDID_COMMON )	/* special channel handling */
    {
      for ( devo = midi_out_devices; devo; devo = devo -> next )
        {
          device_select( devo -> device, len + 2 );
#if defined( MIDI_DEBUG ) || 0
          printf( "gus_midi_sysex - dev %i: ", devo -> device );
          {
            int i;
         
            for ( i = 0; i < len; i++ )
            printf( "%02x:", data[ i ] );
          }
          printf( "\n" );
#endif
          put_byte( GUS_MCMD_COMMON_SYSEX );
          while ( len-- > 0 )
            put_byte( *data++ );
          put_byte( GUS_MCMD_COMMON_SYSEX_END );
        }
    }
   else
    {
      device_select( device, len + 2 );
#if defined( MIDI_DEBUG ) || 0
      printf( "gus_midi_sysex: " );
      {
        int i;
         
        for ( i = 0; i < len; i++ )
        printf( "%02x:", data[ i ] );
      }
      printf( "\n" );
#endif
      put_byte( GUS_MCMD_COMMON_SYSEX );
      while ( len-- > 0 )
        put_byte( *data++ );
      put_byte( GUS_MCMD_COMMON_SYSEX_END );
    }
  return 0;
}

int gus_midi_master_volume( int device, int volume )
{
  static unsigned char gm_master_volume[ 6 ] = { 0x7f, 0x7f, 0x04, 0x01, 0x00, 0x00 };
  int back = 0;

  gm_master_volume[ 5 ] = volume > 127 ? 127 : volume;
  if ( !realtime_flag )
    {
      back++;
      if ( gus_midi_realtime( 1 ) ) return -1;
    }
  if ( gus_midi_sysex( device, gm_master_volume, 6 ) < 0 ) return -1;
  if ( gus_midi_write() ) return -1;
  if ( back )
    if ( gus_midi_realtime( 0 ) ) return -1;
  return 0;
}

int gus_midi_data_raw( int device, unsigned char *data, int len )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_data_raw: device = %i, data = 0x%lx, len = %i\n", device, (long)data, len );
#endif
  if ( len > 4096 - 3 ) return -1;	/* too long */
  device_select( device, len );
  while ( len-- > 0 )
    put_byte( *data );
  return 0;
}

int gus_midi_echo( unsigned char *data, int len )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_echo: data = 0x%lx, len = %i\n", (long)data, len );
#endif
  if ( len > 4096 - 3 ) return -1;	/* too long */
  device_select( GUS_MIDID_COMMON, len + 1 );
  put_byte( GUS_MCMD_ECHO );
  while ( len-- > 0 )
    put_byte( *data++ );
  return 0;
}

int gus_midi_timer_base( int base )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_timer_base: base = %i\n", base );
#endif
  if ( base < 5 ) base = 5;
  if ( base > 1000 ) base = 1000;
  timer_base = base;
  if ( timer_running )
    {
      device_select( GUS_MIDID_COMMON, 3 );      
      put_byte( GUS_MCMD_TBASE );
      put_byte( (unsigned char)(base >> 8) );
      put_byte( (unsigned char)base );
    }
   else
    if ( ioctl( midi_fd, GUS_MIDI_TIMER_BASE, &base ) < 0 )
      {
        gus_dprintf( "gus_midi_timer_base: %s\n", strerror( errno ) );
        return -1;
      }
  return 0;
}

int gus_midi_timer_tempo( int tempo )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_timer_tempo: tempo = %i\n", tempo );
#endif
  if ( tempo < 5 ) tempo = 5;
  if ( tempo > 400 ) tempo = 400;
  timer_tempo = tempo;
  if ( timer_running )
    {
      device_select( GUS_MIDID_COMMON, 3 );      
      put_byte( GUS_MCMD_TEMPO );
      put_byte( (unsigned char)(tempo >> 8) );
      put_byte( (unsigned char)tempo );
    }
   else
    if ( ioctl( midi_fd, GUS_MIDI_TIMER_TEMPO, &tempo ) < 0 )
      {
        gus_dprintf( "gus_midi_timer_tempo: %s\n", strerror( errno ) );
        return -1;
      }
  return 0;
}

int gus_midi_timer_start( void )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_timer_start\n" );
#endif
  if ( ioctl( midi_fd, GUS_MIDI_TIMER_START ) < 0 )
    {
      gus_dprintf( "gus_midi_timer_start: %s\n", strerror( errno ) );
      return -1;
    }
  timer_running = 1;
  return 0;
}

int gus_midi_timer_stop( void )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_timer_stop\n" );
#endif
  if ( ioctl( midi_fd, GUS_MIDI_TIMER_STOP ) < 0 )
    {
      gus_dprintf( "gus_midi_timer_stop: %s\n", strerror( errno ) );
      return -1;
    }
  timer_running = 0;
  return 0;
}

int gus_midi_timer_continue( void )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_timer_continue\n" );
#endif
  if ( !timer_running )
    {
      if ( ioctl( midi_fd, GUS_MIDI_TIMER_CONTINUE ) < 0 )
        {
          gus_dprintf( "gus_midi_timer_continue: %s\n", strerror( errno ) );
          return -1;
        }
      timer_running = 1;
    }
  return 0;
}

int gus_midi_wait( int ticks )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_wait: %i ticks\n", ticks );
#endif
  if ( !gus_midi_cards ) return -1;
  if ( ticks <= 0 ) return -1;
  if ( !realtime_flag )
    {
      device_select( GUS_MIDID_COMMON, 5 );
      put_byte( GUS_MCMD_WAIT );
      put_byte( (unsigned char)(ticks >> 24) );
      put_byte( (unsigned char)(ticks >> 16) );
      put_byte( (unsigned char)(ticks >>  8) );
      put_byte( (unsigned char)(ticks >>  0) );
    }
   else
    return -1;
  return 0;
}

int gus_midi_write( void )
{
  int result = 0;

#ifdef MIDI_DEBUG
  printf( "gus_midi_write\n" );
#endif
  if ( !midi_out_devices ) return -1;
  if ( midi_out_used > 0 )
    {
#if 0
      {
        int i;
      
        printf( ">%s>> ", realtime_flag ? "realtime" : "" );
        for ( i = 0; i < midi_out_used; i++ )
          printf( "%02x:", midi_out_buffer[ i ] );
        printf( "\n" );
      }
#endif
      if ( !realtime_flag )
        {
          if ( write( midi_fd, midi_out_buffer, midi_out_used ) != midi_out_used )
            {
              gus_dprintf( "gus_midi_write: %s\n", strerror( errno ) );
              result = -1;
            }
        }
       else
        {
          struct GUS_STRU_MIDI_REALTIME realtime;
        
          realtime.buffer = midi_out_buffer;
          realtime.count = midi_out_used;
          if ( ioctl( midi_fd, GUS_MIDI_REALTIME, &realtime ) < 0 )
            {
              gus_dprintf( "gus_midi_write (realtime): %s\n", strerror( errno ) );
              result = -1;
            }
        }
      midi_out_used = 0;
    }
  return result;
}

int gus_midi_flush( void )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_flush\n" );
#endif
  if ( !midi_out_devices ) return -1;
  gus_midi_write();
  if ( ioctl( midi_fd, GUS_MIDI_FLUSH ) < 0 )
    {
      gus_dprintf( "gus_midi_flush: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_abort( void )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_abort\n" );
#endif
  midi_out_used = 0;
  if ( ioctl( midi_fd, GUS_MIDI_ABORT ) < 0 )
    {
      gus_dprintf( "gus_midi_abort: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_threshold( int threshold )
{
#ifdef MIDI_DEBUG
  printf( "gus_midi_threshold: threshold = %i\n", threshold );
#endif
  if ( ioctl( midi_fd, GUS_MIDI_THRESHOLD, &threshold ) < 0 )
    {
      gus_dprintf( "gus_midi_threshold: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

/*
 *  MIDI INPUT
 */

int gus_midi_input( gus_midi_callbacks_t *callbacks )
{
  unsigned char *ptr, *ptr1;
  int size, errors = 0, count, count1, device, cmd;
  
  if ( !callbacks || callbacks -> version < 17 ) return -1;
  while ( ( size = read( midi_fd, midi_in_buffer, midi_in_size ) ) > 0 )
    {
#if 0 
      printf( "gus_midi_input: start - size = %i\n", size );
      printf( "data = " );
      {
        int ii;
        for ( ii = 0; ii < size; ii++ )
          printf( "%02x:", midi_in_buffer[ ii ] );
      }
      printf( "\n" );
#endif
      ptr = midi_in_buffer;
      while ( size > 0 )
        {
          count = ( *ptr << 8 ) | *(ptr + 1);
          device = *(ptr + 2);
#if 0
          printf( "gus_midi_input: size = %i, count = %i, device = %i, cmd1 = %i\n", size, count, device, *(ptr + 3) );
#endif
          size -= count + 3;
          ptr += 3;
          while ( callbacks && count > 0 )
            {
              cmd = *ptr++; count--;
              if ( device == GUS_MIDID_COMMON )	/* common event */
                {
                  switch ( cmd ) {
                    case GUS_MCMD_ECHO:
                      if ( callbacks -> call_echo )
                        callbacks -> call_echo( callbacks -> privatedata, ptr, count );
                      ptr += count;
                      count = 0;
                      break;
                    case GUS_MCMD_WAIT:
                      if ( callbacks -> call_wait )
                        callbacks -> call_wait( callbacks -> privatedata, ( *ptr << 24 ) | ( *(ptr + 1) << 16 ) | ( *(ptr + 2) << 8 ) | *(ptr + 3) );
                      ptr += count;
                      count = 0;
                      break;
                    default:
                      errors++;
                  }
                  continue;
                }
              switch ( cmd & 0xf0 ) {
                case GUS_MCMD_NOTE_OFF:
                  if ( callbacks -> call_note_off )
                    callbacks -> call_note_off( callbacks -> privatedata, device, cmd & 0x0f, *ptr, *(ptr+1) );
                    count -= 2;
                    ptr += 2;
                    break;
                case GUS_MCMD_NOTE_ON:
                  if ( callbacks -> call_note_on )
                    callbacks -> call_note_on( callbacks -> privatedata, device, cmd & 0x0f, *ptr, *(ptr+1) );
                  count -= 2;
                  ptr += 2;
                  break;
                case GUS_MCMD_NOTE_PRESSURE:
                  if ( callbacks -> call_note_off )
                    callbacks -> call_note_pressure( callbacks -> privatedata, device, cmd & 0x0f, *ptr, *(ptr+1) );
                  count -= 2;
                  ptr += 2;
                  break;
                case GUS_MCMD_CONTROL:
                  if ( callbacks -> call_control )
                    callbacks -> call_control( callbacks -> privatedata, device, cmd & 0x0f, *ptr, *(ptr+1) );
                  count -= 2;
                  ptr += 2;
                  break;
                case GUS_MCMD_PGM_CHANGE:
                  if ( callbacks -> call_program_change )
                    callbacks -> call_program_change( callbacks -> privatedata, device, cmd & 0x0f, *ptr );
                  count--;
                  ptr++;
                  break;
                case GUS_MCMD_CHANNEL_PRESSURE:
                  if ( callbacks -> call_channel_pressure )
                    callbacks -> call_channel_pressure( callbacks -> privatedata, device, cmd & 0x0f, *ptr );
                  count--;
                  ptr++;
                  break;
                case GUS_MCMD_BENDER:
                  if ( callbacks -> call_bender )
                    callbacks -> call_bender( callbacks -> privatedata, device, cmd & 0x0f, ( *ptr & 0x7f ) | ( ( *(ptr+1) & 0x7f ) << 7 ) );
                  count -= 2;
                  ptr += 2;
                  break;
                case 0xf0:
                  switch ( cmd ) {
                    case GUS_MCMD_COMMON_SYSEX:
                      count1 = 0;
                      ptr1 = ptr;
                      while ( *ptr1 != GUS_MCMD_COMMON_SYSEX_END )
                        {
                          ptr1++;
                          count1++;
                        }
                      if ( callbacks -> call_sysex )
                        callbacks -> call_sysex( callbacks -> privatedata, device, ptr, count1 );
                      count += count1 + 1;
                      ptr += count1 + 1;
                      break;
                    case GUS_MCMD_COMMON_MTC_QUARTER:
                      if ( callbacks -> call_mtc_quarter )
                        callbacks -> call_mtc_quarter( callbacks -> privatedata, device, *ptr );
                      ptr++;
                      count--;
                      break;
                    case GUS_MCMD_COMMON_SONG_SELECT:
                      if ( callbacks -> call_song_select )
                        callbacks -> call_song_select( callbacks -> privatedata, device, *ptr );
                      ptr++;
                      count--;
                      break;
                    case GUS_MCMD_COMMON_SONG_POS:
                      if ( callbacks -> call_song_position )
                        callbacks -> call_song_position( callbacks -> privatedata, device, ( *ptr & 0x7f ) | ( *(ptr+1) << 7 ) );
                      ptr += 2;
                      count -= 2;
                      break;
                    case GUS_MCMD_COMMON_TUNE_REQUEST:
                      if ( callbacks -> call_tune_request )
                        callbacks -> call_tune_request( callbacks -> privatedata, device );
                      break;
                    case GUS_MCMD_COMMON_START:
                      if ( callbacks -> call_start )
                        callbacks -> call_start( callbacks -> privatedata, device );
                      break;
                    case GUS_MCMD_COMMON_CONTINUE:
                      if ( callbacks -> call_continue )
                        callbacks -> call_continue( callbacks -> privatedata, device );
                      break;
                    case GUS_MCMD_COMMON_STOP:
                      if ( callbacks -> call_stop )
                        callbacks -> call_stop( callbacks -> privatedata, device );
                      break;
                    default:
                      ptr += count;
                      count = 0;
                      errors++;
                  }
                  break;
                default:
                  ptr += count;
                  count = 0;
                  errors++;
              }
            }
        }
    }
  if ( size < 0 ) errors++;
  return errors ? -1 : 0;
}

/*
 *  MIDI THRU
 */

int gus_midi_thru_set( struct GUS_STRU_MIDI_THRU *thru )
{
  if ( ioctl( midi_fd, GUS_MIDI_SET_THRU, thru ) < 0 )
    {
      gus_dprintf( "gus_midi_set_thru - ioctl error - '%s'", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_thru_get( struct GUS_STRU_MIDI_THRU *thru )
{
  if ( ioctl( midi_fd, GUS_MIDI_GET_THRU, thru ) < 0 )
    {
      gus_dprintf( "gus_midi_get_thru - ioctl error - '%s'", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_thru_clear( int i_dev )
{
  struct GUS_STRU_MIDI_THRU thru;
  
  if ( i_dev < 0 || i_dev > GUS_MIDID_LAST ) return -1;
  memset( &thru, 0xff, sizeof( thru ) );
  thru.device = i_dev;
  return gus_midi_thru_set( &thru );
}

int gus_midi_thru_channel_set( int i_dev, int i_chn, int o_dev, int o_chn, int o_vel )
{
  int res, dev, found, free1, i;
  struct GUS_STRU_MIDI_THRU thru;
  
  if ( i_dev < 0 || i_dev > 255 ) return -1;
  if ( o_dev < 0 || o_dev > 255 ) return -1;
  if ( i_chn < 0 || i_chn > 15 ) return -1;
  if ( o_chn < 0 || o_chn > 15 ) return -1;
  if ( o_vel > 255 ) o_vel = 255;
  if ( o_vel < 0 ) return -1;
  thru.device = i_dev;
  if ( ( res = gus_midi_thru_get( &thru ) ) < 0 ) return res;
  found = 0; free1 = -1;
  for ( i = 0; i <= GUS_MIDID_LAST; i++ )
    {
      dev = thru.routing[ i_chn ][ i ].device;
      if ( dev == o_dev )		/* change only */
        {
          thru.routing[ i_chn ][ i ].channel = o_chn;
          thru.routing[ i_chn ][ i ].velocity = o_vel;
          found++;
          break;
        }
       else
      if ( dev == 255 && free1 < 0 ) free1 = i;
    }
  if ( !found )
    {
      if ( free < 0 ) return -1;	/* can't set this routing... */
      thru.routing[ i_chn ][ free1 ].device = o_dev;
      thru.routing[ i_chn ][ free1 ].channel = o_chn;
      thru.routing[ i_chn ][ free1 ].velocity = o_vel;
    }
  return gus_midi_thru_set( &thru );
}

int gus_midi_thru_channel_clear( int i_dev, int i_chn, int o_dev )
{
  int res, i;
  struct GUS_STRU_MIDI_THRU thru;
  
  if ( i_dev < 0 || i_dev > 255 ) return -1;
  if ( o_dev < 0 || o_dev > 255 ) return -1;
  if ( i_chn < 0 || i_chn > 15 ) return -1;
  thru.device = i_dev;
  if ( ( res = gus_midi_thru_get( &thru ) ) < 0 ) return res;
  for ( i = 0; i <= GUS_MIDID_LAST; i++ )
    if ( thru.routing[ i_chn ][ i ].device == o_dev )
      {
        thru.routing[ i_chn ][ i ].device = 255;
        thru.routing[ i_chn ][ i ].channel = 255;
        thru.routing[ i_chn ][ i ].velocity = 255;
        break;
      }
  if ( i > GUS_MIDID_LAST ) return 0;	/* or fail? */
  return gus_midi_thru_set( &thru );
}

/*
 *  INSTRUMENT PRELOADING
 */

int gus_midi_preload_bank( int device, char *bank_name )
{
  struct GUS_STRU_MIDI_PRELOAD_BANK bank;

#ifdef MIDI_DEBUG
  printf( "gus_midi_preload: bank = '%s'\n", bank_name );
#endif
  if ( !midi_out_devices ) return -1;
  bank.device = device;
  strcpy( bank.name, bank_name );
  if ( ioctl( midi_fd, GUS_MIDI_PRELOAD_BANK, &bank ) < 0 )
    {
      gus_dprintf( "gus_midi_preload_bank: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_preload_program( int device, int *instruments, int count )
{
  int i;
  struct GUS_STRU_MIDI_PRELOAD_INSTRUMENTS instr;

#ifdef MIDI_DEBUG
  printf( "gus_midi_preload_program - device = %i, count = %i, first = %i\n", device, count, *instruments );
#endif
  if ( !midi_out_devices ) return -1;
  while ( count > 0 )
    {
      instr.device = device;
      instr.instruments_used = count > 64 ? 64 : count;
      for ( i = 0; i < instr.instruments_used; i++ )
        instr.instruments[ i ] = *instruments++;
      count -= instr.instruments_used;
      if ( ioctl( midi_fd, GUS_MIDI_PRELOAD_INSTRUMENTS, &instr ) < 0 )
        {
          gus_dprintf( "gus_midi_preload_program: %s\n", strerror( errno ) );
          return -1;
        }
    }
  return 0;
}

/*
 *  ONBOARD MEMORY ALLOCATION
 */
 
int gus_midi_memory_reset_all( void )
{
  DEVICE *card;
  struct GUS_STRU_MIDI_MEMORY_RESET reset;
  int errors = 0;

  for ( card = midi_out_devices; card; card = card -> next )
    {
      int device = card -> device;
    
      if ( ( device & 0x0f ) != GUS_MIDID_SYNTH ) continue;
      reset.device = device;
      if ( ioctl( midi_fd, GUS_MIDI_MEMORY_RESET, &reset ) < 0 )
        {
          gus_dprintf( "gus_midi_memory_reset: %s\n", strerror( errno ) );
          errors++;
        }
    }
  return errors ? -1 : 0;
}

int gus_midi_memory_reset( int device )
{  
  struct GUS_STRU_MIDI_MEMORY_RESET reset;
  
  reset.device = device;
  if ( ioctl( midi_fd, GUS_MIDI_MEMORY_RESET, &reset ) < 0 )
    {
      gus_dprintf( "gus_midi_memory_reset: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_memory_test( int device, gus_instrument_t *instr )
{  
  struct GUS_STRU_MIDI_INSTRUMENT sinstr;
  
  sinstr.device = device;
  sinstr.instrument = instr;
  if ( ioctl( midi_fd, GUS_MIDI_MEMORY_TEST, &sinstr ) < 0 )
    {
      gus_dprintf( "gus_midi_memory_test: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_memory_alloc( int device, gus_instrument_t *instr )
{  
  struct GUS_STRU_MIDI_INSTRUMENT sinstr;
  
  sinstr.device = device;
  sinstr.instrument = instr;
  if ( ioctl( midi_fd, GUS_MIDI_MEMORY_ALLOC, &sinstr ) < 0 )
    {
      gus_dprintf( "gus_midi_memory_alloc: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_memory_free( int device, gus_instrument_t *instr )
{  
  struct GUS_STRU_MIDI_INSTRUMENT sinstr;
  
  sinstr.device = device;
  sinstr.instrument = instr;
  if ( ioctl( midi_fd, GUS_MIDI_MEMORY_FREE, &sinstr ) < 0 )
    {
      gus_dprintf( "gus_midi_memory_free: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_memory_pack( int device )
{  
  struct GUS_STRU_MIDI_MEMORY_PACK pack;
  
  pack.device = device;
  if ( ioctl( midi_fd, GUS_MIDI_MEMORY_RESET, &pack ) < 0 )
    {
      gus_dprintf( "gus_midi_memory_pack: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_memory_list( int device, struct GUS_STRU_MEMORY_LIST *list )
{
  struct GUS_STRU_MIDI_MEMORY_LIST mlist;
  
  mlist.device = device;
  mlist.list = list;
  if ( ioctl( midi_fd, GUS_MIDI_MEMORY_LIST, &mlist ) < 0 )
    {
      gus_dprintf( "gus_midi_memory_list: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_memory_block_alloc( int device, struct GUS_STRU_MEMORY_BLOCK *block )
{  
  struct GUS_STRU_MIDI_MEMORY_BLOCK sblock;
  
  sblock.device = device;
  sblock.block = block;
  if ( ioctl( midi_fd, GUS_MIDI_MEMORY_BALLOC, &sblock ) < 0 )
    {
      gus_dprintf( "gus_midi_memory_block_alloc: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_memory_block_free( int device, struct GUS_STRU_MEMORY_BLOCK *block )
{  
  struct GUS_STRU_MIDI_MEMORY_BLOCK sblock;
  
  sblock.device = device;
  sblock.block = block;
  if ( ioctl( midi_fd, GUS_MIDI_MEMORY_BFREE, &sblock ) < 0 )
    {
      gus_dprintf( "gus_midi_memory_block_free: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_memory_get_name( int device, gus_instrument_name_t *name )
{  
  struct GUS_STRU_MIDI_INSTRUMENT_NAME sname;
  
  sname.device = device;
  sname.name = name;
  if ( ioctl( midi_fd, GUS_MIDI_MEMORY_GET_NAME, &sname ) < 0 )
    {
      gus_dprintf( "gus_midi_memory_get_name: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_memory_dump( int device, struct GUS_STRU_MEMORY_DUMP *dump )
{  
  struct GUS_STRU_MIDI_MEMORY_DUMP sdump;
  
  sdump.device = device;
  sdump.dump = dump;
  if ( ioctl( midi_fd, GUS_MIDI_MEMORY_DUMP, &sdump ) < 0 )
    {
      gus_dprintf( "gus_midi_memory_dump: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_synth_info( int device, gus_info_t *info )
{  
  struct GUS_STRU_MIDI_CARD_INFO sinfo;
  
  sinfo.device = device;
  sinfo.info = info;
  if ( ioctl( midi_fd, GUS_MIDI_CARD_INFO, &sinfo ) < 0 )
    {
      gus_dprintf( "gus_midi_synth_info: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_effect_reset( int device )
{  
  struct GUS_STRU_MIDI_EFFECT_RESET sreset;
  
  sreset.device = device;
  if ( ioctl( midi_fd, GUS_MIDI_EFFECT_RESET, &sreset ) < 0 )
    {
      gus_dprintf( "gus_midi_effect_reset: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_midi_effect_setup( int device, struct GUS_STRU_EFFECT *effect )
{  
  struct GUS_STRU_MIDI_EFFECT seffect;
  
  seffect.device = device;
  seffect.effect = effect;
  if ( ioctl( midi_fd, GUS_MIDI_EFFECT_SETUP, &seffect ) < 0 )
    {
      gus_dprintf( "gus_midi_effect_setup: %s\n", strerror( errno ) );
      return -1;
    }
  return 0;
}
 
/*
 *
 */

static void gus_midi_icfg_error( void *privatedata, int device, int warning, char *filename, int line, char *format, va_list va )
{
  fprintf( stderr, ">> gus_midi_icfg_%s: device = %i, filename = '%s', line = %i - ", warning ? "warning" : "error", device, filename, line );
  vfprintf( stderr, format, va );
  fprintf( stderr, "\n" );
}

static int gus_midi_icfg_memory_alloc( void *privatedata, int device, gus_instrument_t *instrument )
{
  return gus_midi_memory_alloc( device, instrument );
}

static int gus_midi_icfg_memory_test( void *privatedata, int device, unsigned int instrument )
{
  gus_instrument_name_t name;
  
  name.number.instrument = instrument;
  if ( gus_midi_memory_get_name( device, &name ) < 0 ) return 0;
  if ( name.flags == GUS_INSTR_F_NOT_LOADED ) return 0;
  return 1;
}
 
int gus_midi_icfg_open( int device, char *filename, char *source, char *options )
{
  struct gus_icfg_open icfg;
  void *handle;

  icfg.version = 11;
  icfg.filename = !filename ? GUS_FILE_SYNTH_CONF : filename;
  icfg.card = ( device & 0xff ) >> 4;
  icfg.gus_fd = midi_fd;
  icfg.gus_access = gus_access_midi( device );
  icfg.device = device;
  icfg.emulation = gus_midi_emulation_get( device );
  if ( icfg.emulation < 0 ) return -1;
  icfg.chip = GUS_ICFG_CHIP_NONE;
  icfg.memory_alloc = gus_midi_icfg_memory_alloc;
  icfg.memory_test = gus_midi_icfg_memory_test;
  icfg.error = gus_midi_icfg_error;
  if ( gus_icfg_open( &handle, &icfg ) < 0 ) return -1;
  if ( !handle ) return -1;
  if ( gus_icfg_load( handle, source, options ) < 0 )
    {
      gus_icfg_close( handle );
      return -1;
    }
  return 0;
}

int gus_midi_icfg_close( int device )
{
  if ( gus_icfg_close( gus_icfg_handle( device ) ) < 0 ) {
    return -1;
  }
  return 0; 
}

int gus_midi_icfg_emulation_sync( int device )
{
  int emul1, emul2;
  void *handle;
  
  handle = gus_icfg_handle( device );
  if ( !handle ) return -1;
  emul1 = gus_icfg_emulation_get( handle );
  emul2 = gus_midi_emulation_get( device );
  if ( emul1 < 0 || emul2 < 0 )
    return -1;
  if ( emul1 != emul2 )
    return gus_icfg_emulation_set( handle, emul2 );
  return 0;
}

void gus_midi_icfg_error_handler( int device,
	void (*error)( int device, char *filename, int line, char *format, va_list va ),
	void (*warning)( int device, char *filename, int line, char *format, va_list va ) )
{
#if 0
  gus_icfg_config -> error = error;
  gus_icfg_config -> warning = warning;
#endif
}

int gus_midi_icfg_preload( int device, char *source, char *options )
{
  if ( gus_icfg_preload( gus_icfg_handle( device ), source, options ) )
    return -1;
  return 0;
}

int gus_midi_icfg_download_program( int device, int *instrument, int count )
{
  void *handle;

  handle = gus_icfg_handle( device );
  if ( !handle ) return -1;
  while ( count-- > 0 )
    if ( gus_icfg_download( handle, *instrument++ ) < 0 ) return -1;    
  return 0;
}

int gus_midi_icfg_info( int device, char *filename, struct gus_icfg_info **info )
{
  void *handle;
  struct gus_icfg_open icfg;

  icfg.version = 11;
  icfg.filename = !filename ? GUS_FILE_SYNTH_CONF : filename;
  icfg.card = ( device & 0xff ) >> 4;
  icfg.gus_fd = midi_fd;
  icfg.gus_access = gus_access_midi( device );
  icfg.device = device;
  icfg.emulation = gus_midi_emulation_get( device );
  if ( icfg.emulation < 0 ) return -1;
  icfg.chip = GUS_ICFG_CHIP_NONE;
  icfg.memory_alloc = gus_midi_icfg_memory_alloc;
  icfg.memory_test = gus_midi_icfg_memory_test;
  icfg.error = gus_midi_icfg_error;
  if ( gus_icfg_open( &handle, &icfg ) < 0 ) return -1;
  if ( !handle ) return -1;
  if ( gus_icfg_info( handle, info ) < 0 )
    {
      gus_icfg_close( handle );
      return -1;
    }
  gus_icfg_close( handle );
  return 0;  
}
