
#ifndef __render_h__
#define __render_h__

#include "texture2d.h"
#include "interpolator.h"
#include "fixed.h"
#include "pixels.h"
#include "inliners.h"

/** Clear : clear the texture to a specific color
 */
void Clear(CTexture2d& aTexture, PlatPixel aPixel);



/** Line : draw a line. Use class T to interpolate colors.
 * T should be any one of the Pixelizer
 */
template<class T>
inline void Line(PlatInt x0, PlatInt y0, PlatUInt z0, PlatPixel aColor0,
                 PlatInt x1, PlatInt y1, PlatUInt z1, PlatPixel aColor1,
                 T& aPixelize)
{
    CChangingColor color(aColor0,aColor1);

    CSingleInterpolate zinter;
    CInterpolator inter;
    inter.Initialize(x0,y0,x1,y1);
    PlatInt nr=inter.NrSteps();
    color.Initialize(0,nr-1);
    zinter.Initialize(0,nr-1);
    
    while ((nr--) > 0)
    {
        PlatInt x,y,z;
        PlatInt   weight;
        inter.Get(x,y);
        weight = zinter.CurrentValue();
        z = Interpolate(z0,z1,weight);
        aPixelize.PutPixel(x,y,z,color());
        inter.Next();
        color.Next();
    }
}

/** Triangle : render a filled triangle.
 */
template<class T>
inline void Triangle(PlatInt x0, PlatInt y0, PlatInt z0, PlatPixel aColor0,
                     PlatInt x1, PlatInt y1, PlatInt z1, PlatPixel aColor1,
                     PlatInt x2, PlatInt y2, PlatInt z2, PlatPixel aColor2,
                     T& aPixelize)
{
    // First sort by y:
    if (y0>y1)
    {
        Triangle(
                 x1, y1, z1, aColor1,
                 x0, y0, z0, aColor0,
                 x2, y2, z2, aColor1,
                 aPixelize);
        return;
    }
    if (y1>y2)
    {
        Triangle(
                 x0, y0, z0, aColor0,
                 x2, y2, z2, aColor1,
                 x1, y1, z1, aColor1,
                 aPixelize);
        return;
    }

    LISPASSERT(y0<=y1);
    LISPASSERT(y1<=y2);

    // Line from 0 to 2 (the long line)
    CChangingColor c0(aColor0,aColor2);
    CSingleInterpolate i0;
    i0.Initialize(y0,y2);
    c0.Initialize(0,y2-y0);

    // The upper line from 0 to 1
    CChangingColor c1(aColor0,aColor1);
    CSingleInterpolate i1;
    i1.Initialize(y0,y1);
    c1.Initialize(0,y1-y0);
    
    PlatInt y;
    
    // First render triangle from y0 to y1
    y=y0;
    while (y<y1)
    {
        PlatInt weight;

        weight = i0.CurrentValue();
        PlatInt xfrom=Interpolate(x0,x2,weight);
        PlatInt zfrom=Interpolate(z0,z2,weight);

        weight = i1.CurrentValue();
        PlatInt xto=Interpolate(x0,x1,weight);
        PlatInt zto=Interpolate(z0,z1,weight);

        CSingleInterpolate line;
        line.Initialize(xfrom,xto);

        while (line.Done()<=0)
        {
            PlatPixel clow;
            PlatPixel chigh;
            PlatPixel color;
            clow  = Mix(aColor0,aColor2,c0.CurrentValue());
            chigh = Mix(aColor0,aColor1,c1.CurrentValue());
            weight = line.CurrentValue();
            color = Mix(clow,chigh,weight);
            PlatInt z=Interpolate(zfrom,zto,weight);
            aPixelize.PutPixel(line.CurrentIndex(),y,z,color);
            line.Next();
        }
        i0.Next();
        c0.Next();
        i1.Next();
        c1.Next();
        y++;
    }

    
    
    // Now render triangle from y1 to y2
    CChangingColor c2(aColor1,aColor2);
    CSingleInterpolate i2;
    i2.Initialize(y1,y2);
    c2.Initialize(0,y2-y1);

    while (y<=y2)
    {
        PlatInt weight;

        weight = i0.CurrentValue();
        PlatInt xfrom=Interpolate(x0,x2,weight);
        PlatInt zfrom=Interpolate(z0,z2,weight);

        weight = i2.CurrentValue();
        PlatInt xto=Interpolate(x1,x2,weight);
        PlatInt zto=Interpolate(z1,z2,weight);

        CSingleInterpolate line;

        line.Initialize(xfrom,xto);

        while (line.Done()<=0)
        {
            PlatPixel clow;
            PlatPixel chigh;
            PlatPixel color;
            clow  = Mix(aColor0,aColor2,c0.CurrentValue());
            chigh = Mix(aColor1,aColor2,c2.CurrentValue());
            weight = line.CurrentValue();
            color = Mix(clow,chigh,weight);
            PlatInt z=Interpolate(zfrom,zto,weight);
            aPixelize.PutPixel(line.CurrentIndex(),y,z,color);
            line.Next();
        }
        i0.Next();
        c0.Next();
        i2.Next();
        c2.Next();
        y++;
    }
}

#endif

