/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "HeightFieldReader.h"
#include "FileIO.h"
#include "HeightFieldOps.h"
#include "GlobalTrace.h"
#include "GlobalSanityCheck.h"

#ifdef HAVE_IMLIB
#include <gdk_imlib.h>
#endif



/*
 *  constructor: initialize
 */
HeightFieldReader::HeightFieldReader (HeightField *HF, FileIO *Fio) 
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightFieldReader\n");

	SanityCheck::bailout ((!HF), "HF==NULL", "HeightFieldReader::HeightFieldReader");
	SanityCheck::bailout ((!Fio), "Fio==NULL", "HeightFieldReader::HeightFieldReader");

	p_HF = HF;
	p_Fio = Fio;
	Bitmap_File_Head = NULL;
	Bitmap_Head = NULL;
	Bitmap_OS2_Head = NULL;
}


/*
 *  destructor: nothing to clean up
 */
HeightFieldReader::~HeightFieldReader ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- HeightFieldReader\n");
}


/*
 *  checkRead: see if all required points were read. If not, fill missing 
 *  		points with 0
 */
void HeightFieldReader::checkRead (unsigned long points_read, 
				   unsigned long npoints)
{
	int	dotinterval=(int)(p_HF->getSize()/50); 
	char	buf[80];

	if (points_read != npoints) 
		{
		for (unsigned long ii=points_read; ii<npoints; ii++)
			{
			p_HF->setEl (ii, 0);
			if (!(ii%dotinterval) && 
			    GlobalTrace::trace(GlobalTrace::TRACE_VERBOSE, "-") )
				;
			}
		GlobalTrace::trace(GlobalTrace::TRACE_VERBOSE, "\n");
		sprintf (buf, _("%ld points missing"), npoints-points_read);
		}

	readOK ();
}


/*
 *  readOK: post-process the height field we just read 
 */
void HeightFieldReader::readOK ()
{
	p_HF->setName (p_Fio->getName()); 
	p_HF->gatherStatistics ();
}


#ifdef HAVE_IMLIB
/*
 *  readImlib: read an image using the imlib library using the red and 
 *  green spectrum to get a 16 bit value
 */
int HeightFieldReader::readImlib ()
{
	int		w, h, i;
	unsigned char	r, g;		// use red+green, discard blue
	GdkImlibImage	*image; 

	gdk_imlib_init ();
	image = gdk_imlib_load_image (p_Fio->getName());
	if (!image || image->rgb_width<5 || image->rgb_height<5)
		return -1;

	w = image->rgb_width;
	h = image->rgb_height;

	p_HF->newHF (w, h);

	for (i=0; i<p_HF->getSize(); i++)
		{
		int 	off=i*3;

		r = image->rgb_data [off++];
		g = image->rgb_data [off];
		p_HF->setEl (i, (((PTYPE)r*256.0+(PTYPE)g)/65535.0));
		}

	checkRead (i, p_HF->getSize());
	return(0);
}
#endif // HAVE_IMLIB


/*
 *  readPGM: allocate memory and read a PGM binary or ascii file
 */
int HeightFieldReader::readPGM ()

{
	FILE		*fp;
	char		*fname;
	int		dotinterval=1;
	unsigned long 	npoints=0;	// # points that should be in file 
	unsigned long 	points_read=0;	// # points read from hf datafile 
	char 		sbuf[512],	// buffer for file input 
			dummy[2]; 	// used for \n and \0   
	char 		*junk;		// unconverted part of a number 
	PTYPE		a, maxval;
	BYTE 		fbin; 		// PGM binary format flag 
	BYTE 		oneb; 		// one byte from file 
	int		x, y, rval=0;

	fp = p_Fio->open ("r");
	if (!fp)
		return -1;

	fname = p_Fio->getName ();

	/* *** check read *** */
	if (fread(sbuf,sizeof(char),2,fp) != 2)
		{
		SanityCheck::warning ("Error reading file", "HeightFieldReader::readPGM");
		return (-1); 
		}
	else 
		sbuf[2] = 0x00;

	fbin = 0;				// assume PGM ascii 
	if (strcmp (sbuf,"P5") == 0) 
		fbin = 1;			// PGM binary format 

	if (strcmp (sbuf,"GI") == 0) 
		{
		p_Fio->close ();
 		 } 
	else 
	if ((strcmp (sbuf,"P5") != 0) && (strcmp (sbuf,"P2") != 0)) 
		{
		p_Fio->close ();
		cerr << "Bad file type\n";
		return (-1);
		} 
	/* *** read PGM ASCII or Binary file *** */
	else 
		{ 
		/* *** read header *** */
		while ((fscanf(fp,"%s",sbuf) != EOF) && sbuf[0] == '#') 
			{
			fscanf(fp,"%[^\n]",sbuf);  	// read comment beyond '#' 
			fscanf(fp,"%[\n]",dummy);  	// read newline 
			}

		x = atoi(sbuf);				// store xsize of array 
		fscanf(fp,"%s",sbuf);			// read ysize of array 
		y = atoi(sbuf);				// store ysize of array
		fscanf(fp,"%s\n",sbuf);			// read maxval of array 
		maxval = strtod(sbuf, &junk);		// store maxval. could throw away.
		p_HF->newHF (x, y);

		if (GlobalTrace::isSet (GlobalTrace::TRACE_VERBOSE))
			{
			printf ("%s is a PGM %s file (xsize=%d, ysize=%d)\nreading: \n", 
				fname, (fbin ? "BINARY" : "ASCII"), x, y);
        		dotinterval=(int)(p_HF->getSize()/50); 
			}

		npoints = ((unsigned long int)p_HF->getSize()); /* # of points to read in */

		/* *** read PGM binary *** */
		if (fbin) 
			{
			while ((fread(&oneb,sizeof(BYTE),1,fp) == 1) && (points_read<npoints)) 
				{
				p_HF->setEl(points_read++, ((PTYPE)oneb/255));
				if (!(points_read%dotinterval) &&
				    GlobalTrace::trace(GlobalTrace::TRACE_VERBOSE , ".") )
					;
				}
			}
		/* *** read PGM ascii *** */
		else 
			{
			while ((fscanf(fp,"%s",sbuf) != EOF) && (points_read < npoints) ) 
				{
				a = strtod( sbuf, &junk);
				p_HF->setEl(points_read++, ((PTYPE)a/maxval));
				if (!(points_read%dotinterval) &&
				    GlobalTrace::trace(GlobalTrace::TRACE_VERBOSE, ".") )
					;
				}
			}
		} 

	p_Fio->close ();

	checkRead (points_read, npoints);

	return(rval);
} 


/*
 *  readTGA: read a TGA file (using red and green spectrum for a 16 bit value)
 */
int HeightFieldReader::readTGA ()
{
	FILE		*fp;
	char		*fname;
	int	 	ix, iy, dotinterval=1;
	unsigned long 	npoints;		// # points that should be in file 
	unsigned long 	points_read=0;		// # points read from hf datafile 

	unsigned char 	red, green, blue;
	unsigned char 	IDLength;		// length of Identifier String 
	unsigned char 	CoMapType;		// 0 = no map 
	unsigned char 	ImgType;		// image type (see below for values) 
	unsigned char 	Index_lo, Index_hi;	// index of first color map entry 
	unsigned char	Length_lo, Length_hi;	// number of entries in color map 
	unsigned char 	CoSize;			// size colormap entry (15,16,24,32) 
	unsigned char 	X_org_lo, X_org_hi;	// x origin of image 
	unsigned char 	Y_org_lo, Y_org_hi;	// y origin of image 
	unsigned char 	Width_lo, Width_hi;	// width of image 
	unsigned char 	Height_lo, Height_hi;	// height of image 
	unsigned char 	PixelSize;		// pixel size (8,16,24,32) 
	unsigned char 	Dbyte;			// pixel size (8,16,24,32) 
   
	fp = p_Fio->open ("r");
	if (!fp)
		return -1;

	fname = p_Fio->getName ();

	/* *** read 18 byte header *** */
	fscanf(fp,"%c",&IDLength);
	fscanf(fp,"%c",&CoMapType);
	fscanf(fp,"%c",&ImgType);
	fscanf(fp,"%c",&Index_lo);
	fscanf(fp,"%c",&Index_hi);
	fscanf(fp,"%c",&Length_lo);
	fscanf(fp,"%c",&Length_hi);
	fscanf(fp,"%c",&CoSize);
	fscanf(fp,"%c",&X_org_lo); 
	fscanf(fp,"%c",&X_org_hi);
	fscanf(fp,"%c",&Y_org_lo); 
	fscanf(fp,"%c",&Y_org_hi);
	fscanf(fp,"%c",&Width_lo);
	fscanf(fp,"%c",&Width_hi);		/* file in lo-byte, hi-byte order b12,b13*/
	fscanf(fp,"%c",&Height_lo);
	fscanf(fp,"%c",&Height_hi);		/* ysize b14, b15 */
	fscanf(fp,"%c",&PixelSize);
	fscanf(fp,"%c",&Dbyte);		/* descriptor byte b17 */
  
	ix = (unsigned int) Width_lo + 256*Width_hi;
	iy = (unsigned int) Height_lo + 256*Height_hi;
	p_HF->newHF (ix, iy);

	if (GlobalTrace::isSet (GlobalTrace::TRACE_VERBOSE))
		{
		printf ("%s is a TGA image file (type=%d, xsize=%d, ysize=%d)\nreading: ", 
			fname, (int)ImgType, ix, ix);
        	dotinterval=(int)(p_HF->getSize()/50); 
		}

	fflush (stdout);
	npoints = ((unsigned long int)p_HF->getSize()); /* # of points to read in */

	/* *** read tga data *** */
	for (iy = 0; iy < p_HF->getHeight(); iy++) 
		for (ix = 0; ix < p_HF->getWidth(); ix++) 
			if (fscanf(fp,"%c%c%c",&blue,&green,&red) == EOF) 
				{
				ix = p_HF->getWidth(); 
				iy = p_HF->getHeight();
				}
			else 
				{
				p_HF->setEl (points_read++, (((PTYPE)red*256.0+(PTYPE)green)/65535.0));
				if (!(points_read%dotinterval) && 
				    GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, ".") )
					;
				}

	p_Fio->close ();

	checkRead (points_read, npoints);

	return(0);
}



/* 
 * readMAT: read in data matrix in Matlab Level 1.0 format 
 *          and put data in header "x" and data array "hf"
 */

int HeightFieldReader::readMAT ()
{
	int 		cflag; 		// complex type flag 0=real, 1=complex 
	char 		pname[80]="dat1\x00"; 	// data matrix name, null-terminated
	Fmatrix 	hd;		// Level 1.0 MAT-file header 
	unsigned long 	points_read=0;	// # points read from hf datafile 
	unsigned long 	npoints=0;	// # points that should be in file 

	int 		a,b,c,i,j;
	int 		etype;
	int 		mtype, prec, rem, numtype;
	double 		tdbl;
	float 		tflt;
	FILE 		*fp;
	char		*fname;
	int		dotinterval=1;

	// --------- read in Matlab-format (Level 1.0) file ------- 
	etype = (NUM_FMT*1000) + (PREC*10);    // expected data type flag 

	fp = p_Fio->open ("r");
	if (!fp)
		return -1;

	fname = p_Fio->getName ();

	a = fread(&hd, sizeof(Fmatrix),(size_t)1, fp);   // read header 
	if (a != 1) 
		{
		SanityCheck::warning ("Error reading file header", "HeightFieldReader::readMAT");
		return (-1); 
		}

	numtype = (hd.type % 10000)/1000;
	prec = (hd.type % 100)/10;
	mtype = (hd.type % 10);
	rem = (hd.type - (1000*numtype + 10*prec + mtype));
	cflag = hd.imagf;
	//printf("Data type flag: M%d P%d T%d R%d CPLX%d\n", numtype,prec,mtype,rem,cflag); 

	if ((prec!=0) && (prec!=1))
		{
		perror ("Unsupported floating-point type\n");
		return (1);
		}
	//if ((numtype!=NUM_FMT) || (mtype!=0))  
	//	{
	//	perror ("Unsupported file format type\n");
	//	return (1);
	//	}

	b = fread(pname, sizeof(char), (size_t)hd.namelen, fp); /* read mx name */
	p_HF->newHF (hd.ncols, hd.mrows);
	npoints = p_HF->getSize();

	if (GlobalTrace::isSet (GlobalTrace::TRACE_VERBOSE))
		{
		printf ("%s is a MAT file (xsize=%d, ysize=%d) in %s format\nreading: \n", 
			fname, p_HF->getWidth(), p_HF->getHeight(), (prec ? "float" : "double"));
       		dotinterval=(int)(p_HF->getSize()/50); 
		}

   
	for (i=0; i<p_HF->getWidth(); i++) 	// read in floats 1 by 1 
	    { 
	    for (j=0; j<p_HF->getHeight(); j++) 
		{   		// ysize first 
		if (prec==0)
			c = fread (&tdbl, sizeof(double), 1, fp);
		else
			c = fread (&tflt, sizeof(float), 1, fp);

		if (c == 1) 
			{
			p_HF->setEl (i,j, (PTYPE)tflt);
			if (!(++points_read%dotinterval) &&
		    	    GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "."))
				;
			}
		else
			{
			i = p_HF->getWidth();
			j = p_HF->getHeight();
			}
		} 
	    } 

	p_Fio->close ();

	checkRead (points_read, npoints);

	return(0);
} 


/*
 *  readOCT: read an OCT file  
 */
int HeightFieldReader::readOCT ()
{
	int 	xsize, ysize;
	int 	i;
	unsigned long 	idx, points_read;
	long int file_pos, file_pos_new;
	FILE 	*fp;
	int 	oct_flag, break_flag;
	int	dotinterval=1;
	PTYPE 	array[ ARRAY_SIZE ];
	char 	in_str[ STRING_SIZE ],
		oct_name[ NAME_SIZE ],
		oct_type[ NAME_SIZE ];
	char 	*str_ptr, *fname, 
		c;
	char 	*str_name = OCT_NAME,
		*str_type = OCT_TYPE,
		*str_xsize = OCT_ROWS,
		*str_col  = OCT_COLUMNS;
	PTYPE	*hf=NULL;
  
	fp = p_Fio->open ("r");
	if (!fp)
		return -1;

	fname = p_Fio->getName ();

	// initilize our data 
	xsize = ysize = 0;
	idx = points_read = 0;
	file_pos = file_pos_new = 0;             
	oct_flag = OCT_None;      	// received no info about OCT file yet 
	oct_name[0] = oct_type[0] = 0;	// strings of length zero 
	break_flag = 0;			// flag to indicate a break in a loop 
  
	do 
		{
		file_pos = ftell(fp);
		if (!fgets(in_str, ARRAY_SIZE, fp)) 	// read in an entire line 
			{ 
			// premature end of file (?)
			if(!xsize || !ysize || !points_read) 
				{  
				perror ("Unexpected end of file (1) encountered while reading file." );
				return (1);
				}
			return 0; 	// assume that we're done reading this image 
			}
    
    		file_pos_new = ftell( fp );
    
		// find the first non-whitespace 
		str_ptr = in_str;
    		while (isspace(*str_ptr) && *str_ptr)  
			str_ptr++;
    
		// check to see if the file has image meta-data 
		if (*str_ptr == META_DATA_CHARACTER)
			{
			// This line contains information about the image 
			str_ptr++;
      			while (isspace(*str_ptr) && *str_ptr)  
				str_ptr++;
      
			// check for the image name 
			if (!strncmp (str_ptr, str_name, strlen(str_name)))
				{
				// this has information about the image name 
				str_ptr += strlen (str_name);
				sscanf (str_ptr, "%s", oct_name);
				oct_flag |= OCT_Name;
				// assume that there's no more information on this line 
				continue;
				}
			else 
			// check for the image type 
			if (!strncmp (str_ptr, str_type, strlen(str_type)))
				{
				// this has information about the image type 
				str_ptr += strlen (str_type);
				sscanf (str_ptr, "%s", oct_type);
				oct_flag |= OCT_Type;
				// assume that there's no more information on this line 
				continue;
				}

			// check for the number of xsize */
			if (!strncmp (str_ptr, str_xsize, strlen(str_xsize)))
				{
				// this has information about the number of xsize 
				str_ptr += strlen (str_xsize);
				sscanf (str_ptr, "%d", &ysize);
				oct_flag |= OCT_Rows;
				// assume that there's no more information on this line */
				continue;
				}
			else 
			// check for the number of ysize 
			if (!strncmp (str_ptr, str_col, strlen(str_col)))
				{
				// this has information about the number of ysize
				str_ptr += strlen (str_col);
				sscanf (str_ptr, "%d", &xsize);
				oct_flag |= OCT_Columns;
				// assume that there's no more information on this line 
				continue;
				}
			else 			// do this on default 
				{
				// this line is either a comment(?) or the parse didn't 
				// go correctly...  skip this line 
				continue;
      				}
    			}
    
		// is our first character part of a number? 
		c = *str_ptr;
		if (isdigit(c) || c == '+' || c == '-' )
			{
			// we're dealing with a number of some sort 
        		// do we know the dimensions of this image ?
			if ((oct_flag & OCT_Rows) && (oct_flag & OCT_Columns))
				{
				if (GlobalTrace::isSet (GlobalTrace::TRACE_VERBOSE))
					{
					printf ("%s is an OCT image file (xsize=%d, ysize=%d)\nreading: ", 
						fname, xsize, ysize);
        				dotinterval=(int)((xsize*ysize)/50); 
					}

				// Yep, we have all that we need to start. 
				hf = new PTYPE[xsize*ysize];
        
				// assume that the rest of the information in the file is image data
				// go back to the start of the line and start reading 
				fseek (fp, file_pos, SEEK_SET);
				for (i = points_read; i<xsize*ysize; i++)
					{
					if (fscanf(fp, "%e", &hf[i]) != 1)
						{
						printf ("Unexpected end of file (2) encountered when reading element %ld [.oct].", points_read);
						return ( 1 );
						}
					if (! (++points_read%dotinterval))
						{
						printf (".");
						fflush (stdout);
						}
					}

				if (GlobalTrace::isSet (GlobalTrace::TRACE_VERBOSE))
					{
					printf ("\n");
					fflush (stdout);
					}
				break_flag = 1;  // signify that we're done reading the file 
				break;
				}
			else 
			if (oct_flag & OCT_Columns)
				{
				// we only know the number of ysize 
				//  read in another row... 
				if (!ysize)
					{
					// haven't read in any data yet -- initilize the hf structure 
					hf = new PTYPE[xsize*(++ysize)];
					//h_newr (hfp, xsize, ++ysize, oct_name);
					}
				else
					{
					// have already read in some image data, reallocate memory for one more row 
					++ysize; 
					hf = (PTYPE *)realloc (hf, (size_t)xsize*ysize*sizeof(PTYPE));
					if (!hf)
						{
						perror ("Memory allocation error when reading [.oct] file ");
						return 1;
						}
					}
        
				// go back to the start of the line and start reading 
				fseek (fp, file_pos, SEEK_SET);
        
				/* read in the next row */
				for (i=0; i<xsize; i++)
					if (fscanf (fp, "%e", &hf[i+points_read]) != 1)
						{
						perror ("Unexpected end of file (3) encountered when reading [.oct].");
						return 1;
						}
				points_read += xsize;  	// keep track of the number of elements read 

				// done reading this line -- continue parsing the image file 
				fseek (fp, file_pos_new, SEEK_SET);  // go to the next line 

				continue;
				}
			else  			// if (oct_flag & OCT_Rows)
				{
				// we know the number of xsize but not the number of ysize 
				// I don't know what type of situation would use this...  
        
				// conditions leading to this section of code:
				//   o  have not read any image data yet
				//   o  xsize = 0  
				//   o  points_read = 0 
          
				while (sscanf(str_ptr, "%e", &array[xsize]) == 1)
					{
					xsize++;  // count the number of elements in this row 
					// skip to next number in the string 
					while (!isspace(*(++str_ptr))) ;
					while (isspace(*(++str_ptr))) ;
					}
        
				if ( !xsize )
					{
					perror ("Error reading:  No elements read in [.oct] file.");
					perror ("Do not understand the following:");
					perror (str_ptr);
					return 1;
					}
        
				points_read += xsize;

				// allocate memory for this image 
				hf = new PTYPE [xsize*(++ysize)]; 
          
				// copy the array contents into hfp 
				memcpy (hf, array, (size_t)xsize*sizeof(PTYPE));
        
				// we now know the width of the image data 
				oct_flag |= OCT_Columns;
				continue;
				}
			}
    
    		perror ("Do not understand the following:  ");
    		perror (str_ptr);
  		} 
	while (!break_flag);
      
	p_Fio->close ();

	p_HF->init (hf, xsize, ysize);

	checkRead (points_read, p_HF->getWidth());

	return 0;
}


/* ******************************************************************** */
/* *** readGTOPO is based on code from MarkStock's dem2pgm program **** */
/* ******************************************************************** */
int HeightFieldReader::readGTOPO()
{
	FILE		*fptr; 
	char		*fname, 
			*junk;
	char		name[128],
			buf1[20], buf2[20];
	int		dotinterval=1;
	unsigned long 	npoints=0;	// # points that should be in file 
	unsigned long 	points_read=0;	// # points read from hf datafile 
	int 		xsize=0, ysize=0;
	int 		row_bytes;
	int 		ix=0, iy=0;
	short int	twob;

	fname = p_Fio->getName ();

	// Read the .HDR file for pertinent information 
	(void)sprintf(name, "%s.HDR", p_Fio->getBasename());
	fptr = fopen(name ,"rb");
	if (!fptr)
		{
		fprintf (stderr,"Could not open input file %s\n", name);
		return -1;
		}

	// parse the data 
	while (fscanf(fptr,"%s %s", buf1, buf2) != EOF) 
		{
		if (!strncmp (buf1,"NROWS",2))
			ysize = (int) strtol(buf2, &junk, 10);
		else 
		if (!strncmp (buf1,"NCOLS",2))
			xsize = (int) strtol(buf2, &junk, 10);
		row_bytes = (int) strtol(buf2, &junk, 10);

		fprintf(stderr,"   read variable %s with value %s\n",buf1,buf2);
		}

	// close the HDR file 
	fclose(fptr);
	fprintf(stderr,"File closed\n\n");

	/*
	* Data is stored in DEM file as short ints, 16-bit data, -32k to +32k,
	* starting at the top left corner and going across, same as a PGM, good.
	* Some values will be -9999, that is where there is no data, just make
	* those the average of its neighbors.
	*/

	// open file 
	(void)sprintf(name ,"%s.DEM", p_Fio->getBasename());
	fptr = fopen(name,"rb");
	if (fptr==NULL) 
		{
		fprintf (stderr,"Could not open input file %s\n", name);
		return -1;
		}

	p_HF->newHF (xsize, ysize);

	if (GlobalTrace::isSet (GlobalTrace::TRACE_VERBOSE))
		{
		printf ("%s is a TGA image file (xsize=%d, ysize=%d)\nreading: ", 
			name, xsize, ysize);
        	dotinterval=(int)(p_HF->getSize()/50); 
		}

	fflush (stdout);
	npoints = ((unsigned long int)p_HF->getSize()); /* # of points to read in */

	// there's an extra byte at the beginning! 
	fread (&twob, 1, 1, fptr);

	// then, read a row, and skip the appropriate # of rows 
	for (iy=0; iy<ysize; iy++) 
		{
		// then, read a number and skip a few 
		for (ix=0; ix<xsize; ix++) 
			{
			// read two bytes, store it as a short int 
			fread (&twob, sizeof(short int), 1, fptr);

			// store the value in the array 
			p_HF->setEl (points_read++, (PTYPE)twob);
			}
		}
	fclose(fptr);

	for (ix=1, iy=p_HF->getSize()-2; ix<xsize; ix++, iy--)
		{
		if (p_HF->getEl(ix) == -9999)
			p_HF->setEl(ix, (p_HF->getEl(ix-1) + p_HF->getEl(ix+1))/2);
		if (p_HF->getEl(iy) == -9999)
			p_HF->setEl(iy, (p_HF->getEl(iy-1) + p_HF->getEl(iy+1))/2);
		}
	iy = p_HF->getSize()-xsize;
	for (ix=xsize; ix<iy; ix++) 
		if (p_HF->getEl(ix) == -9999)
			p_HF->setEl (ix, (p_HF->getEl(ix-1) + p_HF->getEl(ix+1) + 
					 p_HF->getEl(ix+xsize) + p_HF->getEl(ix-xsize))/4);

	checkRead (points_read, npoints);

	return 0;
}



/* ******************************************************************** */
/* **** readDEM is based on code from MarkStock's dem2pgm program ***** */
/* ******************************************************************** */
int HeightFieldReader::readDEM()
{
	FILE		*fptr; 
	char		*fname;
	char		name[144];
	int		dotinterval=1;
	unsigned long 	npoints=0;	// # points that should be in file 
	unsigned long 	points_read=0;	// # points read from hf datafile 
	int 		i, j;
	int 		maxelev, minelev;
	int 		maxxsize, xsize, ysize;
	int 		rowoffset;
	int 		ix=0, iy=0;
	double 		row_ymin, ymin, yres, yval;
	double xres, xval;

	fptr = p_Fio->open ("r");
	if (!fptr)
		return -1;

	fname = p_Fio->getName ();

	// DEM TYPE A RECORDS 
	// get the name field (144 characters) 
	for (i=0; i<144; i++) 
		name[i] = fgetc (fptr); 
	name[i+1] = '\0';

	// clean off the whitespace at the end 
	for (i=strlen(name)-2; i>0; i--) 
        	if (!isspace(name[i])) 
			i=0;
        	else 
			name[i] = '\0';

	fprintf (stdout, "Quad name field: %s\n", name);

	// don't need the next 19 items for anything 
	for (i=0; i<4; i++) 
		(void) DEMnextint();
	for (i=0; i<15; i++) 
		(void) DEMnextdouble();
	fprintf(stdout, "Units code (ground planametric coordinates): %i\n", DEMnextint());
	fprintf(stdout, "Units code (elevation coordinates): %i\n", DEMnextint());
	(void) DEMnextint();

	printf("Ground coordinates of 4 corners of DEM: (in arc-seconds) \n");
	for (i=0; i<4; i++) 
		{
		xval = DEMnextdouble();
		yval = DEMnextdouble();
		fprintf(stdout, "  %.4f  %.4f\n", xval, yval);
		if (i==0) 
			ymin = yval;
		if (yval < ymin) 
			ymin = yval;
		}

	minelev = (int)DEMnextdouble();
	maxelev = (int)DEMnextdouble();
	fprintf(stdout, "\nMinimum elevation: %i\n", minelev);
	fprintf(stdout, "Maximum elevation: %i\n", maxelev);

	// only need one of the next items 
	(void)DEMnextdouble();
	(void)DEMnextint();
	xres = (double)DEMnextfloat();
	yres = (double)DEMnextfloat();
	(void)DEMnextfloat();

	ix = DEMnextint();
	ysize = DEMnextint();
	if ((ysize-1)%300 == 0) 
		maxxsize = 1201; 	// all 1-deg DEMs are 1201x1201 
        // but some alaskan DEMs are 601, or 301 ysize wide 
    	else 				// just to be safe, 7.5' DEMs 
		maxxsize = 470;		// are 463 xsize high 

	// we need to see how many xsize of data before we write the header 
	// FIND the next int, its not just the next 6 chars 
	ix = DEMfindint();
	iy = DEMnextint();
	// Note: this number of xsize is not the total number of xsize! 
	xsize = DEMnextint();

	// set up HF
	p_HF->newHF (maxxsize, ysize);

	if (GlobalTrace::isSet (GlobalTrace::TRACE_VERBOSE))
		{
		printf ("%s is a DEM image file (xsize=%d, ysize=%d)\nreading: ", 
			fname, maxxsize, ysize);
        	dotinterval=(int)(p_HF->getSize()/50); 
		}

	fflush (stdout);
	npoints = ((unsigned long int)p_HF->getSize()); // # of points to read in 

	// now ready to read the image data 
	while (iy <= ysize) 
		{
		(void)DEMnextint();
		(void)DEMnextdouble();
		row_ymin = DEMnextdouble();
		for (i=0; i<3; i++) 
			(void)DEMnextdouble();

		// determine row offset if the data begins some xsize up from 
		// the lower edge, but only if the file is not a 1-deg DEM 
		if ((ysize-1)%300 != 0) 
			{
			rowoffset = (int)((row_ymin - ymin)/yres);
			if (rowoffset > 0) 
                		for (i=0; i<rowoffset; i++) 
					p_HF->setEl (points_read++, 0);
        		} 
		else 
            		rowoffset = 0;

		// read the data present 
		for (i=0; i<xsize; i++) 
			{ 
			j = DEMnextint(); 
			if (j < 0) 
				{
				if (j == -1) 
					j = DEMnextint();
				else 
				if (j == -2) 
					j = DEMfindint();
            			}
			p_HF->setEl (points_read++, (PTYPE)j);
			}

		// fill in all of the rest of the xsize with zero 
        	// fprintf(stderr,"iy=%d: i=%d+%d; i<%d; i++\n",iy,xsize,rowoffset,maxxsize); 
		for (i=xsize+rowoffset; i<maxxsize; i++) 
			p_HF->setEl (points_read++, 0);

		if (iy < ysize) 
			{
			ix = DEMfindint();
			iy = DEMnextint();
			xsize = DEMnextint();
			} 
		else 
			iy++;
		}

	checkRead (points_read, npoints);

	return 0;
}


/* 
 * search forward to find the next int  
 */ 
int HeightFieldReader::DEMfindint () 
{
	char	ch;
	int 	i = 0;
	char 	in[7];
	FILE	*fptr = p_Fio->getFilePtr ();

	while (isspace(ch = fgetc(fptr))) 
		;
	(void)ungetc (ch,fptr);
	while (!isspace(ch = fgetc(fptr))) 
		in[i++] = ch;
	(void)ungetc (' ',fptr);

	in[i] = '\0';
	return atoi(in);
}


/* 
 * take the next 6 chars and make them into an int
 */
int HeightFieldReader::DEMnextint () 
{
	char 	ch;
	int 	i = 0;
	int 	count;
	char 	in[7] = "";
	FILE	*fptr = p_Fio->getFilePtr ();

	for (count=0; count<6; count++) 
		{
		ch = fgetc(fptr);
		if (isspace(ch)) 
			{
			// check for LF in 7.5' DEMs 
			if (ch == 10) 
				return(-1); 
			} 
		else 
			in[i++] = ch;
		}

	// for 1-degree DEMs, there's no LFs 
	if (strlen(in) == 0) 
		return(-2);

	in[i] = '\0';
	return atoi(in);
} 

      
/* 
 * take the next 12 chars and make them into an float
 */
float HeightFieldReader::DEMnextfloat () 
{
	char 	ch;
	int 	i = 0;
	int 	count;
	char 	in[13];
	FILE	*fptr = p_Fio->getFilePtr ();

	for (count=0; count<12; count++) 
		{
		ch = fgetc(fptr);
		if (!isspace(ch)) 
			in[i++] = ch;
		}

	in[i] = '\0';
	return atof(in);
}


/* 
 * take the next 12 chars and make them into an float
 */
double HeightFieldReader::DEMnextdouble () 
{
	char 	ch;
	int 	i = 0;
	int 	count;
	char 	in[25];
	FILE	*fptr = p_Fio->getFilePtr ();

	for (count=0; count<24; count++) 
		{
		ch = fgetc(fptr);
		if (!isspace(ch)) 
			{
			if (ch == 'D') 
				ch = 'E';
			in[i++] = ch;
			}
		}

	in[i] = '\0';
	return atof(in);
}



/* ********************************************************* */
/* **** readBMP is based on the bmp code from the GIMP  **** */
/* ********************************************************* */ 
/* bmp.c                                          */
/* Version 0.44	                                  */
/* This is a File input and output filter for     */
/* Gimp. It loads and saves images in windows(TM) */
/* bitmap format.                                 */
/* Some Parts that deal with the interaction with */
/* the Gimp are taken from the GIF plugin by      */
/* Peter Mattis & Spencer Kimball and from the    */
/* PCX plugin by Francisco Bustamante.            */
/*                                                */
/* Alexander.Schulz@stud.uni-karlsruhe.de         */

/* Changes:   28.11.1997 Noninteractive operation */
/*            16.03.1998 Endian-independent!!     */
/*	      21.03.1998 Little Bug-fix		  */
/*            06.04.1998 Bugfix in Padding        */
/*            11.04.1998 Arch. cleanup (-Wall)    */
/*                       Parses gtkrc             */
/*            14.04.1998 Another Bug in Padding   */
/* ********************************************************* */ 

int HeightFieldReader::readBMP ()
{
	FILE 		*fd;
	char 		*fname;
	char 		buf[5];
	int 		ColormapSize, SpeicherZeile, Maps, Grey;
	unsigned char 	ColorMap[256][3];
	guchar 		buffer[50];
	int		rc;
  
	fd = p_Fio->open("r");
	if (!fd)
		return -1;

	fname = p_Fio->getName();
  
	Bitmap_File_Head = new struct Bitmap_File_Head_Struct;
	Bitmap_Head = new struct Bitmap_Head_Struct;
	Bitmap_OS2_Head = new struct Bitmap_OS2_Head_Struct;

	// Now is it a Bitmap? 
	if (!ReadOK(fd,buf,2) || (strncmp(buf,"BM",2)))
		{
		SanityCheck::warning (TRUE, "not a valid BMP file\n", "readBMP");
		return -1;
		}

	// How long is the Header? 
	if (!ReadOK (fd, buffer, 0x10))
		{
		SanityCheck::warning (TRUE, "error reading BMP file header\n", "readBMP");
		return -1;
		}

	// bring them to the right byteorder. Not too nice, but it should work 
	Bitmap_File_Head->bfSize=BMPToL(&buffer[0]);
	Bitmap_File_Head->reserverd=BMPToL(&buffer[4]);
	Bitmap_File_Head->bfOffs=BMPToL(&buffer[8]);
	Bitmap_File_Head->biSize=BMPToL(&buffer[12]);
  
	// Is it a Windows (R) Bitmap or not 
	if (Bitmap_File_Head->biSize!=40) 
		{
		SanityCheck::warning (TRUE, "OS/2 unsupported\n", "readBMP");
		if (!ReadOK (fd, buffer, Bitmap_File_Head->biSize))
			{
			SanityCheck::warning (TRUE, "error reading BMP file header\n", "readBMP");
			return -1;
			}

		Bitmap_OS2_Head->bcWidth=BMPToS(&buffer[0]);
		Bitmap_OS2_Head->bcHeight=BMPToS(&buffer[2]);
		Bitmap_OS2_Head->bcPlanes=BMPToS(&buffer[4]);
		Bitmap_OS2_Head->bcBitCnt=BMPToS(&buffer[6]);

		Bitmap_Head->biPlanes=Bitmap_OS2_Head->bcPlanes;
		Bitmap_Head->biBitCnt=Bitmap_OS2_Head->bcBitCnt;
		Bitmap_File_Head->bfSize=(Bitmap_File_Head->bfSize*4)-(Bitmap_File_Head->bfOffs*3);
		Bitmap_Head->biHeight=Bitmap_OS2_Head->bcHeight;
		Bitmap_Head->biWidth=Bitmap_OS2_Head->bcWidth;
		Bitmap_Head->biClrUsed=0;
		Bitmap_Head->biCompr=0;
		Maps=3;
    		}
	else
		{
		if (!ReadOK (fd, buffer, 36))
			{
			SanityCheck::warning (TRUE, "error reading BMP file header\n", "readBMP");
			return -1;
			}
		Bitmap_Head->biWidth=BMPToL(&buffer[0x00]);		/* 12 */
		Bitmap_Head->biHeight=BMPToL(&buffer[0x04]);		/* 16 */
		Bitmap_Head->biPlanes=BMPToS(&buffer[0x08]);		/* 1A */
		Bitmap_Head->biBitCnt=BMPToS(&buffer[0x0A]);		/* 1C */
		Bitmap_Head->biCompr=BMPToL(&buffer[0x0C]);		/* 1E */
		Bitmap_Head->biSizeIm=BMPToL(&buffer[0x10]);		/* 22 */
		Bitmap_Head->biXPels=BMPToL(&buffer[0x14]);		/* 26 */
		Bitmap_Head->biYPels=BMPToL(&buffer[0x18]);		/* 2A */
		Bitmap_Head->biClrUsed=BMPToL(&buffer[0x1C]);		/* 2E */
		Bitmap_Head->biClrImp=BMPToL(&buffer[0x20]);		/* 32 */
    					                		/* 36 */
		Maps=4;
		}
  
	// This means wrong file Format. I test this because it could crash 
	if (Bitmap_Head->biBitCnt>24) 
		{
		SanityCheck::warning (TRUE, "can't handle BMP file with > 24 bits color\n", "readBMP");
		return -1;
		}

	// There should be some colors used! 
	ColormapSize = (Bitmap_File_Head->bfOffs-Bitmap_File_Head->biSize-14) / Maps;
	if ((Bitmap_Head->biClrUsed==0) && (Bitmap_Head->biBitCnt<24)) 
		Bitmap_Head->biClrUsed=ColormapSize;
	if (Bitmap_Head->biBitCnt==24) 
		SpeicherZeile=((Bitmap_File_Head->bfSize-Bitmap_File_Head->bfOffs)/Bitmap_Head->biHeight);
	else 
		SpeicherZeile=((Bitmap_File_Head->bfSize-Bitmap_File_Head->bfOffs)/Bitmap_Head->biHeight)*(8/Bitmap_Head->biBitCnt);
  
#ifdef DEBUG
	printf("\nSize: %u, Colors: %u, Bits: %u, Width: %u, Height: %u, Comp: %u, Zeile: %u\n",
		Bitmap_File_Head->bfSize,Bitmap_Head->biClrUsed,Bitmap_Head->biBitCnt,
		Bitmap_Head->biWidth, Bitmap_Head->biHeight, Bitmap_Head->biCompr, 
		SpeicherZeile);
#endif
  
	// Get the Colormap 
	if (BMPreadColormap(ColorMap, ColormapSize, Maps, &Grey) == -1) 
		return -1;
  
#ifdef DEBUG
	printf("Colormap read\n");
#endif

	// Get the Image and return the ID or -1 on error
	rc = BMPreadImage (Bitmap_Head->biWidth, Bitmap_Head->biHeight, 
			ColorMap, Bitmap_Head->biClrUsed, Bitmap_Head->biBitCnt, 
			Bitmap_Head->biCompr, SpeicherZeile, Grey);

	delete Bitmap_File_Head;
	delete Bitmap_Head;
	delete Bitmap_OS2_Head;

	return rc;
}



int HeightFieldReader::BMPreadColormap (unsigned char buffer[256][3], 
					int number, int size, int *grey)
{
	FILE		*fd;
	int 		i;
	unsigned char 	rgb[4];

	fd = p_Fio->getFilePtr ();
	if (!fd)
		return -1;
  
	*grey=(number>2);
	for (i = 0; i < number ; i++)
		{
		if (!ReadOK (fd, rgb, size))
			{
			SanityCheck::warning (TRUE, "bad colormap\n", "BMPreadColormap");
			return -1;
			}
      
      			// Bitmap save the colors in another order! 
			// But change only once! 
			if (size==4) 
				{      
				buffer[i][0] = rgb[2];
				buffer[i][1] = rgb[1];
				buffer[i][2] = rgb[0];
				} 
			else 
				{
				// this one is for old os2 Bitmaps, but it dosn't work well 
				buffer[i][0] = rgb[1];
				buffer[i][1] = rgb[0];
				buffer[i][2] = rgb[2];
				}
		*grey=((*grey) && (rgb[0]==rgb[1]) && (rgb[1]==rgb[2]));
		}
	return(0);
}


int HeightFieldReader::BMPreadImage (int len, int height, 
		unsigned char cmap[256][3], int ncols, int bpp, 
		int compression, int spzeile, int grey)
{
	FILE		*fd;
	unsigned char 	v,howmuch;
	char 		buf[16];
	int 		xpos = 0, 
			ypos = 0;
	guchar 		*dest, *temp;
	long 		rowstride;
	int 		i, j, cur_progress, max_progress, egal;
  
	fd = p_Fio->getFilePtr ();
	if (!fd)
		return -1;

	dest = new unsigned char[Bitmap_Head->biWidth*Bitmap_Head->biHeight];
	rowstride = Bitmap_Head->biWidth;
  
	ypos=height-1;		// Bitmaps begin in the lower left corner 
	cur_progress = 0;
	max_progress = height;
  
	if (bpp==24)
		{
		while (ReadOK(fd,buf,3))
			{
			temp = dest + (ypos * rowstride) + xpos;
			*temp=buf[2];
			temp++;
			*temp=buf[1];
			temp++;
			*temp=buf[0];
			xpos++;
			if (xpos == len)
				{
				egal=ReadOK(fd,buf,spzeile-(len*3));
				ypos--;
				xpos=0;
				cur_progress++;
				}
			if (ypos < 0) 
				break;
			}
		}
	else 
		{ switch(compression)
		    {
		    case 0: 		// uncompressed 
			{
			while (ReadOK(fd,&v,1))
			    {
			    for (i=1;(i<=(8/bpp)) && (xpos<len);i++,xpos++)
				{
				temp = dest + (ypos * rowstride) + xpos;
				// look at my bitmask !! 
				*temp=( v & ( ((1<<bpp)-1) << (8-(i*bpp)) ) ) >> (8-(i*bpp));
				}
			    if (xpos == len)
				{
				egal=ReadOK(fd,buf,(spzeile-len)/(8/bpp));
				ypos--;
				xpos=0;
				cur_progress++;
				}
			   if (ypos < 0) 
				break;
			   }
			break;
			}

		   default:		// compressed 
		   	{
		    	while (TRUE)
			{
			  egal=ReadOK(fd,buf,2);
			  if ((unsigned char) buf[0]!=0) 
			  // Count+Color- record 
			    {
			    for (j=0;((unsigned char) j < (unsigned char) buf[0]) && (xpos<len);)
				{
#ifdef DEBUG2
				printf("%u %u | ",xpos,len);
#endif
				for (i=1;((i<=(8/bpp)) && (xpos<len) && ((unsigned char) j < (unsigned char) buf[0]));i++,xpos++,j++)
					{
					temp = dest + (ypos * rowstride) + xpos;
					*temp=( buf[1] & ( ((1<<bpp)-1) << (8-(i*bpp)) ) ) >> (8-(i*bpp));
					}
				}
			    }

			  // uncompressed record 
			  if (((unsigned char) buf[0]==0) && ((unsigned char) buf[1]>2))
			    {
			    howmuch=buf[1];
			    for (j=0;j<howmuch;j+=(8/bpp))
				{
				egal=ReadOK(fd,&v,1);
				i=1;
				while ((i<=(8/bpp)) && (xpos<len))
					{
					temp = dest + (ypos * rowstride) + xpos;
					*temp=(v & ( ((1<<bpp)-1) << (8-(i*bpp)) ) ) >> (8-(i*bpp));
					i++;
					xpos++;
					}
				}
			    if ( (howmuch / (8/bpp)) % 2) 
				egal=ReadOK(fd,&v,1);

			    /* if odd(x div (8 div bpp )) then blockread(f,z^,1);*/
			    }

			  // Zeilenende 
			  if (((unsigned char) buf[0]==0) && ((unsigned char) buf[1]==0))
				{
				ypos--;
				xpos=0;
				cur_progress++;
				}

			  // Bitmapende 
			  if (((unsigned char) buf[0]==0) && ((unsigned char) buf[1]==1))
				{
				break;
				}

			  // Deltarecord 
			  if (((unsigned char) buf[0]==0) && ((unsigned char) buf[1]==2))
				{
				xpos+=(unsigned char) buf[2];
				ypos+=(unsigned char) buf[3];
				}
			  break;
			  }
		       }
		    }
		}
 
	p_HF->newHF (Bitmap_Head->biWidth, Bitmap_Head->biHeight); 
	for (int i=0; i<p_HF->getSize(); i++)
		{
		if (bpp == 24)
			{
			int	off = i*3;
			p_HF->setEl (i, (((PTYPE)dest[off]*256.0 + (PTYPE)dest[off+1]) / 65535.0));
			}
		else
			p_HF->setEl (i, (((PTYPE)dest[i]*256.0 + (PTYPE)dest[i+1]) / 65535.0));
		}

	readOK ();
		
	g_free(dest);
	return(0);
}


gint32 HeightFieldReader::BMPToL(guchar *buffer)
{
	return(buffer[0] | buffer[1]<<8 | buffer[2]<<16 | buffer[3]<<24);
}

gint16 HeightFieldReader::BMPToS(guchar *buffer)
{
	return(buffer[0] | buffer[1]<<8);
}

