/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "HeightFieldIO.h"
#include "GlobalTrace.h"
#include "GlobalSanityCheck.h"



/*
 *  constructor: initialize
 */
HeightFieldIO::HeightFieldIO (const char *fname) : FileIO (fname)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightFieldIO\n");
}


/*
 *  destructor: initialize
 */
HeightFieldIO::~HeightFieldIO ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- HeightFieldIO\n");
}


/*
 *  getFileType: return the matching file type for a filename extension
 */
int HeightFieldIO::getFileType ()
{
	char 	*ext;

	if (!(ext=this->getExtension(false)))
		{
		printf (_("Unable to determine extension of file [%s]\n"), this->getName());
		return (-1);
		}
	else
	if (GlobalTrace::isSet (GlobalTrace::TRACE_FLOW))
		printf ("filename extension = %s\n", ext);

	if (!strcmp(ext,"gif")) 
		return (GIF);
	else if (!strcmp (ext,"pot")) 
		return (POT);
	else if (!strcmp (ext,"tga")) 
		return (TGA);
	else if (!strcmp (ext,"pgm")) 
		return (PGM);
	else if (!strcmp (ext,"pg8")) 
		return (PG8);
	else if (!strcmp (ext,"mat")) 
		return (MAT);
	else if (!strcmp (ext,"oct")) 
		return (OCT);
#ifdef HAVE_IMLIB
	else if (!strcmp (ext,"png")) 
		return (PNG);
	else if (!strcmp (ext,"tiff") || !strcmp (ext,"tif")) 
		return (TIFF);
#endif
	else if (!strcmp (ext,"dem")) 
		return (DEM);
	else if (!strcmp (ext,"gtop")) 
		return (GTP);
	else if (!strcmp (ext,"bmp"))
		return (BMP);
	else if (!strcmp (ext,"ac"))
		return (AC);
	else
		{ 
     		cerr << "Warning: " << ((ext) ? "unknown " : "missing ") << 
			"file extension (" << ext << ") ... defaulting to TGA \n";
		return (TGA);		// return default 
   		}
}


/*
 *  read: wrapper to call the proper read function 
 */
int HeightFieldIO::read (HeightField *HF)
{
	int			itype, rc=-1;
	char			buf[512];
	HeightFieldReader 	*HFR;

	SanityCheck::bailout ((!HF), "HF==NULL", "HeightFieldIO::read");

	if (!this->exists())
		{
		sprintf (buf, _("file [%s] doesn't exist"), this->getName());
		SanityCheck::warning (buf, "HeightFieldIO::read");
		return (-1);
		}

	itype = this->getFileType ();
	if (itype == -1)
		return (itype);

	HFR = new HeightFieldReader (HF, this);

	if (itype == PGM || itype == PG8)
   		rc = HFR->readPGM ();  
	else
	if (itype == TGA)
		rc = HFR->readTGA ();
	else
	if (itype == MAT)
		rc = HFR->readMAT ();
	else
	if (itype == OCT)
		rc = HFR->readOCT ();
	else
	if (itype == GTP)
		rc = HFR->readGTOPO ();
	else
	if (itype == DEM)
		rc = HFR->readDEM ();
	else
	if (itype == BMP)
		rc = HFR->readBMP ();
#ifdef HAVE_IMLIB
	else
	if (itype == PNG || itype == TIFF)
		rc = HFR->readImlib ();
#endif
	else
		{
		sprintf (buf, _("file type [%d] not implemented"), itype);
		SanityCheck::warning (buf, "HeightFieldIO::read");
		}

	if (rc)
		{
		sprintf (buf, _("file rype [%s] returned an Error (%d)\n"), this->getName(), rc);
		SanityCheck::warning (buf, "HeightFieldIO::read");
		}
	else
		HF->setLoadedFromFile (TRUE);

	delete HFR;
	return (rc);
} 


/*
 *  write: wrapper to call the proper write function 
 */
int HeightFieldIO::write (HeightField *HF, HeightFieldDraw *HFD)
{
	int			itype, rc=-1;
	char			buf[512];
	HeightFieldWriter	*HFW;

	SanityCheck::bailout ((!HF), "HF==NULL", "HeightFieldIO::write");
	//SanityCheck::bailout ((!HFD), "HFD==NULL", "HeightFieldIO::write");

	itype = this->getFileType ();
	if (itype == -1)
		return (itype);

	HFW = new HeightFieldWriter (HF, this);

	if (itype == TGA)
		rc = HFW->writeTGA ();
	else
	if (itype == PGM)
		rc = HFW->writePGM ();
	else
	if (itype == PG8)
		rc = HFW->writePG8 ();
	else
	if (itype == MAT)
		rc = HFW->writeMAT ();
	else
	if (itype == OCT)
		rc = HFW->writeOCT ();
	else
	if (itype == AC)
		{
		SanityCheck::bailout ((!HFD), "HFD==NULL", "HeightFieldIO::write");
		rc = HFW->writeAC (HFD);
		}
	else
		{
		sprintf (buf, _("file type [%d] not implemented"), itype);
		SanityCheck::warning (buf, "HeightFieldIO::read");
		}

	if (rc)
		{
		sprintf (buf, _("file write [%s] returned an Error (%d)\n"), this->getName(), rc);
		SanityCheck::warning (buf, "HeightFieldIO::read");
		}

	delete HFW;
	return (rc);
}

