/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef _HEIGHT_FIELD_CORE_H
#define _HEIGHT_FIELD_CORE_H


#include "GlobalDefs.h"
#include <time.h>

#define PTYPE		float

/*
 *  HeightFieldCore: a class to hold the basic info for 1 height field
 */ 
class HeightFieldCore
	{
	public:
				HeightFieldCore(PTYPE *hf, int xsize, int ysize);
				HeightFieldCore(int xsize=0, int ysize=0);
				~HeightFieldCore();
		int		gatherStatistics ();
		int		printStatistics ();
		void		pMinMax (PTYPE *pfloor, PTYPE *pceil);
		int		rescale (float floor=0.0, float ceil=1.0);

		// inline accessor methods returning data
		PTYPE		*getData () 		{return p_hf;}
		PTYPE		getMin ()		{return d_min;}
		PTYPE		getMax ()		{return d_max;}
		PTYPE		getSealevel ()		{return d_sealevel;}
		int		getWidth () 		{return d_xsize;}		
		int		getHeight () 		{return d_ysize;}
		int		getSize() 		{return d_size;}

		// mutator methods
		void		setSealevel (PTYPE sealevel);
		void		setEl (int off, PTYPE v);
		void		setEl (int x, int y, PTYPE v);

		// inline utility methods defined in this file 
		int		Off2Posx (int off);
		int		Off2Posy (int off);
		int		Pos2Off (int x, int y);
		PTYPE		getEl (int off);
		PTYPE		getEl (int x, int y);
		PTYPE		getElSea (int x, int y);
		PTYPE		getElmod (int x, int y);
		PTYPE		getElmodSea (int x, int y);
		PTYPE		getElclip (int x, int y);
		PTYPE		getElclipSea (int x, int y);

	protected:
		void		setData (PTYPE *hf, int xsize, int ysize, char *dummy=NULL);

		// overloaded operators
		//PTYPE		&operator[](int i);

		PTYPE		*p_hf,			// height field elevation data 
				d_min, d_max,		// pointers to min and max values
				d_avg,			// average elevation 
				d_sealevel;		// sealevel 
		int		d_xsize, d_ysize, 	// side lengths
				d_size;			// map size 
	};


// public inlined methods 

/*
 *  []: provide an array subscript operator we can access at class level
 */ 
//inline PTYPE& HeightFieldCore::operator[](int i)
//{
//	return p_hf[i];
//}


/*
 *  getEl: return the element at offset off
 */ 
inline PTYPE HeightFieldCore::getEl (int off)
{
#ifdef TF_DEBUG
	if (off < 0 || off >= d_size)
		{
		char	buf[80];

		sprintf (buf, "HF offset (%d) out of range", off);
		SanityCheck::bailout (buf, "HeightFieldCore::getEl");
		}
#endif
	return p_hf[off];
}


/*
 *  getEl: return the element at Position x,y
 */ 
inline PTYPE HeightFieldCore::getEl (int px, int py)
{
	return getEl(Pos2Off(px, py));
}


/*
 *  setEl: set the specified elemen's value 
 */ 
inline void HeightFieldCore::setEl (int off, PTYPE value)
{
#ifdef TF_DEBUG
	if (off < 0 || off >= d_size)
		{
		char	buf[80];

		sprintf (buf, "HF offset (%d) out of range", off);
		SanityCheck::bailout (buf, "HeightFieldCore::setEl");
		}
#endif
	p_hf[off] = value;
}


/*
 *  setEl: set the specified elemen's value 
 */ 
inline void HeightFieldCore::setEl (int x, int y, PTYPE value)
{
	setEl (Pos2Off(x,y), value);
}


/*
 *  getElSea: return the element at offset off
 */ 
inline PTYPE HeightFieldCore::getElSea (int px, int py)
{
	PTYPE	el = getEl(px,py);

	return (el>d_sealevel ? el : d_sealevel);
}


/*
 *  getElmod: return the element at (safe) mod(Position)
 */ 
inline PTYPE HeightFieldCore::getElmod (int px, int py)
{
	int 	mx = px%d_xsize,
		my = py%d_ysize,
		yoff = ((py>=0)?my:(my+d_ysize)%d_ysize),
		xoff = ((px>=0)?mx:(mx+d_xsize)%d_xsize);
	
	return getEl (xoff, yoff);
}


/*
 *  getElmod: return getElmod if point above sealevel 
 */ 
inline PTYPE HeightFieldCore::getElmodSea (int px, int py)
{
	PTYPE	el = getElmod(px,py);

	return (el>d_sealevel ? el : d_sealevel);
}


/*
 *  getElclip: return the point constrained to the actual HF boundries
 */ 
inline PTYPE HeightFieldCore::getElclip (int px, int py)
{
	int	yoff = ((py<0)?0:((py>=d_ysize)?(d_ysize-1):py)),
		xoff = ((px<0)?0:((px>=d_xsize)?(d_xsize-1):px));

	return getEl (xoff, yoff);
}


/*
 *  getElclipSea: return getElclip if point is above sealevel
 */ 
inline PTYPE HeightFieldCore::getElclipSea (int px, int py)
{
	PTYPE	el = getElclip(px,py);

	return (el>d_sealevel ? el : d_sealevel);
}


/*
 *  Off2Posx: convert a 1D offset to the x vector of a 2D position
 */ 
inline int HeightFieldCore::Off2Posx (int off)
{
	return off%d_xsize;
}


/*
 *  Off2Posy: convert a 1D offset to the y vector of a 2D position
 */ 
inline int HeightFieldCore::Off2Posy (int off)
{
	return ((int)(off/d_xsize));
}


/*
 *  Pos2Off: convert a 2D position to a 1D offset/vector. 
 */ 
inline int HeightFieldCore::Pos2Off (int px, int py)
{
	return ((int)(py*d_xsize)+px);
}

#endif // _HEIGHT_FIELD_CORE_H

