/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "GuiColormapBands.h"
#include "GlobalSanityCheck.h"
#include "GlobalTrace.h"


#define LOAD		0
#define SAVE		1
#define BANDSIZE	15


/*
 *  constructor: initialize everything
 */
GuiColormapBands::GuiColormapBands (Gtk_Widget *parent, int nBands, 
				    int colsPerBand, int landBandOffset) : 
		  GuiColormap (parent, nBands*colsPerBand)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ GuiColormapBands\n");

	d_nbands = nBands;
	d_bandsize = colsPerBand;
	d_bandLandOffset = landBandOffset;
	this->d_linear = FALSE;
}


/*
 *  destructor: nothing to clean up  
 */
GuiColormapBands::~GuiColormapBands ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- GuiColormapBands\n");
}


/*
 *  getBandColor: get the specified color from the specified color band
 */
GdkColor *GuiColormapBands::getBandColor (int band, int color, bool observe_offset)
{
	if (observe_offset)
		return (getColor (d_bandLandOffset*d_bandsize + 
				  band*d_bandsize + color));
	else
		return (getColor (band*d_bandsize+color));
}


/*
 *  getNumColor: return the number of color bands
 */
int GuiColormapBands::getNumBands ()
{
	return (d_nbands);
}


/*
 *  getBandSize: return the number of colors in a color band
 */
int GuiColormapBands::getBandSize ()
{
	return (d_bandsize);
}


/*
 *  getLandBandOffset: return the number of color bands which are underwater
 */
int GuiColormapBands::getLandBandOffset()
{
	return (d_bandLandOffset);
}


/*
 *  setLandBandOffset: set the number of underwater color bands
 */
int GuiColormapBands::setLandBandOffset(int bandLandOffset)
{
	if (bandLandOffset < 0 || bandLandOffset > d_nbands)
		return (-1);
	else
		d_bandLandOffset = bandLandOffset;
	return (0);
}


/*
 *  allocColorBand: allocate a whole color band
 */
bool GuiColormapBands::allocColorBand (	char 		*colname, // color name
					float		ambient)  // ambient factor ( must be (0<ambient<1) )
{
	GdkColor	basecol, col;
	float		top, bot, intensity;
	short int	red, green, blue;

	if ((addNamedColor (&basecol, colname)))
		{
		for (int shade=1; shade<d_bandsize; shade++)
			{
			top = basecol.red;
			bot = ambient*top;
			intensity = bot + ((shade * (top - bot))/d_bandsize);
			col.red = red = (short int)intensity;

			top = basecol.green;
			bot = ambient*top;
			intensity = bot + ((shade * (top - bot))/d_bandsize);
			col.green = green = (short int)intensity;
			
			top = basecol.blue;
			bot = ambient*top;
			intensity = bot + ((shade * (top - bot))/d_bandsize);
			col.blue = blue = (short int)intensity;

			if (!addColor (&col))
				{
				char	buf[80];

				sprintf (buf, _("Error allocating color %d, %d, %d ... Exiting\n"), 
					red, green, blue);
				SanityCheck::bailout (buf, "GuiColormapBands::allocColorBand");
				}
			}
		}
	else
		printf (_("Unable to allocate color [%s] ... Exiting\n"), colname);

	return (TRUE);
}


/*
 *  allocColorBands: allocte all the color bands 
 */
int GuiColormapBands::allocColorBands 	(char *colorNames[], float ambient)
{
	int 		i;

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Colormap: ");

	for (i=0; i<d_nbands; i++)
		{
		allocColorBand (colorNames[i], ambient);
		//if (!strcmp (colorNames[i], "Green2")) b_land_offset=i;
		}

	if (GlobalTrace::isSet (GlobalTrace::TRACE_VERBOSE))
		{
		char	buf[80];

		sprintf (buf, "Allocated %d colors in %d bands (band_land_offset=%d)  ...\n", 
			d_nbands*d_bandsize, d_nbands, d_bandLandOffset); 
		GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Colormap: ");
		fflush (stdout);
		}

	return (d_ncolors);
}


