/*
 * main.c: Main source file for XTux, including main game loop and game logic.
 * Copyright 1999 David Lawrence (philaw@camtech.net.au)
 * Last modified July 26.
 */

#include <math.h>
#include <signal.h>
#include "header.h"
#include "entity.h"
#include "main.h"
#include "win.h"
#include "map.h"
#include "input.h"
#include "timing.h"
#include "tile.h"
#include "weap.h"
#include "sound.h"
#include "image.h"

extern struct image_data_struct image_data[NUM_ENTITIES];

int fps = DEFAULT_FPS, frame_len = DEFAULT_FRAME_LEN;
/* Global Flags */
int trigger = 0, old_health = 0, old_weap = 0, change_weap = 0;
int status_dirty = 0, map_dirty = 0;
entity *root, *player, *focus;
weapon players_weapons;
struct timeval last_fired, ct, st, game_init;

int main(int argc, char *argv[])
{

  int i, lastfrlen;
  int game_state = INIT; /* menu, game etc.. */
  int screenx, screeny; /* view screen. */
  struct map_t map;

  for(i = 1; i < argc ; i++) {
    if(argv[i][0] == '-') {
      switch(argv[i][1]) {
      case 'f': /* set FPS */
	if( argc >= i + 1 ) {
	  fps = atoi(argv[++i]);
	  frame_len = 1000000 / fps;
	  fprintf(stderr, "fps = %d\n", fps);
	  break;
	} else {
	  fprintf(stderr, "Bad arguments!\n");
	  exit(1);
	}
	break;
      case 'v': /* Print version number */
	printf("Version: %s\n"
	       "Compiled with options: ", XTUXVERSION);
#if VERBOSE
	printf(" VERBOSE");
#endif
#if FOCUSFOLLOWSNUKE
	printf(" FOCUSFOLLOWSNUKE");
#endif
#if CHARS_RUN_IN_MENU
	printf(" CHARS_RUN_IN_MENU");
#endif
	printf(" Default FPS = %d\n", DEFAULT_FPS);
	exit(0);
      case 'h':
      default:
	printf("usage: %s [OPTIONS]\n\n"
	       "-f NUMBER,\tSet frames/second to NUMBER\n"
	       "-h,\t\tHelp (this screen)\n"
	       "-v,\t\tPrint version.\n\n"
	       "Report bugs to philaw@camtech.net.au\n", argv[0]);
	exit(0);
      }
    }
  }

  gettimeofday( &game_init, NULL );
  /* Seed random number generator */
  srand(game_init.tv_usec);

  if( signal(SIGSEGV, sig_handler) == SIG_ERR )
    die("Can't set signal handler\n", 1);

  /* Initialise root entity */
  if( (root = (entity *)malloc( sizeof(entity) )) == NULL)
    die("Malloc failed in creating root entity!\n", 1);

  player = create_entity(TUX);
  create_window();
  game_state = MENU;

  /* Master loop */
  while( 1 ) {

    switch( game_state ) {
    case MENU:

      menu(map.file_name, &player->type);

      /* Load map */
      if( !load_map(&map) )
	die("Error loading map\n", 1);
#if VERBOSE
      printf("\"%s\" Dimensions(%d,%d) Sucessfully loaded\n", map.name,
	     map.width, map.height);
#endif

      /* Init weapons */
      if( player->type == TUX || player->type == GOWN ) {
	players_weapons.has_cd = 1;
	player->weapon = WEAP_CD;
      }
      else
	players_weapons.has_cd = 0;
      
      if( player->type == BSD ) {
	players_weapons.has_fireball = 1;
	player->weapon = WEAP_FIREBALL;
      }
      else
	players_weapons.has_fireball = 0;
      
      player->health = 100;
      player->mode = ALIVE;
      players_weapons.num_nukes = 0;
      players_weapons.has_pitchfork = 0;
      players_weapons.has_bat = 0;
      player->dir = 4;

      gettimeofday( &last_fired, NULL );

      /* Initialsed so some things will be redrawn */
      map_dirty = 1;
      old_weap = 0;
      old_health = 0;
      focus = player;
      
      game_state = PLAYING;

    case PLAYING:
      /* main game loop */
      while( game_state == PLAYING ) {

	gettimeofday( &st, NULL); /* Time @ start of frame */
	
	if( old_weap != player->weapon ) {
	  draw_status_bar(&map, player->weapon);
	  /* Redraw the health bar */
	  update_caff_bar(player->health);
	  old_weap = player->weapon;
	  status_dirty = 1;
	}
	
	/* Health related things */
	if( old_health != player->health ) {
	  
	  if( player->health <= 0) {
	    fprintf(stderr,"All out of caffeine! You drift off to sleep...\n");
	    game_state = MENU;
	    break;
	  }
	  else if( player->health >= 100 ) {
	    player->health = 100;
	    player->speed = 16; /* May give speed bonus for 100 later */
	  }
	  else if( player->health >= 75 )
	    player->speed = 16;
	  else
	    player->speed = 12;
	  
	  /* Redraw the health bar */
	  update_caff_bar(player->health);
	  old_health = player->health;
	  status_dirty = 1;

	}

	handle_events(&game_state);
	
	update_entities(&map);
	check_entity_collisions(&map);
	update_weapons();
	
	calculate_view_window(&map, focus, &screenx, &screeny);
	update_screen(&map, screenx, screeny);
	draw_entities(screenx, screeny);
	update_window();
	
	gettimeofday( &ct, NULL); /* Current time */
	lastfrlen = time_diff( ct, st ); /* time it took to draw last frame */
	if( lastfrlen < frame_len )
	  delay( frame_len - lastfrlen );
      }
      
      remove_all_entities();
      /* Free map data */
      free(map.base);
      free(map.object);
      break;
    case QUIT:
      die("Quitting Xtux.\n", 0);

    } /* Switch */

  }

} /* main. */


/* Adjust screenx (sx) and screeny (sy) to keep it in view.
   Center on ent if possible (witout going out of bounds) */
void calculate_view_window(struct map_t *map, entity *ent, int *sx, int *sy)
{

  *sx = ent->x - VIEW_W/2 + TILE_W/2;
  *sy = ent->y - VIEW_H/2 + TILE_H/2;

  /* Check view area is inside map boundaries. */
  if( *sx >= map->width * TILE_W - VIEW_W )
    *sx = map->width * TILE_W - VIEW_W;
  else if( *sx < 0 )
    *sx = 0;
  
  if( *sy >= map->height * TILE_H - VIEW_H)
    *sy = map->height * TILE_H - VIEW_H;
  else if( *sy < 0 )
    *sy = 0;

}


void update_entities(struct map_t *map)
{

  /* BASE and OBJECT clip levels */
  int b_clip_level, o_clip_level;
  unsigned char *b_tile, *o_tile;
  entity *ent;

  ent = root->next;
  
  if( NULL == ent )
    die("update_entities: ent == NULL!\n", 1);

  for( ent = root->next ; ent != NULL ; ent=ent->next ) {

    if( ent->class == ITEM ) /* skip items */
      continue;

    /* Clean up gibbed entities */
    if( ent->mode == GIB
#if BLOOD_SPOTS_STAY
	&& ent->type != BLOOD_SPOT
#endif
#if BUNNY_HEADS_STAY
	&& ent->type != BUNNY_HEAD
#endif
#if BUNNY_BODIES_STAY
	&& ent->type != BUNNY_DEAD
#endif
	)
      if( time_diff(ct, ent->last_frame) >= GIB_EXPIRE_TIME ) {
	ent = remove_entity(ent->ent_id);
	continue;
      }

    /* Calculate image to be drawn */
    ent->img_no = calculate_frame(ent);

    /* Get rid of dead entities. After death animation is finished, entities
     will be set to dead in calculate frame */
    if( ent->mode == DEAD ) {
      ent = remove_entity(ent->ent_id);
      continue;
    }

    /* Stop dying entities */
    if( ent->mode == DYING ) {
      ent->x_v = ent->y_v = 0;
      continue;
    }

    /* Remove dead entities and make <= 0 health entities DYING or removed
     according to type */
    if( ent->health <= 0 && ent != player ) {
      if( image_data[ent->type].dying_ani == 1 ) {
	ent->frame = 0;
	ent->mode = DYING;
      } else
	ent = remove_entity(ent->ent_id);
      continue;
    }

    switch( ent->class ) {
    case GOODIE:
    case BADDIE:
      b_clip_level = map->tileset? TECH_BASE_SPECIAL : REAL_BASE_SPECIAL;
      o_clip_level = map->tileset? TECH_OBJECT_CLIP : REAL_OBJECT_CLIP;
      if( ent != player ) /* Don't move player */
	if( ent->type != MSCP || !(rand()%8) ) /* Peons act zombie like */
	  move_enemy(ent, HERO);
      break;
    case NEUTRAL:
      if( ent->mode == GIB ) { /* make bunnyhead fly off */
	if( time_diff(ct, ent->last_frame) >= GIB_DECEL_TIME && ent->speed>0){
	  ent->speed--;
	  ent->last_frame = ct;
	}
      } else if( ent != player )
        critter_ai(ent);
      b_clip_level = map->tileset? TECH_BASE_SPECIAL : REAL_BASE_SPECIAL;
      o_clip_level = map->tileset? TECH_OBJECT_CLIP : REAL_OBJECT_CLIP;
      break;
    case PROJECTILE:
      b_clip_level = map->tileset? TECH_BASE_WALL : REAL_BASE_WALL;
      o_clip_level = map->tileset? TECH_OBJECT_CLIP : REAL_OBJECT_CLIP;
      break;
    default:
      fprintf(stderr, "Tried to move unknown entity, of type %d\n", ent->type);
      break;
    }
    
    b_tile = world_collision(map, BASE, ent, b_clip_level);

    if( in_range(map,ent) && b_tile == NULL ) {
      
      o_tile = world_collision(map, OBJECT, ent, o_clip_level);

      if( o_tile == NULL ) {

	/* Check for hitting stuff like messages etc */
	if( ent == player )
	  special_objects(map, ent);
	ent->x += ent->x_v * ent->speed;
	ent->y += ent->y_v * ent->speed;
	
      } else { /* Hit a clipping object */
	
	switch( ent->class ) {
	case GOODIE:
	case BADDIE:
	case NEUTRAL:
	  ent->x_v = 0;
	  ent->y_v = 0;
	  break;
	case PROJECTILE:
	  if( ent->type == NUKE ) {
	    /* Remove object */
	    blowup(map, ent, (ent->x+ent->width/2)/TILE_W,
		   (ent->y+ent->height/2)/TILE_W, NUKE_RADIUS);
	    ent->speed = 0;
	    ent = remove_entity(ent->ent_id);
	    break;
	  }

#if USESOUND
	  if( ent->type == CD )
	    play_sound(S_CHINK);
#endif
	  /* In the real world, we can destroy some object tiles */
	  if( map->tileset == REAL_TILESET ) {
	    if(*o_tile >= REAL_OBJECT_DESTROY )
	      if( *o_tile < REAL_OBJECT_BROKEN ) /* Break it */
		*o_tile += REAL_OBJECT_BROKEN - REAL_OBJECT_DESTROY;
	    /* else
	        *o_tile = RO_RUBBLE; */
	    map_dirty = 1;
	  }

	  ent = remove_entity(ent->ent_id);
	  break;
	default:
	  fprintf(stderr, "Some unknown entity hit an object!!!\n");
	}

      } /* Hit an object */

      
    } else { /* Hit a wall */
      
      switch( ent->class ) {
      case GOODIE:
      case BADDIE:
      case NEUTRAL:
	ent->x_v = 0;
	ent->y_v = 0;
	break;
      case PROJECTILE:
	switch( ent->type ) {
	case NUKE:
	  blowup(map, ent, (ent->x+ent->width/2)/TILE_W,
		 (ent->y+ent->height/2)/TILE_W, NUKE_RADIUS);
	  ent = remove_entity(ent->ent_id);
	  break;
	case CD:
#if USESOUND
	  play_sound(S_CHINK);
#endif
	case FIREBALL:
	  ent = remove_entity(ent->ent_id);
	  break;
	default:
	  fprintf(stderr, "Some unknown entity hit a wall!!!\n");
	}
      }
      
    } /* hit a wall */
    
  }

} /* Move sprites */



void update_weapons(void)
{

  int fire_ok = 0;

  /* FIXME: Setup reload time so it's weapon dependant */
  gettimeofday(&ct, NULL);
  if( trigger && (time_diff(ct, last_fired) >= RELOAD_TIME)) {

    switch( player->weapon ) {
    case WEAP_CD:
      if( players_weapons.has_cd )
	fire_ok = 1;
      break;
    case WEAP_FIREBALL:
      if( players_weapons.has_fireball )
	fire_ok = 1;
      break;
    case WEAP_NUKE:
      if( players_weapons.num_nukes > 0 ) {
	fire_ok = 1;
	players_weapons.num_nukes--;
      }
    }

    if( fire_ok ) {
      fire_weapon(player, player->weapon );
      last_fired = ct;
    }

  }

  /* FIXME: Do this all properly!!!
     Perhaps we'll use the number keys ala quake/doom etc. */
  if( change_weap ) {

    if( player->type == TUX || player->type == GOWN ) {
      if( player->weapon == WEAP_CD )
	player->weapon = WEAP_NUKE;
      else player->weapon = WEAP_CD;
    } else if( player->type == BSD ) {
      if( player->weapon == WEAP_FIREBALL )
	player->weapon = WEAP_NUKE;
      else player->weapon = WEAP_FIREBALL;
    }
    change_weap = 0;

  }

}


void check_entity_collisions(struct map_t *map)
{

  entity *ent, *other;
  entity *first, *second;
  int entisfirst;

  ent = root;

  do {

    if( ent->next->next == NULL )
      break;
    else {
      ent = ent->next;
      other = ent;
    }
    
    if( ent->mode == DYING || ent->mode == GIB || ent->health <= 0 )
      continue; /* Skip him */

    do {

      other = other->next;

      /* One day we may worry about this, but at the moment
	 it saves a heap of cpu time when there are lots of peons (Dragula) */
      if( ent->type == MSCP && other->type == MSCP )
	continue;

      if( other->mode == DYING || other->mode == GIB || other->health <= 0)
	continue;
      
      /* If x is within range */
      if( abs(ent->x - other->x) <= ent->width + other->width ) {
	if( abs(ent->y - other->y) <= ent->height + other->height) {
	  /* Relies that entity height == width! */
	  if( entity_dist( ent, other ) <= ent->width/2 + other->width/2) {
	    
	    /* set the first type of entity to first, and keep track of what
	       was switched, so we can switch them back. */
	    if( ent->type <= other->type) {
	      first = ent;
	      second = other;
	      entisfirst = 1;
	    } else {
	      first = other;
	      second = ent;
	      entisfirst = 0;
	    }

	    /* Collision has occured, determine what happens via classes then
	     special entity type cases */
	    switch( first->class) {
	    case GOODIE:
	      
	      switch( second->class ) {
		/* Check for GOODIE'S??? */
	      case BADDIE:
		first->health -= 5;
		break;
	      case NEUTRAL:
		/* ??? */
		break;
	      case PROJECTILE:
		if( second->class == PROJECTILE )
		  if( second->owner_id == first->ent_id )
		    break; /* Dont' clip on your own Stuff */
		switch(second->type) {
		case NUKE:
		  blowup(map, second, (second->x+second->width/2)/TILE_W,
			 (second->y+second->height/2)/TILE_W, NUKE_RADIUS);
		  first->health -= NUKEHIT_DMG;
		  break;
		case CD:
		  first->health -= CD_DMG;
		  break;
		case FIREBALL:
		  first->health -= FIREBALL_DMG;
		  break;
		}
		/* Remove projectile */
		second = remove_entity(second->ent_id);
		break;
	      case ITEM:
		switch(second->type) {
		case NUKE_CRATE:
		  /* give player 1 more nuke */
#if USESOUND
		  play_sound(S_AMMO_PICKUP);
#endif
		  players_weapons.num_nukes++;
		  second = remove_entity(second->ent_id);
		  break;
		case CAN:
		  first->health += CAN_HEALTH;
		  second = remove_entity(second->ent_id);
		  break;
		case NODOZE:
		  first->health = 100;
		  second = remove_entity(second->ent_id);
		  break;
		case DISK:
		  /* Do something??? */
		  second = remove_entity(second->ent_id);
		  break;
		case SPOON:
		  second = remove_entity(second->ent_id);
		  draw_text_window("There is no spoon!");
		  break;
		default:
#if VERBOSE
		  fprintf(stderr, "Entity type %d hit entity %d\n",
			  first->type, second->type);
#endif
		  break;
		} /* ITEMS */
		break;
	      } /* Tux & BSD */
	      break;

	    case BADDIE:
	      switch( second->class ) {
	      case BADDIE:
		/* ???? */
		break;
	      case NEUTRAL:
		/* ??? */
		break;
	      case PROJECTILE:

		if( second->owner_id == first->ent_id )
		  break; /* Dont' clip on your own Stuff */

		switch( second->type ) {
		case CD:
		  first->health -= CD_DMG;
		  break;
		case FIREBALL:
		  first->health -= FIREBALL_DMG;
		  break;
		case NUKE:
		  first->health -= NUKEHIT_DMG;
		  blowup(map, second,(second->x+second->width/2)/TILE_W,
			 (second->y+second->height/2)/TILE_W, NUKE_RADIUS);
		  break;
		} /* Projectile */
		second = remove_entity(second->ent_id); /* Remove projectile */
		break;
	      case ITEM:
		/* ??? */
		break;
	      default:
#if VERBOSE
		fprintf(stderr, "Entity type %d hit entity %d\n",
			first->type, second->type);
#endif
		break;
	      } /* MSCP && CLIPPY */
	      break;
	    case NEUTRAL:
	      switch( first->type ) {
	      case BUNNY:
		switch( second->type ) {
		case CD:
		case FIREBALL:

		  first->mode = GIB;
		  first->last_frame = ct;		  
		  first->dir = getdir(first->x_v, first->y_v);
		  first->class = NEUTRAL;

		  second->mode = GIB;
		  second->last_frame = ct;
		  second->dir = getdir(second->x_v, second->y_v);
		  second->class = NEUTRAL;

		  if( (rand()%2) ) {
		    /* Make bunny DEAD */
		    first->type = BUNNY_DEAD;
		    first->img_no = 0; /* Headless */
		    first->speed = 4;
		    /* Turn proj. into head. */
		    second->x = first->x; /* start from body's posn */
		    second->y = first->y;
		    second->type = BUNNY_HEAD;
		    second->class = NEUTRAL;
		    second->speed = 12;

		  } else {
		    /* Make blood spot */
		    first->img_no = 0;
		    first->frame = 0;
		    first->x_v = 0;
		    first->y_v = 0;
		    first->type = BLOOD_SPOT;
		    /* Make bunny DEAD */
		    second->type = BUNNY_DEAD;
		    second->x = first->x;
		    second->y = first->y;
		    second->speed = 4;
		    /* Make remaining bit of bunny the smashed up body */
		    second->img_no = 1;
		  }
		  break;
		default:
		  break;
		}
		break; /* Bunny */
		/* Bunny head is not clipped */
	      }
	      break; /* NEUTRAL */
	    case PROJECTILE:
	      if( first->owner_id == second->ent_id )
		break; /* Don't clip on Owner */
	      switch( second->class ) {
	      case PROJECTILE:
		switch( second->type ) {
		case CD:
		  second->health -= CD_DMG;
		  first = remove_entity(first->ent_id);
		  break;
		case FIREBALL:
		  second->health -= FIREBALL_DMG;
		  first = remove_entity(first->ent_id); /* Remove proj. */
		  break;
		case NUKE:
		  switch( first->type ) {
		  case NUKE:
		    /* Twice the carnage! */
		    blowup(map, second,(second->x+second->width/2)/TILE_W,
			   (second->y+second->height/2)/TILE_W, NUKE_RADIUS*2);
		    second = remove_entity(second->ent_id);
		    break;
		  case CD:
		    second->health -= CD_DMG;
		    break;
		  case FIREBALL:
		    second->health -= FIREBALL_DMG;
		    break;
		  } /* switch( first->type ) */
		  first = remove_entity(first->ent_id);
		}
	      }
	    }
	   
	    /* Put ent and other back */
	    if( entisfirst ) {
	      ent = first;
	      other = second;
	    } else {
	      ent = second;
	      other = first;
	    }
 
	  } /* Collision occured */
	}
      }
      
    }while( other->next != NULL ); /* Checked entities collision with */
    
  } while( ent->next != NULL && ent->next->next != NULL );

  
}


/* returns pixels between the CENTER of ent0 & ent1 */
int entity_dist(entity *ent0, entity *ent1)
{

  int x_dist, y_dist;

  x_dist = (ent0->x+ent0->width/2) - (ent1->x+ent1->width/2);
  y_dist = (ent0->y+ent0->height/2) - (ent1->y+ent1->height/2);

  return (int)sqrt( x_dist * x_dist + y_dist * y_dist );

}


/* Returns true if entity is inside the map */
int in_range(struct map_t *map, entity *ent)
{

  int x,y;

  /* X & Y positions next frame. ent->[width|height] is there
     to allow you to go off the screen a little (enough to get
     through tight gaps with nearby walls) */
  x = ent->width + ent->x + ent->x_v * ent->speed;
  y = ent->height + ent->y + ent->y_v * ent->speed;

  /* Is x in range? */
  if( x <= 0 || x >= map->width * TILE_W )
    return 0;

  /* is y? */
  if( y <= 0 || y >= map->height * TILE_H )
    return 0;

  return 1; /* Still inside the map */

}


/* Returns what entity will hit in next turn ( if >= t ) */
unsigned char *world_collision(struct map_t *map, int layer, entity *ent, int t)
{

  int x,y, x_v, y_v;
  unsigned char *tile;
  unsigned char *ptr;

  if( layer == BASE )
    ptr = map->base;
  else if( layer == OBJECT )
    ptr = map->object;
  else 
    die("world_collision: layer is not BASE or OBJECT!!\n", 1);

  if( NULL == ent ) {
    fprintf(stderr, "valid move: Entity is NULL!\n");
    return 0;
  }

  x_v = ent->x_v * ent->speed;
  y_v = ent->y_v * ent->speed;

  x = ent->x + x_v; /* desired x position. */
  y = ent->y + y_v; /* desired y position. */

  /* top left. */
  tile = (ptr + (y+ent->height/2)/TILE_H * map->width + x/TILE_W);
  if( *tile >= t )
    return tile;
  /* top right. */
  tile = (ptr + (y+ent->height/2)/TILE_H * map->width + (x+ent->width)/TILE_W);
  if( *tile >= t )
    return tile;
  /* bottom left. */
  tile = (ptr + (y+3*ent->height/2)/TILE_H * map->width + x/TILE_W);
  if( *tile >= t )
    return tile;
  /* bottom right. */
  tile = (ptr + (y+3*ent->height/2)/TILE_H * map->width +
	  (x+ent->width)/TILE_W);
  if( *tile >=t)
    return tile;

  return NULL; /* All clear. */

}


/* Not in world_collisions because it only needs to be called for the player */
void special_objects(struct map_t *map, entity *ent)
{

  char msg_str[MAXMSGLEN];
  unsigned char *tile;
  int x,y, i, mesg_no;

  /* Work out positions next frame */
  x = ent->x + ent->x_v * ent->speed;
  y = ent->y + ent->y_v * ent->speed;

  for( i=0 ; i<4 ; i++ ) {

    switch( i ) {
    case 0: /* Top Left */
      tile = (map->object + (y+ent->height/2)/TILE_H * map->width + x/TILE_W);
      break;
    case 1: /* top right. */
      tile = (map->object + (y+ent->height/2)/TILE_H * map->width +
	      (x+ent->width)/TILE_W);
      break;
    case 2: /* bottom left. */
      tile = (map->object + (y+3*ent->height/2)/TILE_H*map->width + x/TILE_W);
      break;
    case 3: /* bottom right. */
      tile = (map->object + (y+3*ent->height/2)/TILE_H * map->width +
	      (x+ent->width)/TILE_W);
    }

    if( O_MSG1 <= *tile && *tile <= O_MSG9 ) {

#if VERBOSE
      printf("Hit message: %d\n", *tile);
#endif

      mesg_no = *tile - O_MSG0;
      get_message(map->file_name, mesg_no, msg_str);

      draw_text_window(msg_str);
      *tile = O_NULL; /* Remove read message */
      map_dirty = 1;
      
    }

  }

}


/* Do the ai for entity ent */
void move_enemy(entity *enemy, int bravery)
{

  /* Try and get close to the player */
  if( player->x > enemy->x )
    enemy->x_v = 1;
  else if( player->x < enemy->x )
    enemy->x_v = -1;
  else enemy->x_v = 0;
  
  if( player->y > enemy->y )
    enemy->y_v = 1;
  else if( player->y < enemy->y )
    enemy->y_v = -1;
  else enemy->y_v = 0;
  
  enemy->x_v *= bravery;
  enemy->y_v *= bravery;

  if( enemy->type == FLAMER )
    enemy_proj_attack(enemy);

  /* Melee attack??? */

}

void enemy_proj_attack(entity *enemy)
{

  fire_weapon(enemy, WEAP_FIREBALL);

}


void critter_ai(entity *critter)
{

  int modifier;

  /* Run away if player is near! */
  if( entity_dist(critter, player) <= SCARED_DIST ) {
    critter->mode = ALIVE; /* Stop grooming */
    if( critter->type == BUNNY )
      critter->speed = 8;
    else
      critter->speed = 6;
    move_enemy(critter, COWARD);
  } 
  else if( critter->mode == ALIVE && (critter->x_v || critter->y_v) ) {
    if( !(rand() % (2*fps)) ) {
      critter->x_v = critter->y_v = 0; /* Stop him after a bit of a run */
      critter->mode = GROOMING;
      if( critter->type == BUNNY )
	critter->speed = 4;
      else
	critter->speed = 3;
      critter->frame = 0;
    }
  } else if( critter->mode == ALIVE ) {
    
    critter->mode = GROOMING;
    critter->speed = 4;
    critter->x_v = 0;
    critter->y_v = 0;

  }

  if( critter->mode == GROOMING )
    /* Move around a bit */
    if( !(rand() % (3*fps)) ) {
      
      critter->mode = ALIVE;
      modifier = rand()%2? 1 : -1;
      critter->x_v = rand()%2 * modifier;
      critter->y_v = rand()%2 * modifier;

    }

}


void fire_weapon(entity *ent, int type)
{

  int x_v = 0, y_v = 0;
  entity *new_projectile;

  switch( ent->dir ) {
  case 0:
    y_v = -1;
    break;
  case 1:
    y_v = -1;
    x_v = 1;
    break;
  case 2:
    x_v = 1;
    break;
  case 3:
    x_v = 1;
    y_v = 1;
    break;
  case 4:
    y_v = 1;
    break;
  case 5:
    x_v = -1;
    y_v = 1;
    break;
  case 6:
    x_v = -1;
    break;
  case 7:
    x_v = -1;
    y_v = -1;
    break;
  default:
    die("Direction is > 7!\n", 1);
  }

  switch( type ) {
  case WEAP_CD:
    new_projectile = create_entity(CD);
    new_projectile->speed = (ent->x_v || ent->y_v)*ent->speed + 25;
    break;
  case WEAP_FIREBALL:
    new_projectile = create_entity(FIREBALL);
    new_projectile->speed = (ent->x_v || ent->y_v)*ent->speed + 25;
    break;
  case WEAP_NUKE:
    new_projectile = create_entity(NUKE);
    new_projectile->dir = ent->dir;
    new_projectile->speed = 8;
    new_projectile->health = 25;
#if FOCUSFOLLOWSNUKE
    /* Uncomment if you want to keep on the old nuke
       if( focus == player ) */
    focus = new_projectile;
#endif
    break;
  default:
    fprintf(stderr, "Tried to fire unknown weapon type %d!\n", type);
    die("Exiting\n", 1);
  }

  new_projectile->owner_id = ent->ent_id;
  new_projectile->x = ent->x + ent->width/3;
  new_projectile->y = ent->y + ent->width/3;
  new_projectile->x_v = x_v;
  new_projectile->y_v = y_v;

#if VERBOSE
    printf(" speed = %d, direction %d%d\n", new_projectile->speed,
	   new_projectile->x_v, new_projectile->y_v);
#endif

}


void blowup(struct map_t *map, entity *nuke, int gz_x, int gz_y, int blast_rad)
{

  entity *ent;
  int x,y,i, tile_rad;
  int num_killed = 0;

  flash_color(1); /* gives a quick WHITE flash. */
  update_window();

  for( y=gz_y-blast_rad, i=0 ; y <= gz_y+blast_rad ; y++, y-gz_y>0? i-- : i++)
    if( y > 0 && y < map->height-1)
      for( x=gz_x - i ; x<=gz_x + i; x++ )
	if( x > 0 && x < map->width-1) {
	  if( i && x > gz_x - i && x < gz_x + i) {
	    *(map->base + y*map->width + x) = RB_ASH;
	    *(map->object + y*map->width + x) = O_NULL;
	  } /* Only draw blastmarks on floor tiles */
	  else if( *(map->base + y*map->width + x) < REAL_BASE_SPECIAL )
	    *(map->object + y*map->width + x) = RO_BMARK;
	}

  tile_rad = blast_rad * TILE_W; /* Convert from tile squares -> pixels */

  for( ent = root ; ent->next != NULL ; ent = ent->next) {

    if( ent->type == ITEM )
      continue; /* Don't blow away items */

    if( ent->ent_id != nuke->ent_id && ent->mode )
      /* Relies that entity height == width */
      if( abs(nuke->x - ent->x) <= tile_rad + ent->width)
	if( abs(nuke->y - ent->y) <= tile_rad + ent->height)
	  if( entity_dist( nuke, ent ) <= tile_rad ) {
	    /* Give entity apropriate amount of damage, blast damage is
	       directly related to size of the blast radius */
	    ent->health -= (blast_rad/NUKE_RADIUS) * NUKERADIUS_DMG;
	    if( ent->health <= 0 )
	      num_killed++;
	  }
  }
  
#if VERBOSE
  printf("That blast killed %d entities!!\n", num_killed);
#endif

#if USESOUND
  play_sound(S_NUKE);
#endif

  map_dirty = 1;

}


int getdir(int x_v, int y_v)
{

  if( !x_v )

    if( y_v > 0 )
      return 4;
    else return 0;
  
  else if( x_v > 0 )
    
    if( !y_v )
      return 2;
    else if( y_v > 0 )
      return  3;
    else return 1;
  
  else
    
    if( !y_v )
      return 6;
    else if( y_v > 0 )
      return  5;
    else return 7;
  
}


char *dirtocompass(int dir)
{

  switch( dir ) {
  case 0: return "n";
  case 1: return "ne";
  case 2: return "e";
  case 3: return "se";
  case 4: return "s";
  case 5: return "sw";
  case 6: return "w";
  case 7: return "nw";
  case 8: return "die";
  default:
    fprintf(stderr, "Something went wrong in dirtocompass, dir=%d\n", dir);
    die("Bah!\n", 1);
  }

  return (char *)NULL; /* Never reached */

}


void sig_handler(int signo)
{

  char err_buf[32];

  sprintf(err_buf, "Got signal %d", signo);
  die(err_buf, signo);

}
