/*
 * (C) Copyright Keith Visco 1998  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl;

import com.kvisco.util.List;
import com.kvisco.util.QuickStack;
import org.w3c.dom.*;

/**
 * This class represents a PathExpr and LocationPath 
 * pattern
 * @author <a href="kvisco@ziplink.net">Keith Visco</a>
**/
class PathExpr extends List implements Expr, MatchExpr {

    public final int ABSOLUTE = 0;
    public final int RELATIVE = 1;
    
    
    /**
     * Creates a new PathExpr
    **/
    public PathExpr() {
        super();
    } //-- PathExpr

    public PathExpr(FilterBase filterBase) {
        super();
        this.add(filterBase);
    } //-- PathExpr
    
    
      //------------------/    
     //- Public Methods -/
    //------------------/    
    
    /**
     * Returns the type of Expr this Expr represents
     * @return the type of Expr this Expr represents
    **/
    public short getExprType() {
        return Expr.NODE_SET;
    } //-- getExprType
    
    /**
     * Evaluates this Expr using the given context Node and ProcessorState
     * @param context the current context Node
     * @param ps the ProcessorState that contains the current processing 
     * environment
     * @return the ExprResult
    **/
    public ExprResult evaluate(Node context, ProcessorState ps)
        throws InvalidExprException 
    {
        //-- add selectExpr functionality here
        if ((context == null)  || (size() == 0)) return new NodeSet(0);

        NodeSet nodes = new NodeSet();
        
        if ((isAbsolute()) && (context.getNodeType() != Node.DOCUMENT_NODE))
            nodes.add(context.getOwnerDocument());
        else 
            nodes.add(context);
        

        boolean ancestorMode = false;
        
        QuickStack nodeSets = ps.getNodeSetStack();
        for (int i = 0; i < size(); i++) {
            FilterBase filterBase = (FilterBase)get(i);
            int ancestryOp = filterBase.getAncestryOp();
            ancestorMode = (ancestorMode || 
                (ancestryOp == FilterBase.ANCESTOR_OP));
            
            NodeSet tmpNodes = new NodeSet();
            nodeSets.push(nodes);
            for (int j = 0; j < nodes.size(); j++) {
                Node node = nodes.get(j);
                NodeSet xNodes = filterBase.evaluate(node, ps);
                tmpNodes.add(xNodes);
                //-- handle ancestorMode
                if (ancestorMode)
                    tmpNodes.add(fromDescendants(filterBase, node, ps));
            }
            nodeSets.pop();
            nodes = tmpNodes;
        }
         return nodes;
    } //-- evaluate
    
    private static NodeSet fromDescendants
        (FilterBase filterBase, Node context, ProcessorState ps) 
        throws InvalidExprException
    {
        NodeSet nodeSet = new NodeSet();
        
        if (context == null) return nodeSet;
        
        NodeList nl = context.getChildNodes();
        
        for (int i = 0; i < nl.getLength(); i++) {
            Node child = nl.item(i);
            
            //-- filter DOCTYPE and entity elements
            switch (child.getNodeType()) {
                case Node.ENTITY_NODE:
                case Node.DOCUMENT_TYPE_NODE:
                    continue;
                default:
                    break;
            }
            nodeSet.add(filterBase.evaluate(child, ps));
            //-- check childs descendants
            if (child.hasChildNodes()) {
                NodeSet temp = fromDescendants(filterBase, child, ps);
                nodeSet.add(temp);
            }
        }
        return nodeSet;
        
    } //-- fromDecendants
    
    /**
     * Returns the String representation of this PathExpr
     * @return the String representation of this PathExpr
    **/
    public String toString() {
        StringBuffer sb = new StringBuffer();
        
        for (int i = 0; i < size(); i++) {
            FilterBase filterBase = (FilterBase)get(i);
            switch (filterBase.getAncestryOp()) {
                case FilterBase.ANCESTOR_OP:
                    sb.append("//");
                    break;
                case FilterBase.PARENT_OP:
                    sb.append("/");
                    break;
                default:
                    break;                
            }
            sb.append(filterBase.toString());
        }
        
        return sb.toString();
    } //-- toString

    public void add(FilterBase filter) {
        super.add(filter);
    } //-- add
    
    /**
     * Determines the priority of a PatternExpr as follows:
     * <PRE>
     *  - From the 19990421 XSL Working Draft -
     *  + If the Pattern has the form of a QName optionally preceded by
     *    the @ character, then the priority is 0.
     *  + Otherwise if the pattern consists of just a NodeTest then the
     *    priority is -1
     *  + Otherwise the priority is 1
     * </PRE>
     * @return the priority for this PatternExpr
    **/
    public int getDefaultPriority() {
        
        //if (hasIDExpr()) return 1;
        if (size() > 1) return 1;
        else {
            return ((FilterBase)get(0)).getDefaultPriority();
        }
        
    } //-- getDefaultPriority
    
    
    /**
     * Determines if this PathExpr Represents an AbsolutePathExpr or not
    **/
    public boolean isAbsolute() {
        if (size() > 0) {
            return ( ((FilterBase)get(0)).getAncestryOp() != FilterBase.NO_OP);
        }
        return false;
    } //-- isAbsolute
    
    /**
     * Determines if the given node is matched by this MatchExpr with
     * respect to the given context node.
     * @param node the node to determine a match for
     * @param context the Node which represents the current context
     * @param ps the current ProcessorState
     * @return true if the given node is matched by this MatchExpr
    **/
    public boolean matches(Node node, Node context, ProcessorState ps) 
        throws InvalidExprException 
    {
        if (node == null) return false;
        return matches(node,ps,size()-1);
    } //-- matches
    
    /**
     *
    **/
    private boolean matches(Node node, ProcessorState ps, int index)
        throws InvalidExprException
    {
        
        if (node == null) return false;
        
        FilterBase filterBase;
        Node tnode = node;
        Node context = node;
        
        // This needs some re-working ...I know
        if ((index > size()) || (index < 0)) return false;
        
        int ancestryOp = FilterBase.NO_OP;
        filterBase = (FilterBase)get(index);
        
        ancestryOp = filterBase.getAncestryOp();
        if (index == 0) {
            //- check ancestry op for absolute expression (19990318)
            if (ancestryOp == FilterBase.PARENT_OP) {
                if (getParentNode(tnode, ps) != tnode.getOwnerDocument())
                    return false;
            }
            return filterBase.matches(tnode, context, ps);
        }
        else if ( filterBase.matches(tnode, context, ps) ) {
                        
            // compare ancestry operator
            if (ancestryOp == FilterBase.PARENT_OP) {
                return matches(getParentNode(tnode,ps), ps,index-1);
            }
            else if (ancestryOp == FilterBase.ANCESTOR_OP) {
                // match at all depths of ancestry
                while ( (tnode = getParentNode(tnode,ps)) != null ) {
                    if (matches(tnode,ps,index-1))
                        return true;
                }
            }
        }
        return false;
    } //-- matches
    
    private static Node getParentNode(Node node, ProcessorState ps) {
        if (node == null) return null;
        if (node.getNodeType() == Node.ATTRIBUTE_NODE)
            return ps.findParent((Attr)node);
        else 
            return node.getParentNode();
    } //-- getParentNode,
    
} // -- PathExpr