/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */

package com.kvisco.xsl;

import org.w3c.dom.*;

import com.kvisco.util.QuickStack;
import com.kvisco.util.List;

import java.util.Hashtable;


/**
 * This class represents a Location Step as defined by the
 * XSL Working Draft
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
class LocationStep extends FilterBase {

    public static final short ANCESTORS_AXIS           = 0;
    public static final short ANCESTORS_OR_SELF_AXIS   = 1;
    public static final short ATTRIBUTES_AXIS          = 2;
    public static final short CHILDREN_AXIS            = 3;
    public static final short DESCENDANTS_AXIS         = 4;
    public static final short DESCENDANTS_OR_SELF_AXIS = 5;
    public static final short FOLLOWING_AXIS           = 6;
    public static final short FOLLOWING_SIBLINGS_AXIS  = 7;
    public static final short PARENT_AXIS              = 8;
    public static final short PRECEDING_AXIS           = 9;
    public static final short PRECEDING_SIBLINGS_AXIS  = 10;
    public static final short SELF_AXIS                = 11;

    private static final String FROM_PREFIX = "from-";

    private short    axisIdentifier = CHILDREN_AXIS;
    private NodeExpr nodeExpr       = null;

      //----------------/
     //- Constructors -/
    //----------------/

    /**
     * Creates a new LocationStep
    **/
    protected LocationStep() {
        super();
    } //-- LocationStep

    /**
     * Creates a new LocationStep with the given AxisIdentifier
    **/
    protected LocationStep(short axisIdentifier) {
        super();
        this.axisIdentifier = axisIdentifier;
    } //-- LocationStep

      //------------------/
     //- Public Methods -/
    //------------------/

    public NodeSet evaluate(Node context, ProcessorState ps)
        throws InvalidExprException
    {


        if ((context == null) || (nodeExpr == null))
            return new NodeSet(0);

        NodeSet nodeSet = null;
        NodeList nodeList = null;
        //-- process axis
        switch (axisIdentifier) {

            case ANCESTORS_AXIS:
                nodeSet = AxisIdentifier.fromAncestors(nodeExpr, context, ps);
                break;
            case ANCESTORS_OR_SELF_AXIS:
                nodeSet =
                    AxisIdentifier.fromAncestorsOrSelf(nodeExpr, context, ps);
                break;
            case ATTRIBUTES_AXIS:
                nodeSet = AxisIdentifier.fromAttributes(nodeExpr, context, ps);
                break;
            case DESCENDANTS_AXIS:
                nodeSet = AxisIdentifier.fromDescendants(nodeExpr, context, ps);
                break;
            case DESCENDANTS_OR_SELF_AXIS:
                nodeSet =
                    AxisIdentifier.fromDescendantsOrSelf(nodeExpr, context, ps);
                break;
            case FOLLOWING_AXIS:
                nodeSet = AxisIdentifier.fromFollowing(nodeExpr, context, ps);
                break;
            case FOLLOWING_SIBLINGS_AXIS:
                nodeSet =
                    AxisIdentifier.fromFollowingSiblings(nodeExpr, context, ps);
                break;
            case PARENT_AXIS:
                nodeSet = AxisIdentifier.fromParent(nodeExpr, context, ps);
                break;
            case PRECEDING_AXIS:
                nodeSet = AxisIdentifier.fromPreceding(nodeExpr, context, ps);
                break;
            case PRECEDING_SIBLINGS_AXIS:
                nodeSet =
                    AxisIdentifier.fromPrecedingSiblings(nodeExpr, context, ps);
                break;
            case SELF_AXIS:
                nodeSet = AxisIdentifier.fromSelf(nodeExpr, context, ps);
                break;
            default: // children
                nodeSet = AxisIdentifier.fromChildren(nodeExpr, context, ps);
                break;
        } //-- end switch

        //-- filter nodes
        QuickStack nsStack = ps.getNodeSetStack();
        nsStack.push(nodeSet);
        evaluatePredicates(nodeSet, ps);
        nsStack.pop();
        return nodeSet;
    } //-- evaluate

    /**
     * Determines the priority of a PatternExpr as follows:
     * <PRE>
     *  - From the 19990421 XSL Working Draft -
     *  + If the Pattern has the form of a QName optionally preceded by
     *    the @ character, then the priority is 0.
     *  + Otherwise if the pattern consists of just a NodeTest then the
     *    priority is -1
     *  + Otherwise the priority is 1
     * </PRE>
     * @return the priority for this PatternExpr
    **/
    public int getDefaultPriority() {
        if (nodeExpr == null) return -1;


        int priority = -1;
        switch(nodeExpr.getNodeExprType()) {
            case NodeExpr.ELEMENT_EXPR:
            case NodeExpr.ATTRIBUTE_EXPR:
                priority = 0;
                break;
            default:
                break;
        }

        if (getPredicates().size() > 0) priority = 1;
        return priority;
    } //-- getDefaultPriority

    public boolean matches(Node node, Node context, ProcessorState ps)
        throws InvalidExprException
    {

        /*
        NodeSet nodeSet = (NodeSet)evaluate(context, ps);
        return nodeSet.contains(node);
        */

        if (nodeExpr == null) return false;
        if (nodeExpr.matches(node, context, ps)) {
            NodeSet nodeSet = new NodeSet(1);
            nodeSet.add(node);
            evaluatePredicates(nodeSet, ps);
            return (nodeSet.size() == 1);
        }
        return false;
    } //-- matches

    /**
     * Returns the String representation of this LocationStep
     * @return the String representation of this LocationStep
    **/
    public String toString() {


        StringBuffer strbuf = new StringBuffer(FROM_PREFIX);

        switch (axisIdentifier) {
            case ANCESTORS_AXIS:
                strbuf.append(Names.ANCESTORS_AXIS);
                break;
            case ANCESTORS_OR_SELF_AXIS:
                strbuf.append(Names.ANCESTORS_OR_SELF_AXIS);
                break;
            case ATTRIBUTES_AXIS:
                strbuf.append(Names.ATTRIBUTES_AXIS);
                break;
            case DESCENDANTS_AXIS:
                strbuf.append(Names.DESCENDANTS_AXIS);
                break;
            case DESCENDANTS_OR_SELF_AXIS:
                strbuf.append(Names.DESCENDANTS_OR_SELF_AXIS);
                break;
            case FOLLOWING_AXIS:
                strbuf.append(Names.FOLLOWING_AXIS);
                break;
            case FOLLOWING_SIBLINGS_AXIS:
                strbuf.append(Names.FOLLOWING_SIBLINGS_AXIS);
                break;
            case PARENT_AXIS:
                strbuf.append(Names.PARENT_AXIS);
                break;
            case PRECEDING_AXIS:
                strbuf.append(Names.PRECEDING_AXIS);
                break;
            case PRECEDING_SIBLINGS_AXIS:
                strbuf.append(Names.PRECEDING_SIBLINGS_AXIS);
                break;
            case SELF_AXIS:
                strbuf.append(Names.SELF_AXIS);
                break;
            default:
                strbuf.append(Names.CHILDREN_AXIS);
                break;
        } //-- switch
        strbuf.append("(");

        if (nodeExpr == null)
            strbuf.append(super.toString());
        else {
            strbuf.append(nodeExpr.toString());
            strbuf.append(super.toString());
        }
        strbuf.append(")");
        return strbuf.toString();
    } //-- toString

    public short getAxisIdentifier() {
        return this.axisIdentifier;
    } //-- getAxisIdentifier

      //---------------------/
     //- Protected Methods -/
    //---------------------/

    protected NodeExpr getNodeExpr() {
        return this.nodeExpr;
    } //-- getNodeExpr

    public void setAxisIdentifier(short axisIdentifier) {
        this.axisIdentifier = axisIdentifier;
    } //-- setAxisIdentifier

    protected void setNodeExpr(NodeExpr nodeExpr) {
        this.nodeExpr = nodeExpr;
    } //-- setNodeExpr

} //-- LocationStep