/**
 * org/openxml/parser/ParseException.java
 * 
 * The contents of this file are subject to the OpenXML Public
 * License Version 1.0; you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * http://www.openxml.org/license.html
 *
 * THIS SOFTWARE IS DISTRIBUTED ON AN "AS IS" BASIS WITHOUT WARRANTY
 * OF ANY KIND, EITHER EXPRESSED OR IMPLIED. THE INITIAL DEVELOPER
 * AND ALL CONTRIBUTORS SHALL NOT BE LIABLE FOR ANY DAMAGES AS A
 * RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
 * DERIVATIVES. SEE THE LICENSE FOR THE SPECIFIC LANGUAGE GOVERNING
 * RIGHTS AND LIMITATIONS UNDER THE LICENSE.
 * 
 * The Initial Developer of this code under the License is Assaf Arkin.
 * Portions created by Assaf Arkin are Copyright (C) 1998, 1999.
 * All Rights Reserved.
 */


package org.openxml.parser;


import java.io.IOException;
import org.openxml.io.Parser;


/**
 * Parent class for all parsing exceptions. Three derived classes exist based on
 * severity of error:
 * <UL>
 * <LI>(@link FatalParseException} for errors of {@link #ERROR_FATAL} severity,
 * <LI>(@link WellFormedException} for errors of {@link #ERROR_WELL_FORMED}
 *  severity, and
 * <LI>(@link ValidityException} for errors of {@link #ERROR_VALIDITY} severity
 * </UL>
 * <P>
 * When an exception is issued by the parser, it is not always thrown immediately.
 * Often, an exception is preserved and made available when the parser concludes,
 * depending on the severity flag supplied in the parser constructor.
 * <P>
 * Exceptions of fatal severity are always thrown. Stored exceptions can be
 * obtained by calling {@link Parser#getLastException} and recursively calling
 * {@link #getPrevious} on each returned exception.
 * <P>
 * Well formed exceptions are generated when the document is found to be corrupted,
 * for example, a comment that is not terminated, or a closing tag that has no
 * matching opening tag. Such errors are generated by both validating and
 * non-validating parsers. The parser might attempt to correct/bypass well formed
 * errors and store the exceptions for later retrieval. A well formed exception
 * is always thrown if the parser's <TT>stopAtSeverity</TT> level is {@link
 * #STOP_SEVERITY_WELL_FORMED} or {@link #STOP_SEVERITY_VALIDITY}.
 * <P>
 * Validity exceptions are generated when the document structure is found not to
 * comply with the document definition as found in the document DTD. Such errors are
 * only generated by a validating parser. The parser might attempt to correct/bypass
 * validity errors and store the exceptions for later retrieval. A validity exception
 * is always thrown if the parser's <TT>stopAtSeverity</TT> level is {@link
 * #STOP_SEVERITY_VALIDITY}.
 * <P>
 * The location of the error can be identified by calling one of the methods
 * defined in the {@link SourceLocation} interface. This information is generally,
 * but not always, accurate. It indicates the end (not start) of the token or
 * content element at fault.
 * <P>
 * Parser exceptions extend {@link IOException} so they may be returned when
 * documents are read as part of a larger I/O operation (not explicitly parsed).
 * 
 * 
 * @version $Revision: 1.5 $ $Date: 1999/04/18 01:52:13 $
 * @author <a href="mailto:arkin@trendline.co.il">Assaf Arkin</a>
 * @see Parser
 * @see SourceLocation
 * @see FatalParseException
 * @see ValidityException
 * @see WellFormedException
 * @see Parser#getLastException
 * @deprecated Replaced with {@link org.xml.sax.SAXParseException}
 */
public abstract class ParseException
    extends IOException
    implements SourceLocation
{
    
    
    /**
     * Constructs a new exception with the specified message text. The severity
     * level is determined by the derived class. The previous exception is a
     * stored exception that can be obtained with {@link #getPrevious}.
     * 
     * @param source Identifies the source location of the error
     * @param level The severity level
     * @param message The exception message
     */
    ParseException( SourceLocation source, ParseException previous, String message )
    {
        super( message );
        _previous = previous;
        _lineNumber = source.getLineNumber();
        _position = source.getSourcePosition();
        _sourceURI = source.getSourceURI();
    }
    
    
    public Exception getPrevious()
    {
        return _previous;
    }

    
    public final int getLineNumber()
    {
        return _lineNumber;
    }
    
    
    public final int getSourcePosition()
    {
        return _position;
    }
    
    
    public final String getSourceURI()
    {
        return _sourceURI;
    }

    
    /**
     * Returns the severity level of this exception. The severity level is one of
     * {@link #ERROR_FATAL}, {@link #ERROR_WELL_FORMED} or {@link #ERROR_VALIDITY}.
     * 
     * @return The severity level
     */
    public abstract int getLevel();
    
    
    /**
     * Points to the previous exception in a single-linked (backward) list of
     * created, but not thrown, exceptions.
     * 
     * @serial Link to previous execption in list
     */
    private ParseException    _previous;
    
    
    /**
     * The line number at which the error occured.
     * 
     * @serial 1..n
     */
    private int                _lineNumber;
    
    
    /**
     * The character position at which the error occured.
     * 
     * @serial 0..n
     */
    private int                _position;
    
    
    /**
     * The source URI at which the error occured.
     * 
     * @serial URI of source file
     */
    private String            _sourceURI;


    /**
     * Severity level for fatal errors. Once a fatal error has been encountered
     * (e.g. I/O exception), parsing is immediately terminated. Fatal errors are
     * generally I/O exceptions and runtime exceptions.
     */
    public static final short    ERROR_FATAL = Parser.ERROR_FATAL;

    
    /**
     * Severity level for well-formed errors. Well formed errors may be corrected
     * by the parser, but indicate a document that is faulty and cannot be properly
     * processed. Well-formed errors are issued both by validating and non-validating
     * parsers.
     */
    public static final short    ERROR_WELL_FORMED = Parser.ERROR_WELL_FORMED;

    
    /**
     * Severity level for validity constraint errors. Validity constraint errors
     * may be corrected by the parser, but indicate a document that does not
     * conform to the DTD. Validity constraint errors are only issued by a
     * validating parser and are very common in HTML documents.
     */
    public static final short    ERROR_VALIDITY = Parser.ERROR_VALIDITY;

    
    /**
     * Passed to constructor to indicate the parser should only stop when a
     * fatal parsing error has been issued, throwing an {@link FatalParseException}.
     */
    public static final short    STOP_SEVERITY_FATAL =
                                    Parser.STOP_SEVERITY_FATAL;

    
    /**
     * Passed to constructor to indicate the parser should stop when the first
     * well formed parsing error has been issued, throwing a {@link
     * WellFormedException}. This level also implies {@link #STOP_SEVERITY_FATAL}.
     */
    public static final short    STOP_SEVERITY_WELL_FORMED =
                                    Parser.STOP_SEVERITY_WELL_FORMED;

    
    /**
     * Passed to constructor to indicate the parser should stop when the first
     * validity parsing error has been issued, throwing a {@link ValidityException}.
     * This level also implies {@link #STOP_SEVERITY_WELL_FORMED}.
     */
    public static final short    STOP_SEVERITY_VALIDITY =
                                    Parser.STOP_SEVERITY_VALIDITY;


}
