//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        svbsptree.C
//
// Purpose:     Interface to class SVBSPTree
//
// Created:     24 May 1996  Georg Meszaros
//
// Modified:     
//
//
// Description:
//
// 
// 
//
//</file>


//<class>
//
// Name:       SVBSPTree
//
// Purpose:    
//
//
// Public Interface:
//
//
//
// Description:
//
//
//
//
//</class>


#include "svbsptree.h"

#include <hyperg/utils/types.h>

#include "polygon.h"
#include "clipping.h"
#include "bsptree.h"
#include "camera.h"

#include <iostream.h>
#include <math.h>

// all polygons which projection is greater or equal than 
// (margin_ * size_of_the_largest_polygon) are
// allowed to extend the shadow volume 
// 1 ... not a single polygon spans the shadow volume
// 0 ... all polygons participate
// 
float SVBSPTree::margin_ = 0;
int SVBSPTree::use_x_faces_ = 100;

unsigned long SVBSPTree::node_number_ = 0;
unsigned long SVBSPTree::face_number_ = 0;
unsigned long SVBSPTree::visible_face_count_ = 0;
unsigned long SVBSPTree::hidden_face_count_ = 0;
unsigned long SVBSPTree::back_face_count_ = 0;
unsigned long SVBSPTree::visible_split_face_count_ = 0;
   
point3D SVBSPTree::eye_;
BSPTree* SVBSPTree::bsp_root_;

 
SVBSPTree::SVBSPTree()
{
  type_ = VISIBLE;
  node_number_++;
  visible_ = nil;
  shadow_ = nil;
  plane_.normal.x = 0;
  plane_.normal.y = 0;
  plane_.normal.z = 0;
  plane_.d = 0;
}

SVBSPTree::SVBSPTree(const point3D& eye, float margin, int use_x_faces, BSPTree* bsp_root)
{
// this is the constructor for the root - node 
  node_number_ = 0;
  face_number_ = 0;

  visible_face_count_ = 0;
  hidden_face_count_ = 0;
  back_face_count_ = 0;
  visible_split_face_count_ = 0;
  margin_ = margin;
  use_x_faces_ = use_x_faces;

  type_ = VISIBLE;
  node_number_++;
  visible_ = nil;
  shadow_ = nil;

  plane_.normal.x = 0;
  plane_.normal.y = 0;
  plane_.normal.z = 0;
  plane_.d = 0;
  eye_.x = eye.x;
  eye_.y = eye.y;
  eye_.z = eye.z;
  bsp_root_ = bsp_root;
}


SVBSPTree::SVBSPTree(NodeType type)
{
  type_ = type;
  node_number_++;
  if ((type_ == VISIBLE) || (type_ == SHADOW)) //  a leave
  {
    visible_ = nil;
    shadow_ = nil;
  }
  else  // a node containing a shadow bounding polygon
  {
    visible_ = new SVBSPTree(VISIBLE);
    shadow_ = nil;
  }
  plane_.normal.x = 0;
  plane_.normal.y = 0;
  plane_.normal.z = 0;
  plane_.d = 0;
}


SVBSPTree::~SVBSPTree()
{
  if (visible_) delete visible_;
  if (shadow_) delete shadow_;

  node_number_--;
  //cerr << "SVBSP Node: " << node_number_ << " deleted\n";
}



void SVBSPTree::filter(Face* face, Face* original_face)
{

  if (type_ == VISIBLE)
  {
     replace(face, original_face);
     return;
  }

  if (type_ == SHADOW)
  {     
     cerr << type_ << " filter(...) should not reach this point\n";
     return;
  }

// only draw the front part of the whole scene
/*
  if ((visible_face_count_ + 
       hidden_face_count_ + 
       back_face_count_)  >  0.5 * bsp_root_->faceNumber() ) { return; }
*/

  // type_ == PLANE:

  // if the whole face lies in front of the polygon which spans this shadow plane 

  int remember_vertex_num = original_face->faceAttr()->vertexNum();
  int remember_normal_num = original_face->faceAttr()->normalNum();

  // clip generates new faces if the face was split, otherwise the
  // pointer to the original face is returned
  // if the face lies on the same plane both front_face and back_face are nil

  Face* visible_face;
  Face* shadowed_face; 
  SplitType split_type;

  Clipping::clip(*face, plane_.normal, plane_.d, visible_face, shadowed_face, split_type);

  //attention: if the face has not been split, NO new face has been created
  //           only delete original face if a new one has been created


  if ((split_type == ONLY_FRONT) || (split_type == SPLIT))  
     visible_->filter(visible_face, original_face);   

  if ( ((split_type == ONLY_BACK) || (split_type == SPLIT)) && (shadow_->type_ == PLANE) )
     shadow_->filter(shadowed_face, original_face); 


  // delete the created split polygons
  if (split_type == SPLIT)
  {
    delete visible_face;
    delete shadowed_face;
  }

  original_face->faceAttr()->setVertexListSize(remember_vertex_num);
  original_face->faceAttr()->setNormalListSize(remember_normal_num);  
}




void SVBSPTree::replace(Face* face, Face* original_face) 
{ 
//cout << "*: replace with Face:" << face.id() << "\n";
// replaces this with  a svbsp tree, build for this single face
// and marks the corresponding original face in the BSP Tree as partly visible

  if (type_ != VISIBLE) return;  // type of the svbsp-tree
  
  if (original_face->status() != PARTLY_VISIBLE)
  {
    original_face->setStatus(PARTLY_VISIBLE);
    visible_face_count_++;
  }

  visible_split_face_count_++;


  float area_projection = 0;
  
  // calculate the projected area
  if (margin_)
  {  
   vector3D ray;
   float norm;
   point3D any_vertex = face->anyVertex();
   sub3D(eye_, any_vertex, ray); 
   norm = norm3D(ray);
   if (norm) scl3D(ray, 1/norm);
   area_projection = dot3D(face->normal(), ray) * original_face->area();
  }

  // TODO:
  // this query could be easily improved by holding the maxarea in the SVBSP-Tree
  /*
  if ((face_number_ < use_x_faces_ ) 
      &&
      ( (!margin_) 
        || 
        (area_projection > margin_ * original_face->maxArea())
      )
     ) 
  */
  // extend the shadow volume bsp tree
  {
    face_number_++;
    type_ = PLANE;
    visible_ = new SVBSPTree(VISIBLE);
 
    point3D* vertices = face->vertices();
    int vertex_num = face->vertexNumber();  

    SVBSPTree* shadow_volume = this; 
  
    for (int i=0; i < vertex_num - 1; i++)
    {
       shadow_volume->calcPlaneEqu(eye_, vertices[i], vertices[i+1]);     
       shadow_volume->shadow_ = new SVBSPTree(PLANE);
       shadow_volume = shadow_volume->shadow_;
    }
    shadow_volume->calcPlaneEqu(eye_, vertices[vertex_num - 1], vertices[0]);
    shadow_volume->shadow_ = new SVBSPTree(SHADOW);  

    delete [] vertices;
  }
}



void SVBSPTree::insertViewVolume(Camera* camera)
{
// insert the planes building the view volume into the svbsp tree --> clipping
// near clipping plane, far clipping plane, 
// and the top, bottom, left, abd right planes

  //camera->print();

  if (type_ != VISIBLE) return;  // type of the svbsp-tree
  
  Plane view_volume[6];
  camera->getViewVolume(view_volume);
 
	
  SVBSPTree* clipping_planes = this; 
  for (int i=0; i < 6; i++)
    {
       clipping_planes->type_ = PLANE;
       clipping_planes->setPlaneEqu(view_volume[i]);     
       clipping_planes->shadow_ = new SVBSPTree(SHADOW);
       clipping_planes->visible_ = new SVBSPTree(VISIBLE);
       clipping_planes = clipping_planes->visible_;
    }


/*
for (int i=0; i < 6; i++)
{
  Clipping::print3D(view_volume[i].normal);
  cerr << " = " << view_volume[i].d << "\n";
}
*/
  

}




void SVBSPTree::calcPlaneEqu(const point3D& vertex1, 
                             const point3D& vertex2, 
                             const point3D& vertex3)
{
   // calculates and sets the plane equation for this 

   calcPlaneEqu(vertex1, vertex2, vertex3, plane_);
}



void SVBSPTree::calcPlaneEqu(const point3D& vertex1, 
                             const point3D& vertex2, 
                             const point3D& vertex3,
                             Plane& plane)
{
    vector3D vector1, vector2;
    float norm;

    sub3D(vertex2, vertex1, vector1);
    sub3D(vertex3, vertex1, vector2);

    if ((vector1.x != vector2.x) || (vector1.y != vector2.y) || (vector1.z != vector2.z))
    {
      crp3D(vector1, vector2, plane.normal);
      norm = norm3D(plane.normal);
      if (norm!=0) scl3D(plane.normal, 1/norm);
    }
    plane.d = dot3D(plane.normal, vertex1);
}
